/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.kayenta.s3.storage;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.HeadBucketRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.util.StringUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.netflix.kayenta.aws.security.AwsNamedAccountCredentials;
import com.netflix.kayenta.canary.CanaryConfig;
import com.netflix.kayenta.index.CanaryConfigIndex;
import com.netflix.kayenta.index.config.CanaryConfigIndexAction;
import com.netflix.kayenta.security.AccountCredentials;
import com.netflix.kayenta.security.AccountCredentialsRepository;
import com.netflix.kayenta.storage.ObjectType;
import com.netflix.kayenta.storage.StorageService;
import com.netflix.kayenta.util.Retry;
import com.netflix.spinnaker.kork.web.exceptions.NotFoundException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import javax.validation.constraints.NotNull;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class S3StorageService
implements StorageService {
    private static final Logger log = LoggerFactory.getLogger(S3StorageService.class);
    public final int MAX_RETRIES = 10;
    public final long RETRY_BACKOFF = 1000L;
    @NotNull
    private ObjectMapper objectMapper;
    @NotNull
    private List<String> accountNames;
    @Autowired
    AccountCredentialsRepository accountCredentialsRepository;
    @Autowired
    CanaryConfigIndex canaryConfigIndex;
    private final Retry retry = new Retry();

    public boolean servicesAccount(String accountName) {
        return this.accountNames.contains(accountName);
    }

    public void ensureBucketExists(String accountName) {
        AwsNamedAccountCredentials credentials = (AwsNamedAccountCredentials)this.accountCredentialsRepository.getRequiredOne(accountName);
        AmazonS3 amazonS3 = credentials.getAmazonS3();
        String bucket = credentials.getBucket();
        String region = credentials.getRegion();
        HeadBucketRequest request = new HeadBucketRequest(bucket);
        try {
            amazonS3.headBucket(request);
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                if (StringUtils.isNullOrEmpty((String)region)) {
                    log.warn("Bucket {} does not exist. Creating it in default region.", (Object)bucket);
                    amazonS3.createBucket(bucket);
                } else {
                    log.warn("Bucket {} does not exist. Creating it in region {}.", (Object)bucket, (Object)region);
                    amazonS3.createBucket(bucket, region);
                }
            }
            log.error("Could not create bucket {}: {}", (Object)bucket, (Object)e);
            throw e;
        }
    }

    public <T> T loadObject(String accountName, ObjectType objectType, String objectKey) throws IllegalArgumentException, NotFoundException {
        String path;
        AwsNamedAccountCredentials credentials = (AwsNamedAccountCredentials)this.accountCredentialsRepository.getRequiredOne(accountName);
        AmazonS3 amazonS3 = credentials.getAmazonS3();
        String bucket = credentials.getBucket();
        try {
            path = this.resolveSingularPath(objectType, objectKey, credentials, amazonS3, bucket);
        }
        catch (IllegalArgumentException e) {
            throw new NotFoundException(e.getMessage());
        }
        try {
            S3Object s3Object = amazonS3.getObject(bucket, path);
            return this.deserialize(s3Object, objectType.getTypeReference());
        }
        catch (AmazonS3Exception e) {
            log.error("Failed to load {} {}: {}", new Object[]{objectType.getGroup(), objectKey, e.getStatusCode()});
            if (e.getStatusCode() == 404) {
                throw new NotFoundException("No file at path " + path + ".");
            }
            throw e;
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to deserialize object (key: " + objectKey + ")", e);
        }
    }

    private String resolveSingularPath(ObjectType objectType, String objectKey, AwsNamedAccountCredentials credentials, AmazonS3 amazonS3, String bucket) {
        String rootFolder = this.daoRoot(credentials, objectType.getGroup()) + "/" + objectKey;
        ObjectListing bucketListing = amazonS3.listObjects(new ListObjectsRequest(bucket, rootFolder, null, null, Integer.valueOf(10000)));
        List summaries = bucketListing.getObjectSummaries();
        if (summaries != null && summaries.size() == 1) {
            return ((S3ObjectSummary)summaries.get(0)).getKey();
        }
        throw new IllegalArgumentException("Unable to resolve singular " + objectType + " at " + this.daoRoot(credentials, objectType.getGroup()) + "/" + objectKey + ".");
    }

    private <T> T deserialize(S3Object s3Object, TypeReference<T> typeReference) throws IOException {
        return (T)this.objectMapper.readValue((InputStream)s3Object.getObjectContent(), typeReference);
    }

    public <T> void storeObject(String accountName, ObjectType objectType, String objectKey, T obj, String filename, boolean isAnUpdate) {
        String originalPath;
        AwsNamedAccountCredentials credentials = (AwsNamedAccountCredentials)this.accountCredentialsRepository.getRequiredOne(accountName);
        AmazonS3 amazonS3 = credentials.getAmazonS3();
        String bucket = credentials.getBucket();
        String group = objectType.getGroup();
        String path = this.buildS3Key(credentials, objectType, group, objectKey, filename);
        this.ensureBucketExists(accountName);
        long updatedTimestamp = -1L;
        String correlationId = null;
        String canaryConfigSummaryJson = null;
        if (objectType == ObjectType.CANARY_CONFIG) {
            updatedTimestamp = this.canaryConfigIndex.getRedisTime();
            CanaryConfig canaryConfig = (CanaryConfig)obj;
            this.checkForDuplicateCanaryConfig(canaryConfig, objectKey, credentials);
            originalPath = isAnUpdate ? this.resolveSingularPath(objectType, objectKey, credentials, amazonS3, bucket) : null;
            correlationId = UUID.randomUUID().toString();
            ImmutableMap canaryConfigSummary = new ImmutableMap.Builder().put((Object)"id", (Object)objectKey).put((Object)"name", (Object)canaryConfig.getName()).put((Object)"updatedTimestamp", (Object)updatedTimestamp).put((Object)"updatedTimestampIso", (Object)Instant.ofEpochMilli(updatedTimestamp).toString()).put((Object)"applications", (Object)canaryConfig.getApplications()).build();
            try {
                canaryConfigSummaryJson = this.objectMapper.writeValueAsString((Object)canaryConfigSummary);
            }
            catch (JsonProcessingException e) {
                throw new IllegalArgumentException("Problem serializing canaryConfigSummary -> " + (Map)canaryConfigSummary, e);
            }
            this.canaryConfigIndex.startPendingUpdate((AccountCredentials)credentials, "" + updatedTimestamp, CanaryConfigIndexAction.UPDATE, correlationId, canaryConfigSummaryJson);
        } else {
            originalPath = null;
        }
        try {
            byte[] bytes = this.objectMapper.writeValueAsBytes(obj);
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentLength((long)bytes.length);
            objectMetadata.setContentMD5(new String(Base64.encodeBase64((byte[])DigestUtils.md5((byte[])bytes))));
            this.retry.retry(() -> amazonS3.putObject(bucket, path, (InputStream)new ByteArrayInputStream(bytes), objectMetadata), 10, 1000L);
            if (objectType == ObjectType.CANARY_CONFIG) {
                if (originalPath != null && !originalPath.equals(path)) {
                    this.retry.retry(() -> amazonS3.deleteObject(bucket, originalPath), 10, 1000L);
                }
                this.canaryConfigIndex.finishPendingUpdate((AccountCredentials)credentials, CanaryConfigIndexAction.UPDATE, correlationId);
            }
        }
        catch (Exception e) {
            log.error("Update failed on path {}: {}", (Object)this.buildTypedFolder(credentials, group), (Object)e);
            if (objectType == ObjectType.CANARY_CONFIG) {
                this.canaryConfigIndex.removeFailedPendingUpdate((AccountCredentials)credentials, "" + updatedTimestamp, CanaryConfigIndexAction.UPDATE, correlationId, canaryConfigSummaryJson);
            }
            throw new IllegalArgumentException(e);
        }
    }

    private void checkForDuplicateCanaryConfig(CanaryConfig canaryConfig, String canaryConfigId, AwsNamedAccountCredentials credentials) {
        List applications;
        String canaryConfigName = canaryConfig.getName();
        String existingCanaryConfigId = this.canaryConfigIndex.getIdFromName((AccountCredentials)credentials, canaryConfigName, applications = canaryConfig.getApplications());
        if (!org.springframework.util.StringUtils.isEmpty((Object)existingCanaryConfigId) && !existingCanaryConfigId.equals(canaryConfigId)) {
            throw new IllegalArgumentException("Canary config with name '" + canaryConfigName + "' already exists in the scope of applications " + applications + ".");
        }
    }

    public void deleteObject(String accountName, ObjectType objectType, String objectKey) {
        AwsNamedAccountCredentials credentials = (AwsNamedAccountCredentials)this.accountCredentialsRepository.getRequiredOne(accountName);
        AmazonS3 amazonS3 = credentials.getAmazonS3();
        String bucket = credentials.getBucket();
        String path = this.resolveSingularPath(objectType, objectKey, credentials, amazonS3, bucket);
        long updatedTimestamp = -1L;
        String correlationId = null;
        String canaryConfigSummaryJson = null;
        if (objectType == ObjectType.CANARY_CONFIG) {
            updatedTimestamp = this.canaryConfigIndex.getRedisTime();
            Map existingCanaryConfigSummary = this.canaryConfigIndex.getSummaryFromId((AccountCredentials)credentials, objectKey);
            if (existingCanaryConfigSummary != null) {
                String canaryConfigName = (String)existingCanaryConfigSummary.get("name");
                List applications = (List)existingCanaryConfigSummary.get("applications");
                correlationId = UUID.randomUUID().toString();
                ImmutableMap canaryConfigSummary = new ImmutableMap.Builder().put((Object)"id", (Object)objectKey).put((Object)"name", (Object)canaryConfigName).put((Object)"updatedTimestamp", (Object)updatedTimestamp).put((Object)"updatedTimestampIso", (Object)Instant.ofEpochMilli(updatedTimestamp).toString()).put((Object)"applications", (Object)applications).build();
                try {
                    canaryConfigSummaryJson = this.objectMapper.writeValueAsString((Object)canaryConfigSummary);
                }
                catch (JsonProcessingException e) {
                    throw new IllegalArgumentException("Problem serializing canaryConfigSummary -> " + (Map)canaryConfigSummary, e);
                }
                this.canaryConfigIndex.startPendingUpdate((AccountCredentials)credentials, "" + updatedTimestamp, CanaryConfigIndexAction.DELETE, correlationId, canaryConfigSummaryJson);
            }
        }
        try {
            this.retry.retry(() -> amazonS3.deleteObject(bucket, path), 10, 1000L);
            if (correlationId != null) {
                this.canaryConfigIndex.finishPendingUpdate((AccountCredentials)credentials, CanaryConfigIndexAction.DELETE, correlationId);
            }
        }
        catch (Exception e) {
            log.error("Failed to delete path {}: {}", (Object)path, (Object)e);
            if (correlationId != null) {
                this.canaryConfigIndex.removeFailedPendingUpdate((AccountCredentials)credentials, "" + updatedTimestamp, CanaryConfigIndexAction.DELETE, correlationId, canaryConfigSummaryJson);
            }
            throw new IllegalArgumentException(e);
        }
    }

    public List<Map<String, Object>> listObjectKeys(String accountName, ObjectType objectType, List<String> applications, boolean skipIndex) {
        AwsNamedAccountCredentials credentials = (AwsNamedAccountCredentials)this.accountCredentialsRepository.getRequiredOne(accountName);
        if (!skipIndex && objectType == ObjectType.CANARY_CONFIG) {
            Set canaryConfigSet = this.canaryConfigIndex.getCanaryConfigSummarySet((AccountCredentials)credentials, applications);
            return Lists.newArrayList((Iterable)canaryConfigSet);
        }
        AmazonS3 amazonS3 = credentials.getAmazonS3();
        String bucket = credentials.getBucket();
        String group = objectType.getGroup();
        String prefix = this.buildTypedFolder(credentials, group);
        this.ensureBucketExists(accountName);
        int skipToOffset = prefix.length() + 1;
        ArrayList<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        log.debug("Listing {}", (Object)group);
        ObjectListing bucketListing = amazonS3.listObjects(new ListObjectsRequest(bucket, prefix, null, null, Integer.valueOf(10000)));
        List summaries = bucketListing.getObjectSummaries();
        while (bucketListing.isTruncated()) {
            bucketListing = amazonS3.listNextBatchOfObjects(bucketListing);
            summaries.addAll(bucketListing.getObjectSummaries());
        }
        if (summaries != null) {
            for (S3ObjectSummary summary : summaries) {
                String itemName = summary.getKey();
                int indexOfLastSlash = itemName.lastIndexOf("/");
                HashMap<String, Object> objectMetadataMap = new HashMap<String, Object>();
                long updatedTimestamp = summary.getLastModified().getTime();
                objectMetadataMap.put("id", itemName.substring(skipToOffset, indexOfLastSlash));
                objectMetadataMap.put("updatedTimestamp", updatedTimestamp);
                objectMetadataMap.put("updatedTimestampIso", Instant.ofEpochMilli(updatedTimestamp).toString());
                if (objectType == ObjectType.CANARY_CONFIG) {
                    String name = itemName.substring(indexOfLastSlash + 1);
                    if (name.endsWith(".json")) {
                        name = name.substring(0, name.length() - 5);
                    }
                    objectMetadataMap.put("name", name);
                }
                result.add(objectMetadataMap);
            }
        }
        return result;
    }

    private String daoRoot(AwsNamedAccountCredentials credentials, String daoTypeName) {
        return credentials.getRootFolder() + "/" + daoTypeName;
    }

    private String buildS3Key(AwsNamedAccountCredentials credentials, ObjectType objectType, String group, String objectKey, String metadataFilename) {
        if (metadataFilename == null) {
            metadataFilename = objectType.getDefaultFilename();
        }
        if (objectKey.endsWith(metadataFilename)) {
            return objectKey;
        }
        return (this.buildTypedFolder(credentials, group) + "/" + objectKey + "/" + metadataFilename).replace("//", "/");
    }

    private String buildTypedFolder(AwsNamedAccountCredentials credentials, String type) {
        return this.daoRoot(credentials, type).replaceAll("//", "/");
    }

    S3StorageService(ObjectMapper objectMapper, List<String> accountNames, AccountCredentialsRepository accountCredentialsRepository, CanaryConfigIndex canaryConfigIndex) {
        this.objectMapper = objectMapper;
        this.accountNames = accountNames;
        this.accountCredentialsRepository = accountCredentialsRepository;
        this.canaryConfigIndex = canaryConfigIndex;
    }

    public static S3StorageServiceBuilder builder() {
        return new S3StorageServiceBuilder();
    }

    public List<String> getAccountNames() {
        return this.accountNames;
    }

    public static class S3StorageServiceBuilder {
        private ObjectMapper objectMapper;
        private ArrayList<String> accountNames;
        private AccountCredentialsRepository accountCredentialsRepository;
        private CanaryConfigIndex canaryConfigIndex;

        S3StorageServiceBuilder() {
        }

        public S3StorageServiceBuilder objectMapper(ObjectMapper objectMapper) {
            this.objectMapper = objectMapper;
            return this;
        }

        public S3StorageServiceBuilder accountName(String accountName) {
            if (this.accountNames == null) {
                this.accountNames = new ArrayList();
            }
            this.accountNames.add(accountName);
            return this;
        }

        public S3StorageServiceBuilder accountNames(Collection<? extends String> accountNames) {
            if (accountNames == null) {
                throw new IllegalArgumentException("accountNames cannot be null");
            }
            if (this.accountNames == null) {
                this.accountNames = new ArrayList();
            }
            this.accountNames.addAll(accountNames);
            return this;
        }

        public S3StorageServiceBuilder clearAccountNames() {
            if (this.accountNames != null) {
                this.accountNames.clear();
            }
            return this;
        }

        public S3StorageServiceBuilder accountCredentialsRepository(AccountCredentialsRepository accountCredentialsRepository) {
            this.accountCredentialsRepository = accountCredentialsRepository;
            return this;
        }

        public S3StorageServiceBuilder canaryConfigIndex(CanaryConfigIndex canaryConfigIndex) {
            this.canaryConfigIndex = canaryConfigIndex;
            return this;
        }

        public S3StorageService build() {
            List<String> accountNames;
            switch (this.accountNames == null ? 0 : this.accountNames.size()) {
                case 0: {
                    accountNames = Collections.emptyList();
                    break;
                }
                case 1: {
                    accountNames = Collections.singletonList(this.accountNames.get(0));
                    break;
                }
                default: {
                    accountNames = Collections.unmodifiableList(new ArrayList<String>(this.accountNames));
                }
            }
            return new S3StorageService(this.objectMapper, accountNames, this.accountCredentialsRepository, this.canaryConfigIndex);
        }

        public String toString() {
            return "S3StorageService.S3StorageServiceBuilder(objectMapper=" + this.objectMapper + ", accountNames=" + this.accountNames + ", accountCredentialsRepository=" + this.accountCredentialsRepository + ", canaryConfigIndex=" + this.canaryConfigIndex + ")";
        }
    }
}

