/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.kork.artifacts.artifactstore.s3;

import com.netflix.spinnaker.kork.artifacts.ArtifactTypes;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactDecorator;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactReferenceURI;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactStore;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactStoreURIBuilder;
import com.netflix.spinnaker.kork.artifacts.model.Artifact;
import com.netflix.spinnaker.kork.exceptions.SpinnakerException;
import com.netflix.spinnaker.security.AuthenticatedRequest;
import java.io.Serializable;
import java.util.Base64;
import java.util.NoSuchElementException;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectTaggingRequest;
import software.amazon.awssdk.services.s3.model.GetObjectTaggingResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Exception;
import software.amazon.awssdk.services.s3.model.Tag;
import software.amazon.awssdk.services.s3.model.Tagging;

public class S3ArtifactStore
extends ArtifactStore {
    private static final Logger log = LogManager.getLogger(S3ArtifactStore.class);
    private final S3Client s3Client;
    private final PermissionEvaluator permissionEvaluator;
    private final String bucket;
    private final ArtifactStoreURIBuilder uriBuilder;
    private final String applicationsRegex;
    private static final String ENFORCE_PERMS_KEY = "application";

    public S3ArtifactStore(S3Client s3Client, PermissionEvaluator permissionEvaluator, String bucket, ArtifactStoreURIBuilder uriBuilder, String applicationsRegex) {
        this.s3Client = s3Client;
        this.bucket = bucket;
        this.permissionEvaluator = permissionEvaluator;
        this.uriBuilder = uriBuilder;
        this.applicationsRegex = applicationsRegex;
    }

    @Override
    public Artifact store(Artifact artifact) {
        String application = AuthenticatedRequest.getSpinnakerApplication().orElse(null);
        if (application == null) {
            log.warn("failed to retrieve application from request artifact={}", (Object)artifact.getName());
            return artifact;
        }
        if (this.applicationsRegex != null && !Pattern.matches(this.applicationsRegex, application)) {
            return artifact;
        }
        ArtifactReferenceURI ref = this.uriBuilder.buildArtifactURI(application, artifact);
        Artifact remoteArtifact = artifact.toBuilder().type(ArtifactTypes.REMOTE_BASE64.getMimeType()).reference(ref.uri()).build();
        if (this.objectExists(ref)) {
            return remoteArtifact;
        }
        Tag accountTag = (Tag)Tag.builder().key(ENFORCE_PERMS_KEY).value(application).build();
        PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucket).key(ref.paths()).tagging((Tagging)Tagging.builder().tagSet(new Tag[]{accountTag}).build()).build();
        this.s3Client.putObject(request, RequestBody.fromBytes((byte[])this.getReferenceAsBytes(artifact)));
        return remoteArtifact;
    }

    private byte[] getReferenceAsBytes(Artifact artifact) {
        String reference = artifact.getReference();
        if (reference == null) {
            throw new IllegalArgumentException("reference cannot be null");
        }
        String type = artifact.getType();
        if (type != null && type.endsWith("/base64")) {
            return Base64.getDecoder().decode(reference);
        }
        return reference.getBytes();
    }

    @Override
    public Artifact get(ArtifactReferenceURI uri, ArtifactDecorator ... decorators) {
        this.hasAuthorization(uri, (String)AuthenticatedRequest.getSpinnakerUser().orElseThrow(() -> new NoSuchElementException("Could not authenticate due to missing user id")));
        GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().bucket(this.bucket).key(uri.paths()).build();
        ResponseBytes resp = this.s3Client.getObjectAsBytes(request);
        Artifact.ArtifactBuilder builder = Artifact.builder().type(ArtifactTypes.REMOTE_BASE64.getMimeType()).reference(Base64.getEncoder().encodeToString(resp.asByteArray()));
        if (decorators == null) {
            return builder.build();
        }
        for (ArtifactDecorator decorator : decorators) {
            builder = decorator.decorate(builder);
        }
        return builder.build();
    }

    private void hasAuthorization(ArtifactReferenceURI uri, String userId) {
        GetObjectTaggingRequest request = (GetObjectTaggingRequest)GetObjectTaggingRequest.builder().bucket(this.bucket).key(uri.paths()).build();
        GetObjectTaggingResponse resp = this.s3Client.getObjectTagging(request);
        Tag tag = resp.tagSet().stream().filter(t -> t.key().equals(ENFORCE_PERMS_KEY)).findFirst().orElse(null);
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (tag == null || this.permissionEvaluator != null && !this.permissionEvaluator.hasPermission(auth, (Serializable)((Object)tag.value()), ENFORCE_PERMS_KEY, (Object)"READ")) {
            log.error("Could not authenticate to retrieve artifact user={} applicationOfStoredArtifact={}", (Object)userId, (Object)(tag == null ? "(none)" : tag.value()));
            throw new AuthenticationServiceException(userId + " does not have permission to access this artifact");
        }
    }

    private boolean objectExists(ArtifactReferenceURI uri) {
        HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().bucket(this.bucket).key(uri.paths()).build();
        try {
            this.s3Client.headObject(request);
            log.debug("Artifact exists. No need to store. reference={}", (Object)uri.uri());
            return true;
        }
        catch (NoSuchKeyException e) {
            log.info("Artifact does not exist reference={}", (Object)uri.uri());
            return false;
        }
        catch (S3Exception e) {
            int statusCode = e.statusCode();
            log.error("Artifact store failed head object request statusCode={} reference={}", (Object)statusCode, (Object)uri.uri());
            if (statusCode != 0) {
                throw new SpinnakerException(S3ArtifactStore.buildHeadObjectExceptionMessage(e), (Throwable)e);
            }
            throw new SpinnakerException("S3 head object failed", (Throwable)e);
        }
    }

    private static String buildHeadObjectExceptionMessage(S3Exception e) {
        switch (e.statusCode()) {
            case 403: {
                return "Failed to query artifact due to IAM permissions either on the bucket or object";
            }
            case 400: {
                return "Failed to query artifact due to invalid request";
            }
        }
        return String.format("Failed to query artifact: %d", e.statusCode());
    }
}

