/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.kork.artifacts.artifactstore.s3;

import com.netflix.spinnaker.kork.artifacts.ArtifactTypes;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactReferenceURI;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactStoreStorer;
import com.netflix.spinnaker.kork.artifacts.artifactstore.ArtifactStoreURIBuilder;
import com.netflix.spinnaker.kork.artifacts.model.Artifact;
import com.netflix.spinnaker.kork.exceptions.SpinnakerException;
import com.netflix.spinnaker.security.AuthenticatedRequest;
import java.util.Base64;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Exception;
import software.amazon.awssdk.services.s3.model.Tag;
import software.amazon.awssdk.services.s3.model.Tagging;

public class S3ArtifactStoreStorer
implements ArtifactStoreStorer {
    private static final Logger log = LogManager.getLogger(S3ArtifactStoreStorer.class);
    private final S3Client s3Client;
    private final String bucket;
    private final ArtifactStoreURIBuilder uriBuilder;
    private final Pattern applicationsPattern;

    public S3ArtifactStoreStorer(S3Client s3Client, String bucket, ArtifactStoreURIBuilder uriBuilder, String applicationsRegex) {
        this.s3Client = s3Client;
        this.bucket = bucket;
        this.uriBuilder = uriBuilder;
        this.applicationsPattern = applicationsRegex != null ? Pattern.compile(applicationsRegex, 2) : null;
    }

    @Override
    public Artifact store(Artifact artifact) {
        byte[] referenceBytes;
        String application = AuthenticatedRequest.getSpinnakerApplication().orElse(null);
        if (application == null) {
            log.warn("failed to retrieve application from request artifact={}", (Object)artifact.getName());
            return artifact;
        }
        if (this.applicationsPattern != null && !this.applicationsPattern.matcher(application).matches()) {
            return artifact;
        }
        try {
            referenceBytes = this.getReferenceAsBytes(artifact);
        }
        catch (IllegalArgumentException e) {
            log.warn("Artifact cannot be stored due to reference not being base64 encoded");
            return artifact;
        }
        ArtifactReferenceURI ref = this.uriBuilder.buildArtifactURI(application, artifact);
        Artifact remoteArtifact = artifact.toBuilder().type(ArtifactTypes.REMOTE_BASE64.getMimeType()).reference(ref.uri()).build();
        if (this.objectExists(ref)) {
            return remoteArtifact;
        }
        Tag accountTag = (Tag)Tag.builder().key("application").value(application).build();
        PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucket).key(ref.paths()).tagging((Tagging)Tagging.builder().tagSet(new Tag[]{accountTag}).build()).build();
        this.s3Client.putObject(request, RequestBody.fromBytes((byte[])referenceBytes));
        return remoteArtifact;
    }

    private byte[] getReferenceAsBytes(Artifact artifact) {
        String reference = artifact.getReference();
        if (reference == null) {
            throw new IllegalArgumentException("reference cannot be null");
        }
        String type = artifact.getType();
        if (type != null && type.endsWith("/base64")) {
            return Base64.getDecoder().decode(reference);
        }
        return reference.getBytes();
    }

    private boolean objectExists(ArtifactReferenceURI uri) {
        HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().bucket(this.bucket).key(uri.paths()).build();
        try {
            this.s3Client.headObject(request);
            log.debug("Artifact exists. No need to store. reference={}", (Object)uri.uri());
            return true;
        }
        catch (NoSuchKeyException e) {
            log.info("Artifact does not exist reference={}", (Object)uri.uri());
            return false;
        }
        catch (S3Exception e) {
            int statusCode = e.statusCode();
            log.error("Artifact store failed head object request statusCode={} reference={}", (Object)statusCode, (Object)uri.uri());
            if (statusCode != 0) {
                throw new SpinnakerException(S3ArtifactStoreStorer.buildHeadObjectExceptionMessage(e), (Throwable)e);
            }
            throw new SpinnakerException("S3 head object failed", (Throwable)e);
        }
    }

    private static String buildHeadObjectExceptionMessage(S3Exception e) {
        switch (e.statusCode()) {
            case 403: {
                return "Failed to query artifact due to IAM permissions either on the bucket or object";
            }
            case 400: {
                return "Failed to query artifact due to invalid request";
            }
        }
        return String.format("Failed to query artifact: %d", e.statusCode());
    }
}

