/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.kork.jedis.lock;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.patterns.LongTaskTimer;
import com.netflix.spinnaker.kork.jedis.RedisClientDelegate;
import com.netflix.spinnaker.kork.lock.LockManager;
import com.netflix.spinnaker.kork.lock.RefreshableLockManager;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedisLockManager
implements RefreshableLockManager {
    private static final Logger log = LoggerFactory.getLogger(RedisLockManager.class);
    private static final long DEFAULT_HEARTBEAT_RATE_MILLIS = 5000L;
    private static final long DEFAULT_TTL_MILLIS = 10000L;
    private static final int MAX_HEARTBEAT_RETRIES = 3;
    private final String ownerName;
    private final Clock clock;
    private final Registry registry;
    private final ObjectMapper objectMapper;
    private final RedisClientDelegate redisClientDelegate;
    private final ScheduledExecutorService scheduledExecutorService;
    private final Id acquireId;
    private final Id releaseId;
    private final Id heartbeatId;
    private final Id acquireDurationId;
    private long heartbeatRateMillis;
    private long leaseDurationMillis;
    private BlockingDeque<RefreshableLockManager.HeartbeatLockRequest> heartbeatQueue;

    public RedisLockManager(String ownerName, Clock clock, Registry registry, ObjectMapper objectMapper, RedisClientDelegate redisClientDelegate, Optional<Long> heartbeatRateMillis, Optional<Long> leaseDurationMillis) {
        this.ownerName = Optional.ofNullable(ownerName).orElse(this.getOwnerName());
        this.clock = clock;
        this.registry = registry;
        this.objectMapper = objectMapper;
        this.redisClientDelegate = redisClientDelegate;
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();
        this.heartbeatQueue = new LinkedBlockingDeque<RefreshableLockManager.HeartbeatLockRequest>();
        this.heartbeatRateMillis = heartbeatRateMillis.orElse(5000L);
        this.leaseDurationMillis = leaseDurationMillis.orElse(10000L);
        this.acquireId = registry.createId("kork.lock.acquire");
        this.releaseId = registry.createId("kork.lock.release");
        this.heartbeatId = registry.createId("kork.lock.heartbeat");
        this.acquireDurationId = registry.createId("kork.lock.acquire.duration");
        this.scheduleHeartbeats();
    }

    public RedisLockManager(String ownerName, Clock clock, Registry registry, ObjectMapper objectMapper, RedisClientDelegate redisClientDelegate) {
        this(ownerName, clock, registry, objectMapper, redisClientDelegate, Optional.of(5000L), Optional.of(10000L));
    }

    public <R> LockManager.AcquireLockResponse<R> acquireLock(@Nonnull LockManager.LockOptions lockOptions, @Nonnull Callable<R> onLockAcquiredCallback) {
        return this.acquire(lockOptions, onLockAcquiredCallback);
    }

    public <R> LockManager.AcquireLockResponse<R> acquireLock(@Nonnull String lockName, long maximumLockDurationMillis, @Nonnull Callable<R> onLockAcquiredCallback) {
        LockManager.LockOptions lockOptions = new LockManager.LockOptions().withLockName(lockName).withMaximumLockDuration(Duration.ofMillis(maximumLockDurationMillis));
        return this.acquire(lockOptions, onLockAcquiredCallback);
    }

    public LockManager.AcquireLockResponse<Void> acquireLock(@Nonnull String lockName, long maximumLockDurationMillis, @Nonnull Runnable onLockAcquiredCallback) {
        LockManager.LockOptions lockOptions = new LockManager.LockOptions().withLockName(lockName).withMaximumLockDuration(Duration.ofMillis(maximumLockDurationMillis));
        return this.acquire(lockOptions, onLockAcquiredCallback);
    }

    public LockManager.AcquireLockResponse<Void> acquireLock(@Nonnull LockManager.LockOptions lockOptions, @Nonnull Runnable onLockAcquiredCallback) {
        return this.acquire(lockOptions, onLockAcquiredCallback);
    }

    public boolean releaseLock(@Nonnull LockManager.Lock lock, boolean wasWorkSuccessful) {
        Id lockRelease = this.releaseId.withTag("lockName", lock.getName());
        String status = this.tryReleaseLock(lock, wasWorkSuccessful);
        this.registry.counter(lockRelease.withTag("status", status)).increment();
        switch (status) {
            case "SUCCESS": 
            case "SUCCESS_GONE": {
                log.info("Released lock (wasWorkSuccessful: {}, {})", (Object)wasWorkSuccessful, (Object)lock);
                return true;
            }
            case "FAILED_NOT_OWNER": {
                log.warn("Failed releasing lock, not owner (wasWorkSuccessful: {}, {})", (Object)wasWorkSuccessful, (Object)lock);
                return false;
            }
        }
        log.error("Unknown release response code {} (wasWorkSuccessful: {}, {})", new Object[]{status, wasWorkSuccessful, lock});
        return false;
    }

    public RefreshableLockManager.HeartbeatResponse heartbeat(RefreshableLockManager.HeartbeatLockRequest heartbeatLockRequest) {
        return this.doHeartbeat(heartbeatLockRequest);
    }

    public void queueHeartbeat(RefreshableLockManager.HeartbeatLockRequest heartbeatLockRequest) {
        if (!this.heartbeatQueue.contains(heartbeatLockRequest)) {
            log.info("Lock {} will heartbeats for {}ms", (Object)heartbeatLockRequest.getLock(), (Object)heartbeatLockRequest.getHeartbeatDuration().toMillis());
            this.heartbeatQueue.add(heartbeatLockRequest);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <R> LockManager.AcquireLockResponse<R> doAcquire(@Nonnull LockManager.LockOptions lockOptions, Optional<Callable<R>> onLockAcquiredCallbackCallable, Optional<Runnable> onLockAcquiredCallbackRunnable) {
        lockOptions.validateInputs();
        LockManager.Lock lock = null;
        Object workResult = null;
        LockManager.LockStatus status = LockManager.LockStatus.TAKEN;
        RefreshableLockManager.HeartbeatLockRequest heartbeatLockRequest = null;
        if (lockOptions.getVersion() == null || !lockOptions.isReuseVersion()) {
            lockOptions.setVersion(this.clock.millis());
        }
        try {
            lock = this.tryCreateLock(lockOptions);
            if (!this.matchesLock(lockOptions, lock)) {
                log.debug("Could not acquire already taken lock {}", (Object)lock);
                LockManager.AcquireLockResponse acquireLockResponse = new LockManager.AcquireLockResponse(lock, null, status, null, false);
                return acquireLockResponse;
            }
            LongTaskTimer acquireDurationTimer = LongTaskTimer.get((Registry)this.registry, (Id)this.acquireDurationId.withTag("lockName", lock.getName()));
            status = LockManager.LockStatus.ACQUIRED;
            log.info("Acquired Lock {}.", (Object)lock);
            long timer = acquireDurationTimer.start();
            AtomicInteger heartbeatRetriesOnFailure = new AtomicInteger(3);
            heartbeatLockRequest = new RefreshableLockManager.HeartbeatLockRequest(lock, heartbeatRetriesOnFailure, this.clock, lockOptions.getMaximumLockDuration(), lockOptions.isReuseVersion());
            this.queueHeartbeat(heartbeatLockRequest);
            LockManager.Lock lock2 = heartbeatLockRequest.getLock();
            synchronized (lock2) {
                try {
                    if (onLockAcquiredCallbackCallable.isPresent()) {
                        workResult = onLockAcquiredCallbackCallable.get().call();
                    } else {
                        onLockAcquiredCallbackRunnable.ifPresent(Runnable::run);
                    }
                }
                catch (Exception e) {
                    log.error("Callback failed using lock {}", (Object)lock, (Object)e);
                    throw new LockManager.LockCallbackException((Throwable)e);
                }
                finally {
                    acquireDurationTimer.stop(timer);
                }
            }
            this.heartbeatQueue.remove(heartbeatLockRequest);
            lock = this.findAuthoritativeLockOrNull(lock);
            lock2 = new LockManager.AcquireLockResponse(lock, workResult, status, null, this.tryLockReleaseQuietly(lock, true));
            return lock2;
        }
        catch (Exception e) {
            log.error(e.getMessage());
            this.heartbeatQueue.remove(heartbeatLockRequest);
            lock = this.findAuthoritativeLockOrNull(lock);
            boolean lockWasReleased = this.tryLockReleaseQuietly(lock, false);
            if (e instanceof LockManager.LockCallbackException) {
                throw e;
            }
            status = LockManager.LockStatus.ERROR;
            LockManager.AcquireLockResponse acquireLockResponse = new LockManager.AcquireLockResponse(lock, workResult, status, e, lockWasReleased);
            return acquireLockResponse;
        }
        finally {
            this.registry.counter(this.acquireId.withTag("lockName", lockOptions.getLockName()).withTag("status", status.toString())).increment();
        }
    }

    private LockManager.AcquireLockResponse<Void> acquire(@Nonnull LockManager.LockOptions lockOptions, @Nonnull Runnable onLockAcquiredCallback) {
        return this.doAcquire(lockOptions, Optional.empty(), Optional.of(onLockAcquiredCallback));
    }

    private <R> LockManager.AcquireLockResponse<R> acquire(@Nonnull LockManager.LockOptions lockOptions, @Nonnull Callable<R> onLockAcquiredCallback) {
        return this.doAcquire(lockOptions, Optional.of(onLockAcquiredCallback), Optional.empty());
    }

    @PreDestroy
    private void shutdownHeartbeatScheduler() {
        this.scheduledExecutorService.shutdown();
    }

    private void scheduleHeartbeats() {
        this.scheduledExecutorService.scheduleAtFixedRate(this::sendHeartbeats, 0L, this.heartbeatRateMillis, TimeUnit.MILLISECONDS);
    }

    private void sendHeartbeats() {
        block8: {
            if (this.heartbeatQueue.isEmpty()) {
                return;
            }
            RefreshableLockManager.HeartbeatLockRequest heartbeatLockRequest = (RefreshableLockManager.HeartbeatLockRequest)this.heartbeatQueue.getFirst();
            if (heartbeatLockRequest.timesUp()) {
                log.warn("***MAX HEARTBEAT REACHED***. No longer sending heartbeats to {}", (Object)heartbeatLockRequest.getLock());
                this.heartbeatQueue.remove(heartbeatLockRequest);
                this.registry.counter(this.heartbeatId.withTag("lockName", heartbeatLockRequest.getLock().getName()).withTag("status", RefreshableLockManager.LockHeartbeatStatus.MAX_HEARTBEAT_REACHED.toString())).increment();
            } else {
                try {
                    RefreshableLockManager.HeartbeatResponse heartbeatResponse = this.heartbeat(heartbeatLockRequest);
                    switch (heartbeatResponse.getLockStatus()) {
                        case EXPIRED: 
                        case ERROR: {
                            log.warn("Lock status {} for {}", (Object)heartbeatResponse.getLockStatus(), (Object)heartbeatResponse.getLock());
                            this.heartbeatQueue.remove(heartbeatLockRequest);
                            break;
                        }
                        default: {
                            log.debug("Remaining lock duration {}ms. Refreshed lock {}", (Object)heartbeatLockRequest.getRemainingLockDuration().toMillis(), (Object)heartbeatResponse.getLock());
                            heartbeatLockRequest.setLock(heartbeatResponse.getLock());
                            break;
                        }
                    }
                }
                catch (Exception e) {
                    log.error("Heartbeat {} for {} failed", new Object[]{heartbeatLockRequest, heartbeatLockRequest.getLock(), e});
                    if (heartbeatLockRequest.shouldRetry()) break block8;
                    this.heartbeatQueue.remove(heartbeatLockRequest);
                }
            }
        }
    }

    private RefreshableLockManager.HeartbeatResponse doHeartbeat(RefreshableLockManager.HeartbeatLockRequest heartbeatLockRequest) {
        LockManager.Lock lock = heartbeatLockRequest.getLock();
        long nextVersion = heartbeatLockRequest.reuseVersion() ? lock.getVersion() : lock.nextVersion();
        Id lockHeartbeat = this.heartbeatId.withTag("lockName", lock.getName());
        LockManager.Lock extendedLock = lock;
        try {
            extendedLock = this.tryUpdateLock(lock, nextVersion);
            this.registry.counter(lockHeartbeat.withTag("status", RefreshableLockManager.LockHeartbeatStatus.SUCCESS.toString())).increment();
            return new RefreshableLockManager.HeartbeatResponse(extendedLock, RefreshableLockManager.LockHeartbeatStatus.SUCCESS);
        }
        catch (Exception e) {
            if (e instanceof LockManager.LockExpiredException) {
                this.registry.counter(lockHeartbeat.withTag("status", RefreshableLockManager.LockHeartbeatStatus.EXPIRED.toString())).increment();
                return new RefreshableLockManager.HeartbeatResponse(extendedLock, RefreshableLockManager.LockHeartbeatStatus.EXPIRED);
            }
            log.error("Heartbeat failed for lock {}", (Object)extendedLock, (Object)e);
            this.registry.counter(lockHeartbeat.withTag("status", RefreshableLockManager.LockHeartbeatStatus.ERROR.toString())).increment();
            return new RefreshableLockManager.HeartbeatResponse(extendedLock, RefreshableLockManager.LockHeartbeatStatus.ERROR);
        }
    }

    private boolean tryLockReleaseQuietly(LockManager.Lock lock, boolean wasWorkSuccessful) {
        if (lock != null) {
            try {
                return this.releaseLock(lock, wasWorkSuccessful);
            }
            catch (Exception e) {
                log.warn("Attempt to release lock {} failed", (Object)lock, (Object)e);
                return false;
            }
        }
        return true;
    }

    private boolean matchesLock(LockManager.LockOptions lockOptions, LockManager.Lock lock) {
        return this.ownerName.equals(lock.getOwnerName()) && lockOptions.getVersion().longValue() == lock.getVersion();
    }

    private LockManager.Lock findAuthoritativeLockOrNull(LockManager.Lock lock) {
        Object payload = this.redisClientDelegate.withScriptingClient(c -> c.eval("local payload = redis.call('GET', KEYS[1]) if payload then  local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] then    return redis.call('GET', KEYS[1])  end end", Arrays.asList(this.lockKey(lock.getName())), Arrays.asList(this.ownerName)));
        if (payload == null) {
            return null;
        }
        try {
            return (LockManager.Lock)this.objectMapper.readValue(payload.toString(), LockManager.Lock.class);
        }
        catch (IOException e) {
            log.error("Failed to get lock info for {}", (Object)lock, (Object)e);
            return null;
        }
    }

    public LockManager.Lock tryCreateLock(LockManager.LockOptions lockOptions) {
        try {
            List attributes = Optional.ofNullable(lockOptions.getAttributes()).orElse(Collections.emptyList());
            Object payload = this.redisClientDelegate.withScriptingClient(c -> c.eval("local payload = cjson.encode({  ['leaseDurationMillis']=ARGV[1],  ['successIntervalMillis']=ARGV[3],  ['failureIntervalMillis']=ARGV[4],  ['ownerName']=ARGV[5],  ['ownerSystemTimestamp']=ARGV[6],  ['version']=ARGV[7],  ['name']=ARGV[8],  ['attributes']=ARGV[9]}) if redis.call('SET', KEYS[1], payload, 'NX', 'EX', ARGV[2]) == 'OK' then  return payload end return redis.call('GET', KEYS[1])", Arrays.asList(this.lockKey(lockOptions.getLockName())), Arrays.asList(Long.toString(Duration.ofMillis(this.leaseDurationMillis).toMillis()), Long.toString(Duration.ofMillis(this.leaseDurationMillis).getSeconds()), Long.toString(lockOptions.getSuccessInterval().toMillis()), Long.toString(lockOptions.getFailureInterval().toMillis()), this.ownerName, Long.toString(this.clock.millis()), String.valueOf(lockOptions.getVersion()), lockOptions.getLockName(), String.join((CharSequence)";", attributes))));
            if (payload == null) {
                throw new LockManager.LockNotAcquiredException(String.format("Lock not acquired %s", lockOptions));
            }
            return (LockManager.Lock)this.objectMapper.readValue(payload.toString(), LockManager.Lock.class);
        }
        catch (IOException e) {
            throw new LockManager.LockNotAcquiredException(String.format("Lock not acquired %s", lockOptions), (Throwable)e);
        }
    }

    private String tryReleaseLock(LockManager.Lock lock, boolean wasWorkSuccessful) {
        long releaseTtl = wasWorkSuccessful ? lock.getSuccessIntervalMillis() : lock.getFailureIntervalMillis();
        Object payload = this.redisClientDelegate.withScriptingClient(c -> c.eval("local payload = redis.call('GET', KEYS[1]) if payload then local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] and lock['version'] == ARGV[2] then    redis.call('EXPIRE', KEYS[1], ARGV[3])    return 'SUCCESS'  end  return 'FAILED_NOT_OWNER' end return 'SUCCESS_GONE'", Arrays.asList(this.lockKey(lock.getName())), Arrays.asList(this.ownerName, String.valueOf(lock.getVersion()), String.valueOf(Duration.ofMillis(releaseTtl).getSeconds()))));
        return payload.toString();
    }

    private LockManager.Lock tryUpdateLock(LockManager.Lock lock, long nextVersion) {
        Object payload = this.redisClientDelegate.withScriptingClient(c -> c.eval("local payload = redis.call('GET', KEYS[1]) if payload then  local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] and lock['version'] == ARGV[2] then    lock['version']=ARGV[3]    lock['leaseDurationMillis']=ARGV[4]    lock['ownerSystemTimestamp']=ARGV[5]    redis.call('PSETEX', KEYS[1], ARGV[4], cjson.encode(lock))    return redis.call('GET', KEYS[1])  end end", Arrays.asList(this.lockKey(lock.getName())), Arrays.asList(this.ownerName, String.valueOf(lock.getVersion()), String.valueOf(nextVersion), Long.toString(lock.getLeaseDurationMillis()), Long.toString(this.clock.millis()))));
        if (payload == null) {
            throw new LockManager.LockExpiredException(String.format("Lock expired %s", lock));
        }
        try {
            return (LockManager.Lock)this.objectMapper.readValue(payload.toString(), LockManager.Lock.class);
        }
        catch (IOException e) {
            throw new RefreshableLockManager.LockFailedHeartbeatException(String.format("Lock not acquired %s", lock), (Throwable)e);
        }
    }

    static interface LockScripts {
        public static final String RELEASE_SCRIPT = "local payload = redis.call('GET', KEYS[1]) if payload then local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] and lock['version'] == ARGV[2] then    redis.call('EXPIRE', KEYS[1], ARGV[3])    return 'SUCCESS'  end  return 'FAILED_NOT_OWNER' end return 'SUCCESS_GONE'";
        public static final String ACQUIRE_SCRIPT = "local payload = cjson.encode({  ['leaseDurationMillis']=ARGV[1],  ['successIntervalMillis']=ARGV[3],  ['failureIntervalMillis']=ARGV[4],  ['ownerName']=ARGV[5],  ['ownerSystemTimestamp']=ARGV[6],  ['version']=ARGV[7],  ['name']=ARGV[8],  ['attributes']=ARGV[9]}) if redis.call('SET', KEYS[1], payload, 'NX', 'EX', ARGV[2]) == 'OK' then  return payload end return redis.call('GET', KEYS[1])";
        public static final String FIND_SCRIPT = "local payload = redis.call('GET', KEYS[1]) if payload then  local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] then    return redis.call('GET', KEYS[1])  end end";
        public static final String HEARTBEAT_SCRIPT = "local payload = redis.call('GET', KEYS[1]) if payload then  local lock = cjson.decode(payload)  if lock['ownerName'] == ARGV[1] and lock['version'] == ARGV[2] then    lock['version']=ARGV[3]    lock['leaseDurationMillis']=ARGV[4]    lock['ownerSystemTimestamp']=ARGV[5]    redis.call('PSETEX', KEYS[1], ARGV[4], cjson.encode(lock))    return redis.call('GET', KEYS[1])  end end";
    }
}

