/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.kork.secrets.engines;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.api.gax.rpc.ApiException;
import com.google.cloud.secretmanager.v1.AccessSecretVersionResponse;
import com.google.cloud.secretmanager.v1.SecretManagerServiceClient;
import com.google.cloud.secretmanager.v1.SecretPayload;
import com.google.cloud.secretmanager.v1.SecretVersionName;
import com.netflix.spinnaker.kork.secrets.EncryptedSecret;
import com.netflix.spinnaker.kork.secrets.InvalidSecretFormatException;
import com.netflix.spinnaker.kork.secrets.SecretEngine;
import com.netflix.spinnaker.kork.secrets.SecretException;
import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.springframework.stereotype.Component;

@Component
public class GoogleSecretsManagerSecretEngine
implements SecretEngine {
    private static final String PROJECT_NUMBER = "p";
    private static final String SECRET_ID = "s";
    private static final String SECRET_KEY = "k";
    private static final String VERSION_ID = "v";
    private static final String LATEST = "latest";
    private static final String IDENTIFIER = "google-secrets-manager";
    private final Map<String, Map<String, String>> cache = new ConcurrentHashMap<String, Map<String, String>>();
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static SecretManagerServiceClient client;

    public String identifier() {
        return IDENTIFIER;
    }

    public byte[] decrypt(EncryptedSecret encryptedSecret) {
        String projectNumber = (String)encryptedSecret.getParams().get(PROJECT_NUMBER);
        String secretId = (String)encryptedSecret.getParams().get(SECRET_ID);
        String secretKey = (String)encryptedSecret.getParams().get(SECRET_KEY);
        String secretVersion = (String)encryptedSecret.getParams().get(VERSION_ID);
        if (encryptedSecret.isEncryptedFile()) {
            return this.getSecretPayload(projectNumber, secretId, secretVersion).getData().toStringUtf8().getBytes();
        }
        if (secretKey != null) {
            return this.getSecretPayloadString(projectNumber, secretId, secretVersion, secretKey);
        }
        return this.getSecretPayloadString(projectNumber, secretId, secretVersion);
    }

    public void validate(EncryptedSecret encryptedSecret) {
        Set paramNamesSet = encryptedSecret.getParams().keySet();
        if (!paramNamesSet.contains(PROJECT_NUMBER)) {
            throw new InvalidSecretFormatException("Project number parameter is missing (p=...)");
        }
        if (!paramNamesSet.contains(SECRET_ID)) {
            throw new InvalidSecretFormatException("Secret id parameter is missing (s=...)");
        }
        if (encryptedSecret.isEncryptedFile() && paramNamesSet.contains(SECRET_KEY)) {
            throw new InvalidSecretFormatException("Encrypted file should not specify key");
        }
    }

    protected SecretPayload getSecretPayload(String projectNumber, String secretId, String secretVersion) {
        try {
            if (client == null) {
                client = SecretManagerServiceClient.create();
            }
            if (secretVersion == null) {
                secretVersion = LATEST;
            }
            SecretVersionName secretVersionName = SecretVersionName.of((String)projectNumber, (String)secretId, (String)secretVersion);
            AccessSecretVersionResponse response = client.accessSecretVersion(secretVersionName);
            return response.getPayload();
        }
        catch (ApiException | IOException e) {
            throw new SecretException(String.format("Failed to parse secret when using Google Secrets Manager to fetch: [projectNumber: %s, secretId: %s]", projectNumber, secretId), e);
        }
    }

    public void clearCache() {
        this.cache.clear();
    }

    private byte[] getSecretPayloadString(String projectNumber, String secretId, String secretVersion, String secretKey) {
        if (!this.cache.containsKey(secretId)) {
            String secretString = this.getSecretPayload(projectNumber, secretId, secretVersion).getData().toStringUtf8();
            try {
                Map map = (Map)objectMapper.readValue(secretString, Map.class);
                this.cache.put(secretId, map);
            }
            catch (JsonProcessingException | IllegalArgumentException e) {
                throw new SecretException(String.format("Failed to parse secret when using Google Secrets Manager to fetch: [projectNumber: %s, secretId: %s, secretKey: %s]", projectNumber, secretId, secretKey), e);
            }
        }
        return Optional.ofNullable(this.cache.get(secretId).get(secretKey)).orElseThrow(() -> new SecretException(String.format("Specified key not found in Google Secrets Manager: [projectNumber: %s, secretId: %s, secretKey: %s]", projectNumber, secretId, secretKey))).getBytes();
    }

    private byte[] getSecretPayloadString(String projectNumber, String secretId, String secretVersion) {
        return this.getSecretPayload(projectNumber, secretId, secretVersion).getData().toStringUtf8().getBytes();
    }
}

