/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spectator.stackdriver;

import com.google.api.client.http.HttpResponseException;
import com.google.api.services.monitoring.v3.Monitoring;
import com.google.api.services.monitoring.v3.model.CreateTimeSeriesRequest;
import com.google.api.services.monitoring.v3.model.Metric;
import com.google.api.services.monitoring.v3.model.MonitoredResource;
import com.google.api.services.monitoring.v3.model.Point;
import com.google.api.services.monitoring.v3.model.TimeInterval;
import com.google.api.services.monitoring.v3.model.TimeSeries;
import com.google.api.services.monitoring.v3.model.TypedValue;
import com.google.common.collect.Lists;
import com.netflix.spectator.api.Clock;
import com.netflix.spectator.api.DefaultRegistry;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Measurement;
import com.netflix.spectator.api.Meter;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Tag;
import com.netflix.spectator.stackdriver.ConfigParams;
import com.netflix.spectator.stackdriver.MetricDescriptorCache;
import com.netflix.spectator.stackdriver.MonitoredResourceBuilder;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StackdriverWriter {
    private static final Pattern INVALID_LABEL_REGEX = Pattern.compile("timeSeries\\[(\\d+?)\\]\\.metric\\.labels\\[\\d+?\\] had an invalid value of \\W*(\\w+)");
    public static final String WRITE_TIMER_NAME = "stackdriver.writeRegistry";
    private static final int MAX_TS_PER_REQUEST = 200;
    private static final Registry ID_FACTORY = new DefaultRegistry(Clock.SYSTEM);
    private final Logger log = LoggerFactory.getLogger((String)"StackdriverWriter");
    private final SimpleDateFormat rfc3339;
    protected MonitoredResource monitoredResource;
    private final Monitoring service;
    private final String projectResourceName;
    private final String applicationName;
    private String instanceId;
    private MetricDescriptorCache cache;
    private String counterStartTimeRfc3339;
    private Predicate<Measurement> measurementFilter;
    private Map<Id, Id> timerBaseIds = new HashMap<Id, Id>();

    public MetricDescriptorCache getDescriptorCache() {
        return this.cache;
    }

    public StackdriverWriter(ConfigParams configParams) {
        this.cache = configParams.getDescriptorCache();
        this.service = configParams.getStackdriverStub();
        this.projectResourceName = "projects/" + configParams.getProjectName();
        this.applicationName = configParams.getApplicationName();
        this.instanceId = configParams.getInstanceId();
        this.measurementFilter = configParams.getMeasurementFilter();
        this.rfc3339 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'000000Z'");
        this.rfc3339.setTimeZone(TimeZone.getTimeZone("GMT"));
        Date startDate = new Date(configParams.getCounterStartTime());
        this.counterStartTimeRfc3339 = this.rfc3339.format(startDate);
        this.log.info("Constructing StackdriverWriter {}={}", (Object)"InstanceSrc", (Object)this.instanceId);
    }

    private void handleTimeSeriesResponseException(HttpResponseException rex, String msg, List<TimeSeries> nextN) {
        Matcher matcher = INVALID_LABEL_REGEX.matcher(rex.getContent());
        TimeSeries ts = null;
        String label = null;
        if (matcher.find()) {
            int tsIndex = Integer.parseInt(matcher.group(1));
            ts = nextN.get(tsIndex);
            label = matcher.group(2);
            this.log.error("{}:  time series element: {}", (Object)rex.getMessage(), (Object)ts.toString());
            this.cache.addLabel(ts.getMetric().getType(), label);
            try {
                this.log.info("Retrying individual time series element");
                CreateTimeSeriesRequest tsRequest = new CreateTimeSeriesRequest();
                tsRequest.setTimeSeries(nextN.subList(tsIndex, tsIndex + 1));
                this.service.projects().timeSeries().create(this.projectResourceName, tsRequest).execute();
            }
            catch (IOException ioex) {
                this.log.error("Retry failed with " + String.valueOf(ioex));
            }
        } else {
            this.log.error("Caught HttpResponseException {}", (Object)msg, (Object)rex);
        }
    }

    public TimeSeries measurementToTimeSeries(String descriptorType, Registry registry, Meter meter, Measurement measurement) {
        Map<String, String> labels = this.cache.tagsToTimeSeriesLabels(descriptorType, measurement.id().tags());
        long millis = measurement.timestamp();
        double value = measurement.value();
        TimeInterval timeInterval = new TimeInterval();
        Date date = new Date(millis);
        timeInterval.setEndTime(this.rfc3339.format(date));
        String descriptorKind = this.cache.descriptorTypeToKind(descriptorType, registry, meter);
        if (descriptorKind == "CUMULATIVE") {
            timeInterval.setStartTime(this.counterStartTimeRfc3339);
        }
        TypedValue typedValue = new TypedValue();
        typedValue.setDoubleValue(Double.valueOf(value));
        Point point = new Point();
        point.setValue(typedValue);
        point.setInterval(timeInterval);
        Metric metric = new Metric();
        metric.setType(descriptorType);
        metric.setLabels(labels);
        TimeSeries ts = new TimeSeries();
        ts.setResource(this.monitoredResource);
        ts.setMetric(metric);
        ts.setMetricKind(descriptorKind);
        ts.setValueType("DOUBLE");
        ts.setPoints((List)Lists.newArrayList((Object[])new Point[]{point}));
        return ts;
    }

    Id deriveBaseTimerId(Id id) {
        String suffix = null;
        ArrayList<Tag> tags = new ArrayList<Tag>();
        for (Tag tag : id.tags()) {
            if (tag.key().equals("statistic")) {
                if (tag.value().equals("totalTime")) {
                    suffix = "totalTime";
                    continue;
                }
                if (tag.value().equals("count")) {
                    suffix = "count";
                    continue;
                }
                throw new IllegalStateException("Unexpected statistic=" + tag.value());
            }
            tags.add(tag);
        }
        if (suffix == null) {
            return id;
        }
        return ID_FACTORY.createId(id.name() + "__" + suffix).withTags(tags);
    }

    private Iterable<Measurement> transformTimerMeasurements(Iterable<Measurement> measurements) {
        ArrayList<Measurement> result = new ArrayList<Measurement>();
        for (Measurement measurement : measurements) {
            if (!this.measurementFilter.test(measurement)) continue;
            Id id = this.timerBaseIds.computeIfAbsent(measurement.id(), k -> this.deriveBaseTimerId((Id)k));
            result.add(new Measurement(id, measurement.timestamp(), measurement.value()));
        }
        return result;
    }

    void addMeterToTimeSeries(Registry registry, Meter meter, List<TimeSeries> tsList) {
        Iterable<Measurement> measurements = meter.measure();
        boolean applyFilter = true;
        if (this.cache.meterIsTimer(registry, meter)) {
            measurements = this.transformTimerMeasurements(measurements);
            applyFilter = false;
        }
        for (Measurement measurement : measurements) {
            if (applyFilter && !this.measurementFilter.test(measurement)) continue;
            String descriptorType = this.cache.idToDescriptorType(measurement.id());
            tsList.add(this.measurementToTimeSeries(descriptorType, registry, meter, measurement));
        }
    }

    public List<TimeSeries> registryToTimeSeries(Registry registry) {
        this.log.debug("Collecting metrics...");
        ArrayList<TimeSeries> tsList = new ArrayList<TimeSeries>();
        Iterator iterator = registry.iterator();
        while (iterator.hasNext()) {
            this.addMeterToTimeSeries(registry, (Meter)iterator.next(), tsList);
        }
        return tsList;
    }

    public void writeRegistry(final Registry registry) {
        Id writeId = registry.createId(WRITE_TIMER_NAME);
        registry.timer(writeId).record(new Runnable(){

            @Override
            public void run() {
                StackdriverWriter.this.writeRegistryHelper(registry);
            }
        });
    }

    MonitoredResource determineMonitoredResource() {
        if (this.monitoredResource == null) {
            String project = this.projectResourceName.substring(this.projectResourceName.indexOf(47) + 1);
            try {
                this.monitoredResource = new MonitoredResourceBuilder().setStackdriverProject(project).build();
                if (this.monitoredResource.getType().equals("global")) {
                    this.cache.addExtraTimeSeriesLabel("InstanceSrc", this.instanceId);
                }
                this.log.info("Using monitoredResource={} with extraTimeSeriesLabels={}", (Object)this.monitoredResource, this.cache.getExtraTimeSeriesLabels());
            }
            catch (IOException ioex) {
                this.log.error("Unable to determine monitoredResource at this time.", (Throwable)ioex);
            }
        }
        return this.monitoredResource;
    }

    private void writeRegistryHelper(Registry registry) {
        MonitoredResource resource = this.determineMonitoredResource();
        if (resource == null) {
            this.log.warn("Cannot determine the managed resource - not flushing metrics.");
            return;
        }
        List<TimeSeries> tsList = this.registryToTimeSeries(registry);
        if (tsList.isEmpty()) {
            this.log.debug("No metric data points.");
            return;
        }
        CreateTimeSeriesRequest tsRequest = new CreateTimeSeriesRequest();
        int offset = 0;
        int failed = 0;
        this.log.debug("Writing metrics...");
        while (offset < tsList.size()) {
            List<TimeSeries> nextN;
            if (offset + 200 < tsList.size()) {
                nextN = tsList.subList(offset, offset + 200);
                offset += 200;
            } else {
                nextN = tsList.subList(offset, tsList.size());
                offset = tsList.size();
            }
            tsRequest.setTimeSeries(nextN);
            try {
                this.service.projects().timeSeries().create(this.projectResourceName, tsRequest).execute();
            }
            catch (HttpResponseException rex) {
                this.handleTimeSeriesResponseException(rex, "creating time series", nextN);
                failed += nextN.size();
            }
            catch (IOException ioex) {
                this.log.error("Caught Exception creating time series " + String.valueOf(ioex));
                failed += nextN.size();
            }
        }
        this.log.debug("Wrote {} values", (Object)(tsList.size() - failed));
    }
}

