/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.orca.webhook.tasks;

import com.google.common.base.Strings;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.PathNotFoundException;
import com.jayway.jsonpath.Predicate;
import com.netflix.spinnaker.orca.api.pipeline.OverridableTimeoutRetryableTask;
import com.netflix.spinnaker.orca.api.pipeline.TaskResult;
import com.netflix.spinnaker.orca.api.pipeline.models.ExecutionStatus;
import com.netflix.spinnaker.orca.api.pipeline.models.StageExecution;
import com.netflix.spinnaker.orca.webhook.config.WebhookProperties;
import com.netflix.spinnaker.orca.webhook.pipeline.WebhookStage;
import com.netflix.spinnaker.orca.webhook.service.WebhookService;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpStatusCodeException;

@Component
public class MonitorWebhookTask
implements OverridableTimeoutRetryableTask {
    private static final Logger log = LoggerFactory.getLogger(MonitorWebhookTask.class);
    private static final String JSON_PATH_NOT_FOUND_ERR_FMT = "Unable to parse %s: JSON property '%s' not found in response body";
    private final long backoffPeriod = TimeUnit.SECONDS.toMillis(1L);
    private final long timeout = TimeUnit.HOURS.toMillis(1L);
    private final WebhookService webhookService;
    private final WebhookProperties webhookProperties;

    @Autowired
    public MonitorWebhookTask(WebhookService webhookService, WebhookProperties webhookProperties) {
        this.webhookService = webhookService;
        this.webhookProperties = webhookProperties;
    }

    public long getBackoffPeriod() {
        return this.backoffPeriod;
    }

    public long getTimeout() {
        return this.timeout;
    }

    public long getDynamicBackoffPeriod(StageExecution stage, Duration taskDuration) {
        if (taskDuration.toMillis() > TimeUnit.MINUTES.toMillis(1L)) {
            return Math.max(this.backoffPeriod, TimeUnit.SECONDS.toMillis(15L));
        }
        return this.backoffPeriod;
    }

    public TaskResult execute(StageExecution stage) {
        Object result;
        ResponseEntity<Object> response;
        WebhookStage.StageData stageData = (WebhookStage.StageData)stage.mapTo(WebhookStage.StageData.class);
        if (StringUtils.isBlank((CharSequence)stageData.statusEndpoint) && !stageData.monitorOnly) {
            throw new IllegalStateException("Missing required parameter: statusEndpoint = " + stageData.statusEndpoint);
        }
        if (stageData.monitorOnly && StringUtils.isAllBlank((CharSequence[])new CharSequence[]{stageData.statusEndpoint, stageData.url})) {
            throw new IllegalStateException("Missing required parameter. Either webhook url or statusEndpoint are required");
        }
        HashMap<String, WebhookStage.WebhookResponseStageData> context = new HashMap<String, WebhookStage.WebhookResponseStageData>();
        WebhookStage.WebhookResponseStageData webhook = Optional.ofNullable(stageData.getWebhook()).orElseGet(WebhookStage.WebhookResponseStageData::new);
        context.put("webhook", webhook);
        WebhookStage.WebhookMonitorResponseStageData oldMonitor = webhook.getMonitor();
        WebhookStage.WebhookMonitorResponseStageData monitor = new WebhookStage.WebhookMonitorResponseStageData();
        webhook.setMonitor(monitor);
        ArrayList<Integer> pastStatusCodes = new ArrayList<Integer>();
        if (oldMonitor != null && oldMonitor.getPastStatusCodes() != null) {
            pastStatusCodes.addAll(oldMonitor.getPastStatusCodes());
            monitor.setPastStatusCodes(pastStatusCodes);
        }
        try {
            response = this.webhookService.getWebhookStatus(stage);
            log.debug("Received status code {} from status endpoint {} in execution {} in stage {}", new Object[]{response.getStatusCode(), stageData.statusEndpoint, stage.getExecution().getId(), stage.getId()});
        }
        catch (HttpStatusCodeException e) {
            HttpStatus statusCode = e.getStatusCode();
            if (this.shouldRetry(statusCode, stageData)) {
                log.warn("Failed to get webhook status from {} with statusCode={}, will retry", new Object[]{stageData.statusEndpoint, statusCode.value(), e});
                pastStatusCodes.add(statusCode.value());
                monitor.setPastStatusCodes(pastStatusCodes);
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.RUNNING).context(context).build();
            }
            String errorMessage = String.format("an exception occurred in webhook monitor to %s: %s", new Object[]{stageData.statusEndpoint, e});
            log.error(errorMessage, (Throwable)e);
            monitor.setError(errorMessage);
            Optional.ofNullable(e.getResponseHeaders()).filter(it -> !it.isEmpty()).map(HttpHeaders::toSingleValueMap).ifPresent(monitor::setHeaders);
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
        }
        catch (Exception e) {
            if (e instanceof UnknownHostException || e.getCause() instanceof UnknownHostException) {
                log.warn("name resolution failure in webhook for pipeline {} to {}, will retry.", new Object[]{stage.getExecution().getId(), stageData.statusEndpoint, e});
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.RUNNING).context(context).build();
            }
            if (e instanceof SocketTimeoutException || e.getCause() instanceof SocketTimeoutException) {
                log.warn("Socket timeout when polling {}, will retry.", (Object)stageData.statusEndpoint, (Object)e);
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.RUNNING).context(context).build();
            }
            String errorMessage = String.format("an exception occurred in webhook monitor to %s: %s", stageData.statusEndpoint, e);
            log.error(errorMessage, (Throwable)e);
            monitor.setError(errorMessage);
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
        }
        monitor.setBody(response.getBody());
        monitor.setStatusCode(response.getStatusCode());
        monitor.setStatusCodeValue(response.getStatusCode().value());
        if (!response.getHeaders().isEmpty()) {
            monitor.setHeaders(response.getHeaders().toSingleValueMap());
        }
        if (Strings.isNullOrEmpty((String)stageData.statusJsonPath)) {
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.SUCCEEDED).context(context).build();
        }
        try {
            result = JsonPath.read((Object)response.getBody(), (String)stageData.statusJsonPath, (Predicate[])new Predicate[0]);
        }
        catch (PathNotFoundException e) {
            monitor.setError(String.format(JSON_PATH_NOT_FOUND_ERR_FMT, "status", stageData.statusJsonPath));
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
        }
        if (!(result instanceof String || result instanceof Number || result instanceof Boolean)) {
            monitor.setError(String.format("The json path '%s' did not resolve to a single value", stageData.statusJsonPath));
            monitor.setResolvedValue(result);
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
        }
        if (StringUtils.isNotEmpty((CharSequence)stageData.progressJsonPath)) {
            Object progress;
            try {
                progress = JsonPath.read((Object)response.getBody(), (String)stageData.progressJsonPath, (Predicate[])new Predicate[0]);
            }
            catch (PathNotFoundException e) {
                monitor.setError(String.format(JSON_PATH_NOT_FOUND_ERR_FMT, "progress", stageData.statusJsonPath));
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
            }
            if (!(progress instanceof String)) {
                monitor.setError(String.format("The json path '%s' did not resolve to a String value", stageData.progressJsonPath));
                monitor.setResolvedValue(progress);
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(context).build();
            }
            monitor.setProgressMessage((String)progress);
        }
        if (result instanceof Number) {
            ExecutionStatus status = result.equals(100) ? ExecutionStatus.SUCCEEDED : ExecutionStatus.RUNNING;
            monitor.setPercentComplete((Number)result);
            return TaskResult.builder((ExecutionStatus)status).context(context).build();
        }
        Map<String, ExecutionStatus> statusMap = MonitorWebhookTask.createStatusMap(stageData.successStatuses, stageData.canceledStatuses, stageData.terminalStatuses);
        ExecutionStatus status = statusMap.getOrDefault(result.toString().toUpperCase(), ExecutionStatus.RUNNING);
        return TaskResult.builder((ExecutionStatus)status).context(context).build();
    }

    public void onCancel(@Nonnull StageExecution stage) {
        this.webhookService.cancelWebhook(stage);
    }

    private boolean shouldRetry(HttpStatus statusCode, WebhookStage.StageData stageData) {
        int status = statusCode.value();
        Map<Integer, WebhookStage.RetryData> retries = stageData.getRetries();
        if (retries != null && retries.containsKey(status)) {
            WebhookStage.RetryData retryConfig = retries.get(status);
            long attemptsWithStatus = this.countAttemptsWithStatus(status, stageData);
            return attemptsWithStatus < (long)retryConfig.getMaxAttempts();
        }
        return statusCode.is5xxServerError() || this.webhookProperties.getDefaultRetryStatusCodes().contains(status) || stageData.getRetryStatusCodes() != null && stageData.getRetryStatusCodes().contains(status);
    }

    private long countAttemptsWithStatus(int status, WebhookStage.StageData stageData) {
        return Optional.ofNullable(stageData.getWebhook()).map(WebhookStage.WebhookResponseStageData::getMonitor).map(WebhookStage.WebhookMonitorResponseStageData::getPastStatusCodes).map(past -> past.stream().filter(it -> status == it).count()).orElse(0L);
    }

    private static Map<String, ExecutionStatus> createStatusMap(String successStatuses, String canceledStatuses, String terminalStatuses) {
        HashMap<String, ExecutionStatus> statusMap = new HashMap<String, ExecutionStatus>();
        statusMap.putAll(MonitorWebhookTask.mapStatuses(successStatuses, ExecutionStatus.SUCCEEDED));
        if (StringUtils.isNotEmpty((CharSequence)canceledStatuses)) {
            statusMap.putAll(MonitorWebhookTask.mapStatuses(canceledStatuses, ExecutionStatus.CANCELED));
        }
        if (StringUtils.isNotEmpty((CharSequence)terminalStatuses)) {
            statusMap.putAll(MonitorWebhookTask.mapStatuses(terminalStatuses, ExecutionStatus.TERMINAL));
        }
        return statusMap;
    }

    private static Map<String, ExecutionStatus> mapStatuses(String statuses, ExecutionStatus status) {
        return Arrays.stream(statuses.split(",")).map(String::trim).map(String::toUpperCase).collect(Collectors.toMap(Function.identity(), ignore -> status));
    }
}

