/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.orca.webhook.tasks;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Predicate;
import com.netflix.spinnaker.kork.exceptions.SystemException;
import com.netflix.spinnaker.orca.api.pipeline.TaskResult;
import com.netflix.spinnaker.orca.api.pipeline.models.ExecutionStatus;
import com.netflix.spinnaker.orca.api.pipeline.models.StageExecution;
import com.netflix.spinnaker.orca.webhook.config.WebhookProperties;
import com.netflix.spinnaker.orca.webhook.pipeline.WebhookStage;
import com.netflix.spinnaker.orca.webhook.tasks.WebhookStatusCheckUrlRetriever;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.client.HttpStatusCodeException;

public class WebhookResponseProcessor {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final ObjectMapper objectMapper;
    private final WebhookStage.StageData stageData;
    private final WebhookProperties webhookProperties;
    private final StageExecution stageExecution;
    private final String executionId;

    public WebhookResponseProcessor(ObjectMapper objectMapper, StageExecution stageExecution, WebhookProperties webhookProperties) {
        this.objectMapper = objectMapper;
        this.webhookProperties = webhookProperties;
        this.stageExecution = stageExecution;
        this.stageData = (WebhookStage.StageData)stageExecution.mapTo(WebhookStage.StageData.class);
        this.executionId = stageExecution.getExecution().getId();
    }

    public TaskResult process(ResponseEntity<Object> response, Exception exceptionReceived) {
        if (exceptionReceived != null) {
            if (exceptionReceived instanceof HttpStatusCodeException) {
                return this.processReceivedHttpStatusException((HttpStatusCodeException)((Object)exceptionReceived));
            }
            return this.processClientException(exceptionReceived);
        }
        if (response != null) {
            return this.processResponse(response);
        }
        throw new SystemException("No response or exception is provided to process.");
    }

    private TaskResult processReceivedHttpStatusException(HttpStatusCodeException e) {
        WebhookStage.WebhookResponseStageData webhookOutput = new WebhookStage.WebhookResponseStageData();
        webhookOutput.setStatusCode(e.getStatusCode());
        webhookOutput.setStatusCodeValue(e.getStatusCode().value());
        if (e.getResponseHeaders() != null) {
            webhookOutput.setHeaders(e.getResponseHeaders().toSingleValueMap());
        }
        if (!StringUtils.isEmpty((Object)e.getResponseBodyAsString())) {
            webhookOutput.setBody(this.processResponseBodyAsJson(e.getResponseBodyAsString()));
        }
        TaskResult result = this.processReceivedFailureStatusCode(e.getStatusCode(), webhookOutput);
        this.log.warn(webhookOutput.getError(), (Throwable)e);
        return result;
    }

    private TaskResult processReceivedFailureStatusCode(HttpStatus status, WebhookStage.WebhookResponseStageData webHookOutput) {
        ExecutionStatus executionStatus;
        String errorMessage;
        if (this.stageData.failFastStatusCodes != null && this.stageData.failFastStatusCodes.contains(status.value())) {
            errorMessage = String.format("Received status code %s, which is configured to fail fast, terminating stage for pipeline %s to %s", status.value(), this.executionId, this.stageData.url);
            executionStatus = ExecutionStatus.TERMINAL;
        } else if (status.is5xxServerError() || this.webhookProperties.getDefaultRetryStatusCodes().contains(status.value()) || this.stageData.getWebhookRetryStatusCodes() != null && this.stageData.getWebhookRetryStatusCodes().contains(status.value())) {
            errorMessage = String.format("Error submitting webhook for pipeline %s to %s with status code %s, will retry.", this.executionId, this.stageData.url, status.value());
            executionStatus = ExecutionStatus.RUNNING;
        } else {
            errorMessage = String.format("Error submitting webhook for pipeline %s to %s with status code %s.", this.executionId, this.stageData.url, status.value());
            executionStatus = ExecutionStatus.TERMINAL;
        }
        webHookOutput.setError(errorMessage);
        return TaskResult.builder((ExecutionStatus)executionStatus).context(Map.of("webhook", webHookOutput)).build();
    }

    private TaskResult processClientException(Exception e) {
        ExecutionStatus executionStatus;
        String errorMessage;
        if (e instanceof UnknownHostException || e.getCause() instanceof UnknownHostException) {
            errorMessage = String.format("Remote host resolution failure in webhook for pipeline %s to %s, will retry.", this.executionId, this.stageData.url);
            executionStatus = ExecutionStatus.RUNNING;
        } else if (this.stageData.method == HttpMethod.GET && (e instanceof SocketTimeoutException || e.getCause() instanceof SocketTimeoutException)) {
            errorMessage = String.format("Socket timeout in webhook on GET request for pipeline %s to %s, will retry.", this.executionId, this.stageData.url);
            executionStatus = ExecutionStatus.RUNNING;
        } else {
            errorMessage = String.format("An exception occurred for pipeline %s performing a request to %s. %s", this.executionId, this.stageData.url, e.toString());
            executionStatus = ExecutionStatus.TERMINAL;
        }
        WebhookStage.WebhookResponseStageData webhookOutput = new WebhookStage.WebhookResponseStageData();
        webhookOutput.setError(errorMessage);
        this.log.warn(errorMessage, (Throwable)e);
        return TaskResult.builder((ExecutionStatus)executionStatus).context(Map.of("webhook", webhookOutput)).build();
    }

    private TaskResult processResponse(ResponseEntity response) {
        HttpStatus status;
        HashMap<String, Object> stageOutput = new HashMap<String, Object>();
        WebhookStage.WebhookResponseStageData webhookOutput = new WebhookStage.WebhookResponseStageData();
        stageOutput.put("webhook", webhookOutput);
        webhookOutput.setStatusCode(response.getStatusCode());
        webhookOutput.setStatusCodeValue(response.getStatusCode().value());
        if (response.getBody() != null) {
            webhookOutput.setBody(response.getBody());
        }
        if (!response.getHeaders().isEmpty()) {
            webhookOutput.setHeaders(response.getHeaders().toSingleValueMap());
        }
        if ((status = response.getStatusCode()).is2xxSuccessful() || status.is3xxRedirection()) {
            if (this.stageData.isWaitForCompletion()) {
                String statusUrl;
                try {
                    statusUrl = new WebhookStatusCheckUrlRetriever().getStatusCheckUrl(response, this.stageData);
                }
                catch (Exception e) {
                    webhookOutput.setError("Exception while resolving status check URL: " + e.getMessage());
                    this.log.error("Exception received while determining status check url", (Throwable)e);
                    return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(stageOutput).build();
                }
                if (StringUtils.isEmpty((Object)statusUrl)) {
                    webhookOutput.setError("The status URL couldn't be resolved, but 'Wait for completion' is configured");
                    return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(stageOutput).build();
                }
                this.stageData.setStatusEndpoint(statusUrl);
                this.stageExecution.getContext().put("statusEndpoint", statusUrl);
                webhookOutput.setStatusEndpoint(statusUrl);
                return TaskResult.builder((ExecutionStatus)ExecutionStatus.SUCCEEDED).context(stageOutput).build();
            }
            if (this.stageExecution.getContext().containsKey("expectedArtifacts") && !((List)this.stageExecution.getContext().get("expectedArtifacts")).isEmpty()) {
                try {
                    stageOutput.put("artifacts", JsonPath.parse((Object)response.getBody()).read("artifacts", new Predicate[0]));
                }
                catch (Exception e) {
                    webhookOutput.setError("Expected artifacts in webhook response couldn't be parsed: " + e.toString());
                    return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(stageOutput).build();
                }
            }
            return TaskResult.builder((ExecutionStatus)ExecutionStatus.SUCCEEDED).context(stageOutput).build();
        }
        webhookOutput.setError("The webhook request failed");
        return TaskResult.builder((ExecutionStatus)ExecutionStatus.TERMINAL).context(stageOutput).build();
    }

    private Object processResponseBodyAsJson(String responseBody) {
        Object body = null;
        try {
            if (responseBody.startsWith("{")) {
                body = this.objectMapper.readValue(responseBody, Map.class);
            } else if (responseBody.startsWith("[")) {
                body = this.objectMapper.readValue(responseBody, List.class);
            }
        }
        catch (JsonProcessingException ex) {
            this.log.warn("Failed to parse webhook payload as JSON", (Throwable)ex);
        }
        return body != null ? body : responseBody;
    }
}

