package com.alibaba.tesla.dag.accordion;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import com.alibaba.tesla.dag.ApplicationProperties;
import com.alibaba.tesla.dag.model.domain.TcDag;
import com.alibaba.tesla.dag.model.domain.TcDagInst;
import com.alibaba.tesla.dag.model.domain.TcDagInstNode;
import com.alibaba.tesla.dag.model.domain.TcDagNode;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeFormatType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeType;
import com.alibaba.tesla.dag.model.repository.TcDagInstNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagInstRepository;
import com.alibaba.tesla.dag.model.repository.TcDagRepository;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstStatus;
import com.alibaba.tesla.dag.schedule.task.TaskStatus;
import com.alibaba.tesla.dag.services.AbstractActionNewService;
import com.google.common.base.Throwables;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Service
public class AccordionDagService {

    @Autowired
    TcDagInstRepository dagInstRepository;

    @Autowired
    TcDagInstNodeRepository dagInstNodeRepository;

    @Autowired
    ApplicationProperties ap;

    @Autowired
    TcDagRepository dagRepository;

    @Autowired
    private List<AbstractActionNewService> actionNewServiceList;

    // 获取执行节点
    private static List<TcDagInstNode> getNodeList(TcDag dag, Long dagInstId) {
        List<TcDagInstNode> nodeList = new ArrayList<>();
        for (JSONObject nodeJson : dag.nodes()) {
            nodeList.add(genNode(dagInstId, nodeJson));
        }
        return nodeList;
    }

    // 根据图中节点生成实例节点
    private static TcDagInstNode genNode(Long dagInstId, JSONObject nodeJson) {
        return TcDagInstNode.builder()
                .gmtCreate(System.currentTimeMillis() / 1000)
                .gmtModified(System.currentTimeMillis() / 1000)
                .gmtStart(999999999999999L)
                .dagInstId(dagInstId)
                .tcDagContentNodeSpec(JSONObject.toJSONString(nodeJson))
                .status(DagInstNodeStatus.INIT.toString())
                .build()
                .updateTcDagOrNodeDetail(nodeJson)
                .updateNodeId();
    }

    public String getDagInstStatus(DagInstStatus status) {
        switch (status) {
            case STOPPED:
            case EXCEPTION:
                return "exception";
            case SUCCESS:
                return "success";
            case INIT:
            case RUNNING:
            case STOPPING:
            case WAIT_STOP:
            default:
                return "running";
        }
    }

    public List<TcDagInstNode> dagInstNodeListSortWithIndex(TcDag dag, List<TcDagInstNode> dagInstNodeList) {

        Map<String, JSONObject> nodeMap = dag.nodes().stream()
                .collect(Collectors.toMap(x -> x.getString("id"), x -> x));
        return dagInstNodeList.stream().sorted((dagInstNode1, dagInstNode2) -> {
            String nodeId1 = dagInstNode1.getNodeId();
            String nodeId2 = dagInstNode2.getNodeId();
            long nodeIndex1 = nodeMap.get(nodeId1).getLongValue("index");
            long nodeIndex2 = nodeMap.get(nodeId2).getLongValue("index");
            long gmtStart1 = dagInstNode1.getGmtStart();
            long gmtStart2 = dagInstNode2.getGmtStart();
            if (nodeIndex1 == nodeIndex2) {
                return Long.compare(gmtStart1, gmtStart2);
            }
            if (nodeIndex1 < 0 || nodeIndex2 < 0) {
                return -1 * Long.compare(nodeIndex1, nodeIndex2);
            } else {
                return Long.compare(nodeIndex1, nodeIndex2);
            }
        }).collect(Collectors.toList());

    }

    public String getDagInstShowStatus(Long dagInstId) {
        if (dagInstId == null) {
            return "waiting";
        }
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        String status = getDagInstStatus(dagInst.status());
        if (!"success".equals(status)) {
            return status;
        }
        List<TcDagInstNode> dagInstNodeList = dagInstNodeRepository.findAllByDagInstIdAndNodeIdNotInOrderByGmtStartAsc(
                dagInstId, Arrays.asList("__pre_node__", "__post_node__"));
        dagInstNodeList = dagInstNodeListSortWithIndex(dagInst.dag(), dagInstNodeList);
        List<String> nodeResultStatusList = new ArrayList<>();
        for (TcDagInstNode dagInstNode : dagInstNodeList) {
            switch (dagInstNode.type()) {
                case DAG:
                    nodeResultStatusList.add(getDagInstShowStatus(dagInstNode.getSubDagInstId()));
                    break;
                case NODE:
                    try {
                        JSONObject resultJson = AbstractActionNewService.getDataResult(dagInstNode.outJsonWithCache());
                        nodeResultStatusList.add(resultJson.getString("status"));
                    } catch (Exception ignored) {
                    }
                    break;
                default:
                    break;
            }
        }
        if (nodeResultStatusList.contains("ERROR") || nodeResultStatusList.contains("error")) {
            return "error";
        }
        if (nodeResultStatusList.contains("CRITICAL") || nodeResultStatusList.contains("critical")) {
            return "critical";
        }
        if (nodeResultStatusList.contains("WARNING") || nodeResultStatusList.contains("warning")) {
            return "warning";
        }
        return "success";
    }

    public String getDagInstNodeStatus(TcDagInstNode instNode) {
        if (instNode.type() == DagInstNodeType.NODE) {
            switch (instNode.status()) {
                case INIT:
                case SKIP_CAUSE_BY_EXCEPTION:
                case SKIP_CAUSE_BY_STOPPED:
                case SKIP:
                    return "waiting";
                case STOPPED:
                case EXCEPTION:
                    return "exception";
                case SUCCESS:
                    return "success";
                case RUNNING:
                case STOPPING:
                case WAIT_STOP:
                default:
                    return "running";
            }
        } else {
            return getDagInstShowStatus(instNode.getSubDagInstId());
        }
    }

    public CommonDiagnoseComponent getDag(Long dagId, String locale) {
        TcDag dag = dagRepository.findFirstById(dagId);
        CommonDiagnoseComponent commonDiagnoseComponent = CommonDiagnoseComponent.builder()
                .summary(CommonKvComponent.builder().build())
                .build();

        List<CommonDiagnoseStep> steps = commonDiagnoseComponent.getSteps();

        for (TcDagInstNode dagInstNode : getNodeList(dag, null)) {
            String alias = dagInstNode.alias();
            switch (dagInstNode.type()) {
                case NODE:
                    TcDagNode dagNode = dagInstNode.dagNode();
                    dagNode.obtainOptions(locale);
                    alias = dagNode.alias();
                    break;
                case DAG:
                    TcDag subDag = dagRepository.findFirstById(dagInstNode.defId());
                    subDag.obtainOptions(locale);
                    alias = subDag.alias();
                    break;
                default:
                    break;
            }

            steps.add(CommonDiagnoseStep.builder()
                    .name(alias)
                    .status("waiting")
                    .build());
        }
        return commonDiagnoseComponent;
    }

    public void cacheNodeOutJson(List<TcDagInstNode> dagInstNodeList) {
        dagInstNodeList.parallelStream().forEach(x -> {
            try {
                if (x.isEnd() && StringUtils.isNotEmpty(x.getTaskId()) && x.type() == DagInstNodeType.NODE) {
                    x.outJsonWithCache();
                }
            } catch (Exception e) {
                //log.error("x: " + JSONObject.toJSONString(x, true), e);
            }
        });
    }


    public CommonDiagnoseComponent getDagInst(Long dagInstId, String locale) throws Exception {
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        if (dagInst == null) {
            throw new Exception("not exist dagInstId: " + dagInstId);
        }
        TcDagInst lastDagInst = dagInstRepository.findFirstByDagIdAndStatusOrderByGmtCreateDesc(
                dagInst.getDagId(), DagInstStatus.SUCCESS.name()
        );
        CommonKvComponent summary = CommonKvComponent.builder()
                .submitter(dagInst.getCreator())
                .stime(dagInst.getGmtCreate())
                .etime(dagInst.getGmtModified())
                .status(getDagInstShowStatus(dagInstId))
                .build();
        if (lastDagInst != null) {
            summary.estimatedDuration = lastDagInst.getGmtAccess() - lastDagInst.getGmtCreate();
        }
        TcDagInstNode postNode = dagInst.postNode();
        if (postNode != null) {
            summary.setDetail(getDagInstNode(postNode, locale));
            if (postNode.status() == DagInstNodeStatus.SUCCESS) {
                summary.setDescription(
                        dagInst.globalResultJson()
                                .getJSONObject("__post_node__")
                                .getJSONObject("result")
                                .getString("description")
                );
            }
        }
        CommonDiagnoseComponent commonDiagnoseComponent = CommonDiagnoseComponent.builder()
                .summary(summary)
                .build();
        if (StringUtils.isNotEmpty(dagInst.getStatusDetail())) {
            summary.addKv("错误信息", dagInst.getStatusDetail());
        }
        List<TcDagInstNode> dagInstPrePostNodeList = dagInstNodeRepository
                .findAllByDagInstIdAndNodeIdInOrderByGmtStartAsc(dagInstId, Arrays.asList("__pre_node__", "__post_node__"));
        for (TcDagInstNode node : dagInstPrePostNodeList) {
            String taskId = node.getTaskId();
            String nodeId = node.nodeId();
            String name = nodeId.contains("pre") ? "pre节点" : "post节点";
            if (StringUtils.isNotEmpty(taskId)) {
                if (node.status() != DagInstNodeStatus.SUCCESS) {
                    summary.addLinkKv(name, ap.teslaTaskInstance(dagInst.getAppId(), taskId), "详情");
                }
            }
        }

        List<CommonDiagnoseStep> steps = commonDiagnoseComponent.getSteps();
        List<TcDagInstNode> dagInstNodeList = dagInstNodeRepository.findAllByDagInstIdAndNodeIdNotInOrderByGmtStartAsc(
                dagInstId, Arrays.asList("__pre_node__", "__post_node__"));
        dagInstNodeList = dagInstNodeListSortWithIndex(dagInst.dag(), dagInstNodeList);
        cacheNodeOutJson(dagInstNodeList);
        for (TcDagInstNode dagInstNode : dagInstNodeList) {
            if (dagInstNode.status() == DagInstNodeStatus.SKIP) {
                continue;
            }
            if (dagInstNode.type() == DagInstNodeType.NODE && !dagInstNode.dagNode().show()) {
                Long taskId = dagInstNode.taskId();
                if (taskId != null) {
                    TaskStatus status;

                    DagNodeType dagNodeType = dagInstNode.dagNode().type();
                    AbstractActionNewService abstractActionNewService = actionNewServiceList.stream().filter(
                            actionNewService -> actionNewService.registerNodeType() == dagNodeType)
                            .findFirst().orElse(null);

                    status = abstractActionNewService.status(taskId);

                    if (status != TaskStatus.EXCEPTION) {
                        continue;
                    }
                }
            }
            List<ComponentInterface> detail = new ArrayList<>();
            List<ComponentInterface> detailFromNode = getDagInstNode(dagInstNode, locale);
            CommonDiagnoseStep step = CommonDiagnoseStep.builder()
                    .name(dagInstNode.alias())
                    .nodeId(dagInstNode.nodeId())
                    .status(getDagInstNodeStatus(dagInstNode))
                    .logInfo(dagInstNode.getStatusDetail())
                    .detail(detail)
                    .build();
            if (dagInstNode.type() == DagInstNodeType.DAG && dagInstNode.status() == DagInstNodeStatus.SUCCESS) {
                JSONObject globalResultJson = dagInstNode.subDagInst().globalResultJson();
                if (globalResultJson.containsKey("__post_node__")) {
                    step.setDescription(
                            globalResultJson
                                    .getJSONObject("__post_node__")
                                    .getJSONObject("result")
                                    .getString("description")
                    );
                }
            }
            if (dagInstNode.type() == DagInstNodeType.NODE && dagInstNode.status() == DagInstNodeStatus.SUCCESS) {
                try {
                    JSONObject resultJson = AbstractActionNewService.getDataResult(dagInstNode.outJsonWithCache());
                    step.setDescription(resultJson.getString("description"));
                    switch (resultJson.getString("status")) {
                        case "OK":
                            step.setStatus("success");
                            break;
                        case "ERROR":
                            step.setStatus("error");
                            break;
                        case "CRITICAL":
                            step.setStatus("critical");
                            break;
                        case "WARNING":
                            step.setStatus("warning");
                            break;
                        default:
                            step.setStatus("exception");
                    }
                    Object taskAppId = JSONObject.parseObject(dagInstNode.dagNode().detail().toJson())
                            .getOrDefault("appId", dagInst.getAppId());
                    CommonKvComponent commonKvComponent = new CommonKvComponent().isSummary(false);

                    if (StringUtils.isNotEmpty(resultJson.getString("message"))) {
                        commonKvComponent.addKv("诊断结果", resultJson.get("message"));
                    }

                    if (StringUtils.isNotEmpty(resultJson.getString("action"))) {
                        commonKvComponent.addKv("诊断建议", resultJson.get("action"));
                    }

                    if (!dagInstNode.dagNode().type().equals(DagNodeType.LOCAL)) {
                        if (dagInstNode.status() != DagInstNodeStatus.SUCCESS) {
                            commonKvComponent.addLinkKv("作业",
                                    ap.teslaTaskInstance((String) taskAppId, dagInstNode.getTaskId()), "查看");
                        }
                    }
                    if (StringUtils.isNotEmpty(resultJson.getString("name"))) {
                        step.setName(resultJson.getString("name"));
                    }
                    detail.add(commonKvComponent);

                } catch (Exception e) {
                    detail.add(BaseWidgetJsxComponent.builder()
                            .text("get output error: " + Throwables.getStackTraceAsString(e))
                            .build());
                }
            }
            detail.addAll(detailFromNode);
            steps.add(step);
        }
        return commonDiagnoseComponent;
    }

    public List<ComponentInterface> getDagInstNode(TcDagInstNode dagInstNode, String locale) {
        if (dagInstNode.status() == DagInstNodeStatus.INIT) {
            return Collections.singletonList(BaseWidgetJsxComponent.builder()
                    .text("wait for run")
                    .build());
        }
        try {
            switch (dagInstNode.type()) {
                case DAG:
                    switch (dagInstNode.status()) {
                        case SKIP_CAUSE_BY_EXCEPTION:
                        case SKIP_CAUSE_BY_STOPPED:
                        case SKIP:
                        case STOPPED:
                            return Collections.singletonList(BaseWidgetJsxComponent.builder()
                                    .text(dagInstNode.status().name())
                                    .build());
                        default:
                            break;
                    }
                    TcDagInst dagInst = dagInstNode.subDagInst();
                    return Collections.singletonList(getDagInst(dagInst.getId(), locale));
                case NODE:
                    switch (dagInstNode.status()) {
                        case INIT:
                        case RUNNING:
                        case STOPPING:
                        case WAIT_STOP:
                            BaseWidgetJsxComponent bwjc = BaseWidgetJsxComponent.builder().build();
                            bwjc.addText("running, please wait");
                            if (StringUtils.isNotEmpty(dagInstNode.getTaskId())
                                    && !dagInstNode.dagNode().type().equals(DagNodeType.LOCAL)) {

                                String a = format("<a href=\"%s\" target=\"_blank\">详情</a>",
                                        ap.teslaTaskInstance(dagInstNode.dagInst().getAppId(), dagInstNode.getTaskId()));
                                bwjc.addHtmlText(a, "\n");

                            }
                            return Collections.singletonList(bwjc);
                        case EXCEPTION:
                            bwjc = BaseWidgetJsxComponent.builder().build();
                            bwjc.addText("exception: " + dagInstNode.getStatusDetail());
                            if (StringUtils.isNotEmpty(dagInstNode.getTaskId())
                                    && !dagInstNode.dagNode().type().equals(DagNodeType.LOCAL)) {

                                String a = format("<a href=\"%s\" target=\"_blank\">详情</a>",
                                        ap.teslaTaskInstance(dagInstNode.dagInst().getAppId(), dagInstNode.getTaskId()));
                                bwjc.addHtmlText(a, "\n");

                            }
                            return Collections.singletonList(bwjc);
                        case SKIP_CAUSE_BY_EXCEPTION:
                        case SKIP_CAUSE_BY_STOPPED:
                        case SKIP:
                        case STOPPED:
                            return Collections.singletonList(BaseWidgetJsxComponent.builder()
                                    .text(dagInstNode.status().name())
                                    .build());
                        default:
                            break;
                    }
                    DagNodeFormatType dagNodeFormatType = dagInstNode.dagNode().formatType();
                    switch (dagNodeFormatType) {
                        case METRIC:
                        case KV:
                        case TABLE:
                        case SIMPLE_CUSTOM:
                            List<ComponentInterface> retList = new ArrayList<>();
                            JSONArray formatDetailArray = new JSONArray();
                            try {
                                formatDetailArray = JSONObject.parseArray(dagInstNode.dagNode().getFormatDetail());
                            } catch (Exception ignored) {
                            }
                            JSONArray dataArray = AbstractActionNewService.getDataData(dagInstNode.outJsonWithCache(),
                                    JSONArray.class);
                            dataArray = dataArray == null ? new JSONArray() : dataArray;
                            for (int index = 0; index < dataArray.size(); index++) {
                                JSONObject formatDetailJson =
                                        !CollectionUtils.isEmpty(formatDetailArray) && formatDetailArray.size() > index ?
                                                formatDetailArray.getJSONObject(index) : new JSONObject();
                                Object dataObject = dataArray.get(index);
                                retList.add(getComponentInterface(dagInstNode.getId(), dagNodeFormatType, dataObject,
                                        formatDetailJson));
                            }
                            return retList;
                        case CUSTOM:
                            String elementOutUrl = format(
                                    ap.teslaCheckUrl + "/dagInstNode/elementOut?dagInstNodeId=%s",
                                    dagInstNode.getId()
                            );
                            String formatDetail = dagInstNode.dagNode().getFormatDetail();
                            formatDetail = StringUtils.isEmpty(formatDetail) ?
                                    "" : formatDetail.replace("${elementOutUrl}", elementOutUrl);
                            return Collections.singletonList(BaseWidgetComponent.builder()
                                    .formatDetail(formatDetail)
                                    .build());
                        default:
                            return Collections.singletonList(BaseWidgetJsxComponent.builder()
                                    .text("dagInstNode can not support formatType: " + dagNodeFormatType)
                                    .build());
                    }
                default:
                    return Collections.singletonList(BaseWidgetJsxComponent.builder()
                            .text("dagInstNode can not support type: " + dagInstNode.type())
                            .build());
            }
        } catch (Exception e) {
            log.error("", e);
            return Collections.singletonList(BaseWidgetJsxComponent.builder()
                    .text("dagInstNode show error: " + Throwables.getStackTraceAsString(e))
                    .build());
        }
    }

    public ComponentInterface getComponentInterface(Long dagInstNodeId, DagNodeFormatType nodeFormatType,
                                                    Object dataObject,
                                                    JSONObject formatDetailJson) {

        switch (nodeFormatType) {
            case TABLE:
                JSONArray dataArray = TypeUtils.castToJavaBean(dataObject, JSONArray.class);
                return CommonTableComponent.builder()
                        .rowData(dataArray.toJavaList(JSONObject.class))
                        .build()
                        .setColumnDefs(dataArray, formatDetailJson);
            case KV:
                LinkedHashMap<String, Object> dataLinkedHashMap = castToLinkedHashMap(dataObject);
                return CommonKvComponent.builder()
                        .kvData(dataLinkedHashMap)
                        .build().isSummary(false);
            case METRIC:
                dataLinkedHashMap = castToLinkedHashMap(dataObject);
                return CommonHighchartComponent.builder()
                        .data(dataLinkedHashMap)
                        .build();
            case SIMPLE_CUSTOM:
                if (CollectionUtils.isEmpty(formatDetailJson)) {
                    return BaseWidgetJsxComponent.builder().text("do not define format detail for this part").build();
                } else {
                    DagNodeFormatType type;
                    JSONObject detail = formatDetailJson.getJSONObject("detail");
                    try {
                        type = formatDetailJson.getObject("type", DagNodeFormatType.class);
                    } catch (Exception e) {
                        return BaseWidgetJsxComponent.builder()
                                .build()
                                .addText("get type error")
                                .addErrorToText(e);
                    }
                    return getComponentInterface(dagInstNodeId, type, dataObject, detail);
                }
            case CUSTOM:
                String elementOutUrl = format(ap.teslaCheckUrl + "/dagInstNode/elementOut?dagInstNodeId=%s",
                        dagInstNodeId);
                String formatDetail = JSONObject.toJSONString(formatDetailJson);
                formatDetail = StringUtils.isEmpty(formatDetail) ?
                        "" : formatDetail.replace("${elementOutUrl}", elementOutUrl);
                return BaseWidgetComponent.builder()
                        .formatDetail(formatDetail)
                        .build();
            default:
                break;
        }
        return BaseWidgetJsxComponent.builder()
                .build()
                .addText("do not support type: " + nodeFormatType.name());
    }

    @SuppressWarnings("unchecked")
    private LinkedHashMap<String, Object> castToLinkedHashMap(Object o) {
        return TypeUtils.castToJavaBean(o, LinkedHashMap.class);
    }

}
