package com.alibaba.tesla.dag.api;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.JsonUtil;
import com.alibaba.tesla.dag.model.domain.ParamType;
import com.alibaba.tesla.dag.model.domain.TcDag;
import com.alibaba.tesla.dag.model.domain.TcDagNode;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeInputParam;
import com.alibaba.tesla.dag.model.domain.dagnode.ParamFromType;
import com.alibaba.tesla.dag.model.repository.TcDagNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagRepository;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author jinghua.yjh
 */
@Service
public class DagApiServiceImpl implements DagApiService {

    @Autowired
    TcDagNodeRepository dagNodeRepository;

    @Autowired
    TcDagRepository dagRepository;

    private Map<String, Object> toNode(DagCreateNode node) {
        DagInstNodeType type = node.getType() == null ? DagInstNodeType.NODE : node.getType();
        List<DagNodeInputParam> inputParams = node.params == null ? new ArrayList<>() : node.params.entrySet()
            .stream()
            .map(entry -> DagNodeInputParam.builder()
                .name(entry.getKey())
                .alias(entry.getKey())
                .value(entry.getValue())
                .type(ParamType.STRING)
                .isOverWrite(true)
                .fromType(ParamFromType.CONSTANT)
                .from("")
                .build())
            .collect(Collectors.toList());

        switch (type) {
            case DAG:
                String appId = StringUtils.isEmpty(node.appId) ? "tesla" : node.appId;
                TcDag dag = dagRepository.findFirstByAppIdAndName(appId, node.name);
                Builder<String, Object> dataBuilder = new Builder<String, Object>()
                    .put("defId", dag.getId())
                    .put("id", node.nodeId)
                    .put("label", node.name)
                    .put("type", DagInstNodeType.DAG.name())
                    .put("inputParams", inputParams)
                    .put("outputParams", new JSONArray())
                    .put("tooltip", new JSONObject());
                return ImmutableMap.of(
                    "shape", "rect",
                    "id", node.nodeId,
                    "label", node.name,
                    "data", dataBuilder.build()
                );
            case NODE:
                appId = StringUtils.isEmpty(node.appId) ? "tesla" : node.appId;
                TcDagNode dagNode = dagNodeRepository.findFirstByAppIdAndName(appId, node.name);
                dataBuilder = new Builder<String, Object>()
                    .put("defId", dagNode.getId())
                    .put("id", node.nodeId)
                    .put("label", node.name)
                    .put("type", DagInstNodeType.NODE.name())
                    .put("inputParams", inputParams)
                    .put("outputParams", new JSONArray())
                    .put("tooltip", new JSONObject());
                if (node.maxRetryTimes != null) {
                    dataBuilder.put("maxRetryTimes", node.maxRetryTimes);
                }
                if (node.retryExpression != null) {
                    dataBuilder.put("retryExpression", node.retryExpression);
                }
                if (node.runTimeout != null) {
                    dataBuilder.put("runTimeout", node.runTimeout);
                }
                return ImmutableMap.of(
                    "shape", "rect",
                    "id", node.nodeId,
                    "label", node.name,
                    "data", dataBuilder.build()
                );
            default:
                return null;
        }
    }

    @Override
    public Long create(String name, DagCreateNode preNode, DagCreateNode postNode,
        List<DagCreateNode> nodeList, List<DagCreateEdge> edgeList) throws Exception {
        Map<String, Object> contentJson = JsonUtil.map(
            "nodes", nodeList.stream().map(this::toNode).collect(Collectors.toList()),
            "edges", edgeList.stream().map(x -> ImmutableMap.of(
                "shape", "QuadraticEdge",
                "source", x.sourceNodeId,
                "target", x.targetNodeId,
                "data", ImmutableMap.of(
                    "expression", x.expression
                ),
                "style", ImmutableMap.of(
                    "endArrow", true,
                    "stroke", "#fdad",
                    "lineWidth", 2
                )
            )).collect(Collectors.toList())
        );
        if (preNode != null) {
            contentJson.put("__pre_node__", toNode(preNode));
        }
        if (postNode != null) {
            contentJson.put("__post_node__", toNode(postNode));
        }

        TcDag dag = TcDag.builder()
            .appId("tesla")
            .name(name)
            .alias(name)
            .content(JSONObject.toJSONString(contentJson))
            .inputParams("[]")
            .hasFeedback(1)
            .hasHistory(1)
            .description(name)
            .entity("")
            .notice("")
            .creator("")
            .modifier("")
            .lastUpdateBy("API")
            .exScheduleTaskId("")
            .defaultShowHistory(1)
            .build();

        dag.upsertByAppIdAndName();

        return dag.getId();
    }

    @Override
    public Long create(String name, List<DagCreateNode> nodeList, List<DagCreateEdge> edgeList) throws Exception {
        return create(name, null, null, nodeList, edgeList);
    }

}
