package com.alibaba.tesla.dag.common;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;

/**
 * @author yangjinghua
 */
@Slf4j
public class Requests {

    private static OkHttpClient httpClient;

    private static OkHttpClient getHttpClient() {
        if (httpClient == null) {
            httpClient = new OkHttpClient.Builder()
                .connectionPool(new ConnectionPool())
                .build();
        }
        return httpClient;
    }

    private static Request.Builder createRequestBuilder(String url, JSONObject params, JSONObject headers) {
        if (CollectionUtils.isEmpty(params)) {
            params = new JSONObject();
        }
        if (CollectionUtils.isEmpty(headers)) {
            headers = new JSONObject();
        }
        HttpUrl.Builder queryUrl = Objects.requireNonNull(HttpUrl.parse(url)).newBuilder();
        for (String key : params.keySet()) {
            queryUrl.addQueryParameter(key, params.getString(key));
        }
        Request.Builder requestBuilder = new Request.Builder().url(queryUrl.build());
        for (String key : headers.keySet()) {
            requestBuilder.addHeader(key, headers.getString(key));
        }
        return requestBuilder;
    }

    public static <T> T transfer(String resp, Class<T> tClass) throws Exception {
        if (tClass == JSONObject.class || tClass == JSONArray.class) {
            return (T)JSONObject.parse(resp);
        } else if (tClass == String.class) {
            return (T)resp;
        } else {
            throw new Exception(String.format("resp: %s 不支持class: %s", resp, tClass.getName()));
        }
    }

    public static String get(String url, JSONObject params, JSONObject headers) throws IOException {
        Request.Builder requestBuilder = createRequestBuilder(url, params, headers);
        Request request = requestBuilder.get().build();
        Response response = getHttpClient().newCall(request).execute();
        assert response.body() != null;
        return response.body().string();
    }

    public static <T> T get(String url, JSONObject params, JSONObject headers, Class<T> tClass) throws Exception {
        String resp = get(url, params, headers);
        return transfer(resp, tClass);
    }

    public static <T> T get(String url, Class<T> tClass) throws Exception {
        String resp = get(url, null, null);
        return transfer(resp, tClass);
    }

    public static String post(String url, JSONObject params, String postJson, JSONObject headers)
        throws IOException {
        Request.Builder requestBuilder = createRequestBuilder(url, params, headers);
        Request request = requestBuilder.post(RequestBody.create(MediaType.parse("application/json"),
            postJson)).build();
        Response response = getHttpClient().newCall(request).execute();
        assert response.body() != null;
        return response.body().string();
    }

    public static <T> T post(String url, JSONObject params, String postJson, JSONObject headers, Class<T> tClass)
        throws Exception {
        String resp = post(url, params, postJson, headers);
        return transfer(resp, tClass);
    }

    public static <T> T post(String url, String postJson, Class<T> tClass) throws Exception {
        String resp = post(url, new JSONObject(), postJson, new JSONObject());
        return transfer(resp, tClass);
    }

    public static String put(String url, JSONObject params, String postJson, JSONObject headers)
        throws IOException {
        Request.Builder requestBuilder = createRequestBuilder(url, params, headers);
        Request request = requestBuilder.put(RequestBody.create(MediaType.parse("application/json"), postJson)).build();
        Response response = getHttpClient().newCall(request).execute();
        assert response.body() != null;
        return response.body().string();
    }

    public static <T> T put(String url, JSONObject params, String postJson, JSONObject headers, Class<T> tClass)
        throws Exception {
        String resp = put(url, params, postJson, headers);
        return transfer(resp, tClass);
    }

    public static String delete(String url, JSONObject params, JSONObject headers) throws IOException {
        Request.Builder requestBuilder = createRequestBuilder(url, params, headers);
        Request request = requestBuilder.delete().build();
        Response response = getHttpClient().newCall(request).execute();
        assert response.body() != null;
        return response.body().string();
    }

    public static <T> T delete(String url, JSONObject params, JSONObject headers, Class<T> tClass) throws Exception {
        String resp = delete(url, params, headers);
        return transfer(resp, tClass);
    }

}

