package com.alibaba.tesla.dag.dispatch;

import com.alibaba.tesla.dag.algorithm.DAG;
import com.alibaba.tesla.dag.constant.DagConstant;
import com.alibaba.tesla.dag.notify.*;
import com.alibaba.tesla.dag.repository.dao.DagInstEdgeDAO;
import com.alibaba.tesla.dag.repository.dao.DagInstNodeDAO;
import com.alibaba.tesla.dag.repository.domain.DagInstDO;
import com.alibaba.tesla.dag.repository.domain.DagInstEdgeDO;
import com.alibaba.tesla.dag.repository.domain.DagInstNodeDO;
import com.alibaba.tesla.dag.schedule.status.DagInstEdgeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstStatus;
import com.alibaba.tesla.dag.services.DagInstNewService;
import com.alibaba.tesla.dag.services.DagInstNodeNewService;
import com.alibaba.tesla.dag.util.DagUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author QianMo
 * @date 2021/04/22.
 */
@Service
@Slf4j
public class RunningDagInstDispatcher implements IDagInstDispatcher {

    @Autowired
    private IDagInstNotify dagInstNotify;

    @Autowired
    private IDagInstNodeTaskNotify dagInstNodeTaskNotify;

    @Autowired
    private DagInstNodeDAO dagInstNodeDAO;

    @Autowired
    private DagInstEdgeDAO dagInstEdgeDAO;

    @Autowired
    private DagInstNewService dagInstNewService;

    @Autowired
    private DagInstNodeNewService dagInstNodeNewService;

    private Boolean conditionIsAllSuccess(List<DagInstNodeDO> sourceNodeList, String target,
                                          Map<String, DagInstEdgeDO> key2EdgeMap) {
        for (DagInstNodeDO sourceNode : sourceNodeList) {
            if (sourceNode.fetchNodeStatus() == DagInstNodeStatus.SUCCESS) {
                String key = sourceNode.getNodeId() + "-" + target;
                DagInstEdgeDO edge = key2EdgeMap.get(key);
                if (Objects.nonNull(edge) && edge.fetchEdgeStatus() == DagInstEdgeStatus.INIT) {
                    dagInstNodeNewService.calcEdge(sourceNode);
                    return null;
                } else if (Objects.isNull(edge) || edge.fetchEdgeStatus() != DagInstEdgeStatus.SUCCESS || Objects.equals(
                        edge.getIsPass(), 0)) {
                    return false;
                }
            }

        }

        return true;
    }

    @Override
    public DagInstStatus registerType() {
        return DagInstStatus.RUNNING;
    }

    @Override
    public void dispatch(DagInstDO dagInstDO) {
        Long dagInstId = dagInstDO.getId();
        log.info(">>>[Running]|dispatch|dagInstId={}", dagInstId);

        List<DagInstNodeDO> nodeList = dagInstNodeDAO.getSimpleList(dagInstId);
        List<DagInstEdgeDO> edgeList = dagInstEdgeDAO.getSimpleList(dagInstId);

        //nodeId -> object
        Map<String, DagInstNodeDO> id2NodeMap = nodeList.stream().collect(
                Collectors.toMap(DagInstNodeDO::getNodeId, o -> o));

        //source_target -> object
        Map<String, DagInstEdgeDO> key2EdgeMap = edgeList.stream().collect(
                Collectors.toMap(DagInstEdgeDO::fetchKey, o -> o));

        //根据点和边构建DAG结构
        DAG dag = DagUtil.calcDAG(nodeList, edgeList);

        //出度递归，错误和停止向后传递
        for (DagInstNodeDO node : nodeList) {
            doChildrenOnExceptionOrStopped(node, dag, id2NodeMap);
        }

        //入度递归
        for (DagInstNodeDO node : nodeList) {
            DagInstNodeStatus nodeStatus = node.fetchNodeStatus();

            if (Objects.equals(nodeStatus, DagInstNodeStatus.INIT)) {
                Set parent = dag.getParent(node.getNodeId());

                //如果parent为空，表面无前置条件，直接运行
                if (CollectionUtils.isEmpty(parent)) {
                    log.info(">>>runningDagInstDispatcher|dispatch|start|Header Node={}", node.toSimpleString());
                    dispatch(node);
                    continue;
                }

                //获取parent列表
                List<DagInstNodeDO> parentNodeList = nodeList.stream()
                        .filter(dagInstNodeDO -> parent.contains(dagInstNodeDO.getNodeId())).collect(Collectors.toList());

                List<DagInstNodeStatus> parentStatusList = parentNodeList.stream().map(
                        dagInstNodeDO -> dagInstNodeDO.fetchNodeStatus())
                        .collect(Collectors.toList());

                //如果前置状态都为SKIP，则当前节点也SKIP
                if (isOnly(parentStatusList, Arrays.asList(DagInstNodeStatus.SKIP))) {
                    log.info(">>>runningDagInstDispatcher|dispatch|skip|node={}", node.toSimpleString());
                    // 所有入度都是skip， 节点置为skip
                    node.setStatus(DagInstNodeStatus.SKIP.toString());
                    dagInstNodeDAO.updateStatus(node.getId(), DagInstNodeStatus.SKIP);

                }
                //如果前置状态只有SKIP和SUCCESS，则判断边条件
                else if (isOnly(parentStatusList, Arrays.asList(DagInstNodeStatus.SKIP, DagInstNodeStatus.SUCCESS))) {
                    log.info(">>>runningDagInstDispatcher|dispatch|success|node={}", node.toSimpleString());
                    Boolean condition = conditionIsAllSuccess(parentNodeList, node.getNodeId(), key2EdgeMap);
                    if (condition == Boolean.TRUE) {
                        dispatch(node);
                    } else if (condition == Boolean.FALSE) {
                        log.info(">>>runningDagInstDispatcher|dispatch|edge condition is false|node={}", node.toSimpleString());
                        node.setStatus(DagInstNodeStatus.SKIP.toString());
                        dagInstNodeDAO.updateStatus(node.getId(), DagInstNodeStatus.SKIP);
                    }
                } else {
                    log.info(">>>runningDagInstDispatcher|dispatch|ignore|node={}, parentStatusList={}", node.toSimpleString(), parentStatusList);
                }
            }
        }

        //获取所有节点状态
        List<DagInstNodeStatus> allNodeStatusList = nodeList.stream().map(DagInstNodeDO::fetchNodeStatus).collect(
                Collectors.toList());
        if (allNodeStatusList.contains(DagInstNodeStatus.INIT) || allNodeStatusList.contains(
                DagInstNodeStatus.RUNNING)) {
            return;
        }

        if (allNodeStatusList.contains(DagInstNodeStatus.EXCEPTION) || allNodeStatusList.contains(
                DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION)) {
            dagInstNewService.freshInstStatus(dagInstDO, registerType(), DagInstStatus.EXCEPTION);
            return;
        }

        DagInstNodeDO postNode = dagInstNodeDAO.getDagInstNode(dagInstId, DagConstant.POST_NODE_ID);
        if (Objects.isNull(postNode)) {
            dagInstNewService.freshInstStatus(dagInstDO, registerType(), DagInstStatus.SUCCESS);
        } else {
            dagInstNewService.freshInstStatus(dagInstDO, registerType(), DagInstStatus.POST_RUNNING);
            DagInstDispatch dagInstDispatch = DagInstDispatch.builder().dagInstId(dagInstId).nodeId(postNode.getNodeId()).dagInstStatus(
                    DagInstStatus.POST_RUNNING).build();
            dagInstNotify.sendDagInstDispatch(dagInstDispatch);
        }
    }

    private void doChildrenOnExceptionOrStopped(DagInstNodeDO node, DAG dag, Map<String, DagInstNodeDO> id2NodeMap) {
        DagInstNodeStatus nodeStatus = node.fetchNodeStatus();

        //非异常和非停止
        if (!nodeStatus.isException() && !nodeStatus.isStopped()) {
            return;
        }

        Set children = dag.getChildren(node.getNodeId());
        if (CollectionUtils.isEmpty(children)) {
            return;
        }

        for (Object child : children) {
            DagInstNodeDO childNode = id2NodeMap.get(child);
            if (Objects.isNull(childNode)) {
                return;
            }

            DagInstNodeStatus childNodeStatus = childNode.fetchNodeStatus();
            if (nodeStatus.isException()) {
                if (!childNodeStatus.isException()) {
                    log.info(">>>runningDagInstDispatcher|doChildrenOnException|{}->{}", childNode.getId(),
                            DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION);
                    dagInstNodeDAO.updateStatus(childNode.getId(), DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION);
                }
                childNode.setStatus(DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION.toString());
            } else if (nodeStatus.isStopped()) {
                if (!childNodeStatus.isStopped()) {
                    log.info(">>>runningDagInstDispatcher|doChildrenOnStopped|{}->{}", childNode.getId(),
                            DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION);
                    dagInstNodeDAO.updateStatus(childNode.getId(), DagInstNodeStatus.SKIP_CAUSE_BY_STOPPED);
                }
                childNode.setStatus(DagInstNodeStatus.SKIP_CAUSE_BY_STOPPED.toString());
            }

            doChildrenOnExceptionOrStopped(childNode, dag, id2NodeMap);
        }
    }

    private boolean isOnly(List<DagInstNodeStatus> list1, List<DagInstNodeStatus> list2) {
        return list1.stream().filter(dagInstNodeStatus -> list2.contains(dagInstNodeStatus)).count() == list1.size();
    }

    private void dispatch(DagInstNodeDO node) {
        DagInstNodeTask dagInstNodeTask = DagInstNodeTask.builder().dagInstId(
                node.getDagInstId()).nodeId(node.getNodeId()).nodeTaskType(NodeTaskType.START).build();
        dagInstNodeTaskNotify.sendDagInstNodeTask(dagInstNodeTask);
    }
}
