package com.alibaba.tesla.dag.local;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.alibaba.tesla.dag.ApplicationProperties;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory.Builder;
import org.reflections.Reflections;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Service
public class ClassService {

    @Autowired
    ApplicationProperties ap;

    @Value("${dag.load.local.jar.interval.second:2}")
    public Long dagLoadLocalJarIntervalSecond;

    @Value("${reflections.log.show:true}")
    public boolean reflectionsLogShow;

    public Map<String, Class> nodeMap;

    public Map<String, Class> dagMap;

    private ScheduledThreadPoolExecutor executor;

    @PreDestroy
    public void preDestroy() throws InterruptedException {
        executor.shutdown();
        executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
    }

    @PostConstruct
    public void postConstruct() {
        executor = new ScheduledThreadPoolExecutor(
            2, new Builder().namingPattern("%d").daemon(true).build());
        executor.scheduleAtFixedRate(this::run, 0, dagLoadLocalJarIntervalSecond, TimeUnit.SECONDS);
    }

    public void run() {
        try {
            refresh();
        } catch (Exception e) {
            log.error("", e);
        }
    }

    public void refresh() throws Exception {
        Map<String, Class> nodeMap = new HashMap<>(0);
        Map<String, Class> dagMap = new HashMap<>(0);

        for (Class aClass : listAllClass()) {
            String name = (String)aClass.getField("name").get(aClass);
            name = StringUtils.isEmpty(name) ? aClass.getSimpleName() : name;
            if (isDagClass(aClass)) {
                if (dagMap.containsKey(name)) {
                    throw new Exception("dag name must be unique");
                }
                dagMap.put(name, aClass);
            }
            if (isNodeClass(aClass)) {
                if (nodeMap.containsKey(name)) {
                    throw new Exception("node name must be unique");
                }
                nodeMap.put(name, aClass);
            }
        }
        this.nodeMap = nodeMap;
        this.dagMap = dagMap;
    }

    // 获取所有class
    public List<Class> listAllClass() {
        List<Class> allClassList = new ArrayList<>();
        allClassList.addAll(listLocalClass());
        allClassList.addAll(listJarClass());
        return allClassList.stream().filter(this::isDagOrNodeClass).collect(Collectors.toList());
    }

    // 本地class获取
    public Set<Class<? extends AbstractLocalBase>> listLocalClass() {
        if (!reflectionsLogShow) {
            Reflections.log = null;
        }
        Reflections reflections = new Reflections(ap.teslaDagLocalnodeReflectionPrefix);
        return reflections.getSubTypesOf(AbstractLocalBase.class);
    }

    // jar 文件class获取
    public List<Class<?>> listJarClass() {
        return listClass(listFile(ap.teslaDagLocalnodeJarPath));
    }

    public List<File> listFile(String parentDir) {
        File parentDirFile = new File(parentDir);
        return listFile(parentDirFile);
    }

    public List<File> listFile(File parentDirFile) {
        List<File> fileList = new ArrayList<>();

        if (!parentDirFile.exists() || parentDirFile.isFile()) {
            return fileList;
        }
        File[] subFileList = parentDirFile.listFiles();
        if (subFileList == null) {
            return fileList;
        }
        for (File subFile : subFileList) {
            if (subFile.isFile()) {
                fileList.add(subFile);
            } else {
                fileList.addAll(listFile(subFile));
            }
        }
        return fileList;
    }

    public List<Class<?>> listClass(File file) throws IOException {
        URLClassLoader classLoader = new URLClassLoader(
            new URL[] {file.toURI().toURL()},
            Thread.currentThread().getContextClassLoader()
        );
        JarFile jarFile = new JarFile(file);
        return jarFile.stream()
            .filter(jarEntry -> !jarEntry.isDirectory())
            .filter(jarEntry -> jarEntry.getName().endsWith(".class"))
            .map(jarEntry -> {
                String className = jarEntry.getName().replace(".class", "").replace("/", ".");
                try {
                    return classLoader.loadClass(className);
                } catch (ClassNotFoundException e) {
                    log.error("", e);
                    return null;
                }
            })
            .filter(Objects::nonNull)
            .collect(Collectors.toList());
    }

    public List<Class<?>> listClass(List<File> fileList) {
        return fileList.stream()
            .map(x -> {
                try {
                    return listClass(x);
                } catch (IOException e) {
                    log.error("", e);
                    return null;
                }
            })
            .filter(Objects::nonNull)
            .flatMap(List::stream)
            .collect(Collectors.toList());
    }

    public boolean isDagOrNodeClass(Class<?> aClass) {
        try {
            return aClass.getSuperclass().getSuperclass().getName().equals(AbstractLocalBase.class.getName());
        } catch (Exception e) {
            return false;
        }
    }

    public boolean isDagClass(Class<?> aClass) {
        try {
            return aClass.getSuperclass().getName().equals(AbstractLocalDagBase.class.getName());
        } catch (Exception e) {
            return false;
        }
    }

    public boolean isNodeClass(Class<?> aClass) {
        try {
            return aClass.getSuperclass().getName().equals(AbstractLocalNodeBase.class.getName());
        } catch (Exception e) {
            return false;
        }
    }

}
