package com.alibaba.tesla.dag.local;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.ApplicationProperties;
import com.alibaba.tesla.dag.common.Tools;
import com.alibaba.tesla.dag.model.domain.*;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeRunRet;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeDetailLocal;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeFormatType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeType;
import com.alibaba.tesla.dag.model.repository.TcDagInstNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagInstNodeStdRepository;
import com.alibaba.tesla.dag.model.repository.TcDagNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagRepository;
import com.alibaba.tesla.dag.model.repository.TcDagServiceNodeRepository;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.task.TaskStatus;

import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory.Builder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.DependsOn;
import org.springframework.stereotype.Service;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Service
@DependsOn(value = "beanUtil")
public class LocalService {

    @Autowired
    ClassService classService;

    @Autowired
    TcDagNodeRepository dagNodeRepository;

    @Autowired
    TcDagRepository dagRepository;

    @Autowired
    TcDagInstNodeStdRepository dagInstNodeStdRepository;

    @Autowired
    TcDagInstNodeRepository dagInstNodeRepository;

    @Autowired
    ApplicationProperties ap;

    @Autowired
    TcDagServiceNodeRepository dagServiceNodeRepository;

    public static String LAST_UPDATE_BY = "LOCAL";

    @PostConstruct
    public void postConstruct() {
        executor = Tools.createThreadPool(ap.teslaDagLocalnodeRunMaxSize, "localNodeRun");

        ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(
            2, new Builder().namingPattern("%d").daemon(true).build());
        executor.scheduleAtFixedRate(this::run, 0, 10, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void preDestroy() throws InterruptedException {
        executor.shutdown();
        executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
    }

    public void run() {
        try {
            TcDagServiceNode tcDagServiceNode = dagServiceNodeRepository.findFirstByIp(Tools.localIp);
            if(Objects.isNull(tcDagServiceNode)){
                return;
            }
            boolean enable = tcDagServiceNode.enable();
            if (!enable) {
                return;
            }
            if (classService.dagMap == null || classService.nodeMap == null) {
                return;
            }
            registerNodeList();
            registerDagList();
        } catch (Exception e) {
            log.error("", e);
        }
    }

    public void cleanInvalidDag(JSONArray appIdNameArray) {
        List<TcDag> dagList = dagRepository.findAllByLastUpdateBy(LAST_UPDATE_BY);
        for (TcDag dag : dagList) {
            String key = JSONObject.toJSONString(Arrays.asList(dag.getAppId(), dag.getName()));
            if (!appIdNameArray.contains(key)) {
                log.info("delete dag: " + dag.getName());
                dagRepository.delete(dag);
            }
        }
    }

    public void cleanInvalidNode(JSONArray appIdNameArray) {
        List<TcDagNode> dagNodeList = dagNodeRepository.findAllByLastUpdateBy(LAST_UPDATE_BY);
        for (TcDagNode dagNode : dagNodeList) {
            String key = JSONObject.toJSONString(Arrays.asList(dagNode.getAppId(), dagNode.getName()));
            if (!appIdNameArray.contains(key)) {
                log.info("delete node: " + dagNode.getName());
                dagNodeRepository.delete(dagNode);
            }
        }
    }

    public List<String> registerNode(String name, Class nodeClass) throws Exception {
        String appId = (String)nodeClass.getField("appId").get(nodeClass);
        String alias = (String)nodeClass.getField("alias").get(nodeClass);
        String description = (String)nodeClass.getField("description").get(nodeClass);
        Integer isShare = (Integer)nodeClass.getField("isShare").get(nodeClass);
        Integer isShow = (Integer)nodeClass.getField("isShow").get(nodeClass);
        Object inputParams = nodeClass.getField("inputParams").get(nodeClass);
        Object outputParams = nodeClass.getField("outputParams").get(nodeClass);
        DagNodeType nodeType = (DagNodeType)nodeClass.getField("nodeType").get(nodeClass);
        DagNodeFormatType formatType = (DagNodeFormatType)nodeClass.getField("formatType").get(nodeClass);
        String formatDetail = (String)nodeClass.getField("formatDetail").get(nodeClass);
        String creator = (String)nodeClass.getField("creator").get(nodeClass);
        Integer runTimeout = (Integer)nodeClass.getField("runTimeout").get(nodeClass);
        Long maxRetryTimes = (Long)nodeClass.getField("maxRetryTimes").get(nodeClass);
        String retryExpression = (String)nodeClass.getField("retryExpression").get(nodeClass);

        TcDagNode dagNode = TcDagNode.builder()
            .gmtCreate(System.currentTimeMillis() / 1000)
            .gmtModified(System.currentTimeMillis() / 1000)
            .appId(appId)
            .name(name)
            .alias(StringUtils.isEmpty(alias) ? name : alias)
            .description(description)
            .isShare(isShare)
            .isShow(isShow)
            .inputParams(JSONObject.toJSONString(inputParams))
            .outputParams(JSONObject.toJSONString(outputParams))
            .type(nodeType.name())
            .detail(DagNodeDetailLocal.builder().appId(appId).name(name).build().toJson())
            .formatType(formatType.name())
            .formatDetail(formatDetail)
            .creator(creator)
            .modifier(creator)
            .runTimeout(runTimeout)
            .maxRetryTimes(maxRetryTimes)
            .retryExpression(retryExpression)
            .lastUpdateBy(LAST_UPDATE_BY)
            .build();
        dagNode.upsertByAppIdAndName();
        dagNode.updateOptions();
        return Arrays.asList(appId, name);
    }

    public void registerNodeList() {

        JSONArray appIdNameArray = new JSONArray();

        for (Entry<String, Class> entry : classService.nodeMap.entrySet()) {
            String name = entry.getKey();
            Class nodeClass = entry.getValue();
            List<String> appIdAndName;
            try {
                appIdAndName = registerNode(name, nodeClass);
            } catch (Exception e) {
                log.error(format("registerNode ERROR; name: %s, nodeClass: %s", name, nodeClass.getName()), e);
                continue;
            }
            appIdNameArray.add(JSONObject.toJSONString(appIdAndName));
        }
        cleanInvalidNode(appIdNameArray);
    }

    public List<String> registerDag(String name, Class dagClass) throws Exception {
        String appId = (String)dagClass.getField("appId").get(dagClass);
        String alias = (String)dagClass.getField("alias").get(dagClass);
        Object inputParams = dagClass.getField("inputParams").get(dagClass);
        Integer hasFeedback = (Integer)dagClass.getField("hasFeedback").get(dagClass);
        Integer hasHistory = (Integer)dagClass.getField("hasHistory").get(dagClass);
        String creator = (String)dagClass.getField("creator").get(dagClass);

        AbstractLocalDagBase localDag = (AbstractLocalDagBase)(dagClass.getConstructor().newInstance());
        localDag.draw();
        TcDag dag = TcDag.builder()
            .gmtCreate(System.currentTimeMillis() / 1000)
            .gmtModified(System.currentTimeMillis() / 1000)
            .appId(appId)
            .name(name)
            .alias(StringUtils.isEmpty(alias) ? name : alias)
            .content(JSONObject.toJSONString(ImmutableMap.of(
                "nodes", localDag.nodeMap.values(),
                "edges", localDag.edgeMap.values()
            )))
            .inputParams(JSONObject.toJSONString(inputParams))
            .hasFeedback(hasFeedback)
            .hasHistory(hasHistory)
            .creator(creator)
            .modifier(creator)
            .lastUpdateBy(LAST_UPDATE_BY)
            .build();
        dag.upsertByAppIdAndName();
        dag.updateOptions();
        return Arrays.asList(appId, name);
    }

    public JSONArray registerDagList(Set<Entry<String, Class>> dagClassEntrySet, JSONArray appIdNameArray) {
        Set<Entry<String, Class>> failedDagClassEntrySet = new HashSet<>();
        List<String> logList = new ArrayList<>();
        for (Entry<String, Class> entry : dagClassEntrySet) {
            String name = entry.getKey();
            Class dagClass = entry.getValue();
            List<String> appIdAndName;
            try {
                appIdAndName = registerDag(name, dagClass);
            } catch (Exception e) {
                logList.add(format("registerDag ERROR; name: %s, dagClass: %s, detail: %s",
                    name, dagClass.getName(), e.getMessage()));
                failedDagClassEntrySet.add(entry);
                continue;
            }
            appIdNameArray.add(JSONObject.toJSONString(appIdAndName));
        }
        if (failedDagClassEntrySet.size() != dagClassEntrySet.size()) {
            registerDagList(failedDagClassEntrySet, appIdNameArray);
        } else {
            logList.forEach(log::warn);
        }
        return appIdNameArray;
    }

    public void registerDagList() {
        cleanInvalidDag(registerDagList(classService.dagMap.entrySet(), new JSONArray()));

    }

    private ThreadPoolExecutor executor = null;

}