package com.alibaba.tesla.dag.local;

import java.io.File;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.UUID;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.alibaba.tesla.dag.ApplicationProperties;

import com.aliyun.oss.OSS;

import java.nio.file.Files;

import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.GetObjectRequest;
import io.minio.MinioClient;
import io.minio.messages.Item;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory.Builder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Data
@Service
public class RemoteJarService {

    @Autowired
    private ApplicationProperties ap;

    private OSS ossClient;

    private MinioClient minioClient;

    private File rootDirFile;

    private Map<String, Date> jarFileLastDownloadDate = new Hashtable<>();

    private List<String> sourceList = Arrays.asList("oss", "minio");

    private void init() throws Exception {
        File rootFile = new File(ap.teslaDagLocalnodeJarPath);
        rootFile.mkdirs();
        if (!rootFile.exists()) {
            throw new Exception(ap.teslaDagLocalnodeJarPath + " is not exists");
        }
        for (String source : sourceList) {
            File sourceFile = new File(ap.teslaDagLocalnodeJarPath + "/" + source);
            sourceFile.mkdirs();
            if (!rootFile.exists()) {
                throw new Exception(ap.teslaDagLocalnodeJarPath + "/" + source + " is not exists");
            }
        }
    }

    private OSS ossClient() {
        if (ossClient == null) {
            if (StringUtils.isNotEmpty(ap.dagHotLoadJarOssEndpoint)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarOssAccessKeyId)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarOssAccessKeySecret)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarOssBucketName)) {
                ossClient = new OSSClientBuilder().build(
                    ap.dagHotLoadJarOssEndpoint,
                    ap.dagHotLoadJarOssAccessKeyId,
                    ap.dagHotLoadJarOssAccessKeySecret
                );
            }
        }
        return ossClient;
    }

    private MinioClient minioClient() throws Exception {
        if (minioClient == null) {
            if (StringUtils.isNotEmpty(ap.dagHotLoadJarMinioEndpoint)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarMinioAccessKeyId)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarMinioAccessKeySecret)
                && StringUtils.isNotEmpty(ap.dagHotLoadJarMinioBucketName)) {
                minioClient = new MinioClient(
                    ap.dagHotLoadJarMinioEndpoint,
                    ap.dagHotLoadJarMinioAccessKeyId,
                    ap.dagHotLoadJarMinioAccessKeySecret
                );
            }
        }
        return minioClient;
    }

    private List<FileDetail> getOssFileList() {
        return ossClient() == null ? new ArrayList<>() :
            ossClient().listObjects(ap.dagHotLoadJarOssBucketName).getObjectSummaries()
                .stream()
                .map(x ->
                    FileDetail.builder()
                        .name(x.getKey())
                        .lastModified(x.getLastModified())
                        .source("oss")
                        .build())
                .collect(Collectors.toList());
    }

    private List<FileDetail> getMinioFileList() throws Exception {
        if (minioClient() == null) {
            return new ArrayList<>();
        }
        List<FileDetail> fileDetailList = new ArrayList<>();
        try {
            minioClient().listObjects(ap.dagHotLoadJarMinioBucketName)
                .forEach(x -> {
                        try {
                            Item item = x.get();
                            fileDetailList.add(
                                FileDetail.builder()
                                    .name(item.objectName())
                                    .lastModified(item.lastModified())
                                    .source("minio")
                                    .build()
                            );
                        } catch (Exception e) {
                            log.warn(e.getLocalizedMessage());
                        }
                    }
                );
        } catch (Exception e) {
            log.warn(e.getLocalizedMessage());
        }
        return fileDetailList;
    }

    private List<FileDetail> getFileList() throws Exception {
        return Stream.of(getOssFileList(), getMinioFileList())
            .flatMap(List::stream)
            .collect(Collectors.toList());
    }

    private List<FileDetail> listShouldDownloadFileName() throws Exception {
        return getFileList().stream()
            .filter(x ->
                !jarFileLastDownloadDate.containsKey(x.source + "/" + x.name)
                    || jarFileLastDownloadDate.get(x.source + "/" + x.name).before(x.lastModified)
                    || !new File(format("%s/%s/%s", ap.teslaDagLocalnodeJarPath, x.source, x.name)).exists()
            )
            .collect(Collectors.toList());
    }

    private void download() throws Exception {
        for (FileDetail x : listShouldDownloadFileName()) {
            log.info(format("start download file: %s from %s ", x.name, x.source));
            String filePath = ap.teslaDagLocalnodeJarPath + "/" + x.source + "/" + x.name;
            String tmpFilePath = "/tmp/" + UUID.randomUUID();
            switch (x.source) {
                case "oss":
                    if (ossClient() != null) {
                        ossClient().getObject(
                            new GetObjectRequest(ap.dagHotLoadJarOssBucketName, x.name),
                            new File(tmpFilePath)
                        );
                    }
                    break;
                case "minio":
                    if (minioClient() != null) {
                        minioClient().getObject(
                            ap.dagHotLoadJarMinioBucketName,
                            x.name,
                            tmpFilePath
                        );
                    }
                    break;
                default:
                    break;
            }
            Files.move(Paths.get(tmpFilePath), Paths.get(filePath), StandardCopyOption.REPLACE_EXISTING);
            jarFileLastDownloadDate.put(x.source + "/" + x.name, new Date());
            log.info(format("end download file: %s from %s ", x.name, x.source));
        }

    }

    private void cleanDeleted() throws Exception {
        List<String> fileSourceNameList = getFileList().stream()
            .map(x -> format("%s/%s", x.source, x.name))
            .collect(Collectors.toList());

        for (String source : Arrays.asList("oss", "minio")) {
            for (File file : Objects.requireNonNull(new File(ap.teslaDagLocalnodeJarPath + "/" + source).listFiles())) {
                if (!fileSourceNameList.contains(source + "/" + file.getName())) {
                    boolean isDelete = file.delete();
                    log.info(format("delete local file: %s with result: %s", file.getPath(), isDelete));
                }
            }
        }

    }

    private void run() {
        try {
            init();
            download();
            cleanDeleted();
        } catch (Exception e) {
            log.error("", e);
        }
    }

    ScheduledThreadPoolExecutor executor;

    @PostConstruct
    public void postConstruct() {
        log.info("start schedule rsync local hot jar from remote");
        executor = new ScheduledThreadPoolExecutor(2, new Builder().namingPattern("%d").daemon(true).build());
        executor.scheduleAtFixedRate(this::run, 0, ap.dagHotLoadJarIntervalSecond, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void preDestroy() throws InterruptedException {
        executor.shutdown();
        executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
    }

}

@lombok.Builder
@AllArgsConstructor
@NoArgsConstructor
class FileDetail {

    String name;

    Date lastModified;

    String source;

}