package com.alibaba.tesla.dag.model.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.persistence.*;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.BeanUtil;
import com.alibaba.tesla.dag.model.domain.dag.DagInputParam;
import com.alibaba.tesla.dag.model.repository.TcDagNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagOptionsRepository;
import com.alibaba.tesla.dag.model.repository.TcDagRepository;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;
import org.springframework.util.CollectionUtils;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@EqualsAndHashCode(callSuper = true)
@Slf4j
@Table(uniqueConstraints = {@UniqueConstraint(columnNames = {"appId", "name"})})
@Entity
@EntityListeners(AuditingEntityListener.class)
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
public class TcDag extends AbstractTcDag {

    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;

    @Column
    private Long gmtCreate;

    @Column
    private Long gmtModified;

    @Column
    private String appId;

    @Column
    private String name;

    @Column
    private String alias;

    @Column(columnDefinition = "longtext")
    private String content;

    @Column(columnDefinition = "longtext")
    private String inputParams;

    @Column(columnDefinition = "longtext")
    private Integer hasFeedback;

    @Column(columnDefinition = "longtext")
    private Integer hasHistory;

    @Column(columnDefinition = "longtext")
    private String description;

    @Column(columnDefinition = "longtext")
    private String entity;

    @Column
    private String notice;

    @Column
    private String creator;

    @Column
    private String modifier;

    @Column
    private String lastUpdateBy;

    @Column
    private String exScheduleTaskId;

    @Column
    private Integer defaultShowHistory;

    public boolean defaultShowHistory() {
        return defaultShowHistory != null && 1 == defaultShowHistory;
    }

    public String alias() {
        return StringUtils.isEmpty(alias) ? name : alias;
    }

    public JSONObject contentJson() {
        JSONObject contentJson = JSONObject.parseObject(content);
        if (CollectionUtils.isEmpty(contentJson)) {
            contentJson = new JSONObject();
            contentJson.put("nodes", new JSONArray());
            contentJson.put("edges", new JSONArray());
        }
        return contentJson;
    }

    public JSONObject preNode() {
        return contentJson().getJSONObject("__pre_node__");
    }

    public JSONObject postNode() {
        return contentJson().getJSONObject("__post_node__");
    }

    public List<JSONObject> nodes() {
        JSONArray nodeArray = contentJson().getJSONArray("nodes");
        if (nodeArray == null) {
            return new ArrayList<>();
        }
        return nodeArray.toJavaList(JSONObject.class);
    }

    public void checkNodes() throws Exception {
        for (JSONObject node : nodes()) {
            JSONObject dataJson = node.getJSONObject("data");
            if (CollectionUtils.isEmpty(dataJson)) {
                throw new Exception("data can not be null");
            }
            long defId = dataJson.getLongValue("defId");
            if (defId <= 0) {
                throw new Exception("defId can not be negative");
            }
            String type = dataJson.getString("type");
            switch (type) {
                case "NODE":
                    TcDagNode dagNode = BeanUtil.getBean(TcDagNodeRepository.class).findFirstById(defId);
                    if (dagNode == null) {
                        throw new Exception("can not find defid with dagNode: " + defId);
                    }
                    break;
                case "DAG":
                    TcDag dag = BeanUtil.getBean(TcDagRepository.class).findFirstById(defId);
                    if (dag == null) {
                        throw new Exception("can not find defid with dag: " + defId);
                    }
                    break;
                default:
                    throw new Exception("no such type: " + type);
            }

        }
    }

    public List<JSONObject> edges() {
        JSONArray nodeArray = contentJson().getJSONArray("edges");
        if (nodeArray == null) {
            return new ArrayList<>();
        }
        return nodeArray.toJavaList(JSONObject.class);
    }

    public void setInputParams(String inputParams) {
        this.inputParams = inputParams;
    }

    public void setInputParams(List<DagInputParam> inputParams) throws Exception {
        inputParams = inputParams == null ? new ArrayList<>() : inputParams;
        for (DagInputParam inputParam : inputParams) {
            inputParam.check();
        }
        this.inputParams = JSONObject.toJSONString(inputParams, true);
    }

    public List<DagInputParam> inputParams() {
        if (StringUtils.isEmpty(inputParams)) {
            return new ArrayList<>();
        }
        JSONArray jsonArray = JSONObject.parseArray(inputParams);
        if (CollectionUtils.isEmpty(jsonArray)) {
            return new ArrayList<>();
        }
        return jsonArray.toJavaList(DagInputParam.class);
    }

    public void addInputParam(DagInputParam inputParam) throws Exception {
        List<DagInputParam> inputParams = inputParams();
        for (DagInputParam param : inputParams) {
            if (param.getName().equals(inputParam.getName())) {
                throw new Exception("name has already exists");
            }
        }
        inputParams.add(inputParam);
        this.inputParams = JSONObject.toJSONString(inputParams);
        saveAndFlush();
    }

    public void deleteInputParam(String name) throws Exception {
        List<DagInputParam> inputParams = inputParams();
        List<DagInputParam> newInputParams = new ArrayList<>();
        for (DagInputParam param : inputParams) {
            if (!param.getName().equals(name)) {
                newInputParams.add(param);
            }
        }
        this.inputParams = JSONObject.toJSONString(newInputParams);
        saveAndFlush();
    }

    public void updateInputParam(DagInputParam inputParam) throws Exception {
        List<DagInputParam> inputParams = inputParams();
        for (int index = 0; index < inputParams.size(); index++) {
            if (inputParams.get(index).getName().equals(inputParam.getName())) {
                inputParams.set(index, inputParam);
            }
        }
        this.inputParams = JSONObject.toJSONString(inputParams);
        saveAndFlush();
    }

    public void saveAndFlush() throws Exception {
        if (StringUtils.isEmpty(name)) {
            throw new Exception("name can not be null");
        }
        if (StringUtils.isEmpty(appId)) {
            throw new Exception("appId can not be null");
        }
        BeanUtil.getBean(TcDagRepository.class).saveAndFlush(this);
    }

    public void updateOptions() {
        updateOptions("zh_CN");
    }

    public void updateOptions(String locale) {
        TcDagOptions.builder()
            .locale(locale).name(format("dag:%s:alias", id)).value(alias)
            .build().upsert();
        TcDagOptions.builder()
            .locale(locale).name(format("dag:%s:description", id)).value(description)
            .build().upsert();
    }

    public void obtainOptions(String locale) {
        TcDagOptionsRepository dagOptionsRepository = BeanUtil.getBean(TcDagOptionsRepository.class);
        TcDagOptions dagOptions = dagOptionsRepository.findFirstByLocaleAndName(locale, format("dag:%s:alias", id));
        if (dagOptions != null) {
            alias = dagOptions.getValue();
        }
        dagOptions = dagOptionsRepository.findFirstByLocaleAndName(locale, format("dag:%s:description", id));
        if (dagOptions != null) {
            description = dagOptions.getValue();
        }
    }

    public List<String> getFatherNodeIdListByNodeId(String nodeId) {
        List<String> nodeIdList = edges()
            .stream()
            .filter(x -> x.getString("target").equals(nodeId))
            .map(x -> x.getString("source"))
            .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(nodeIdList)) {
            nodeIdList = new ArrayList<>();
        }
        List<String> copyList = new ArrayList<>(nodeIdList);
        for (String fatherNodeId : copyList) {
            for (String fNodeId : getFatherNodeIdListByNodeId(fatherNodeId)) {
                if (!nodeIdList.contains(fNodeId)) {
                    nodeIdList.add(fNodeId);
                }
            }
        }
        return nodeIdList;
    }

    public JSONObject getNodeByNodeId(String nodeId) throws Exception {
        for (JSONObject node : nodes()) {
            if (node.getString("id").equals(nodeId)) {
                return node;
            }
        }
        throw new Exception("can not find by nodeId: " + nodeId);
    }

    public void upsertByAppIdAndName() throws Exception {
        if (id == null) {
            TcDagRepository dagRepository = BeanUtil.getBean(TcDagRepository.class);
            TcDag searchDag = dagRepository.findFirstByAppIdAndName(appId, name);
            if (searchDag != null) {
                gmtCreate = searchDag.getGmtCreate();
                id = searchDag.getId();
            }
        }
        if (id == null) {
            setGmtCreate(System.currentTimeMillis() / 1000);
            setGmtModified(System.currentTimeMillis() / 1000);
        } else {
            setGmtModified(System.currentTimeMillis() / 1000);
        }
        saveAndFlush();
    }

    @Override
    public void save() {
        BeanUtil.getBean(TcDagRepository.class).save(this);
    }

    @Override
    public void flush() {
        BeanUtil.getBean(TcDagRepository.class).flush();
    }
}