package com.alibaba.tesla.dag.model.domain;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.BeanUtil;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeInputParam;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeOutputParam;
import com.alibaba.tesla.dag.model.repository.*;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.services.LocalActionNewService;
import com.google.common.base.Throwables;
import lombok.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;
import org.springframework.util.CollectionUtils;

import javax.persistence.*;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jinghua.yjh
 */
@EqualsAndHashCode(callSuper = true)
@Table
@Entity
@EntityListeners(AuditingEntityListener.class)
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
@Slf4j
public class TcDagInstNode extends AbstractTcDag {

    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;

    @Column
    private Long gmtCreate;

    @Column
    private Long gmtModified;

    @Column
    private Long gmtStart;

    @Column
    private Long dagInstId;

    @Column
    private String nodeId;

    @Column
    private String status;

    @Column(columnDefinition = "longtext")
    private String statusDetail;

    @Column
    private String taskId;

    @Column
    private String stopTaskId;

    @Column
    private String lockId;

    @Column
    private Long subDagInstId;

    @Column
    private Long retryTimes;

    /**
     * 根据defId拿到的快照，可能是NODE也可能是DAG
     */
    @Column(columnDefinition = "longtext")
    private String tcDagOrNodeDetail;

    /**
     * 图里面node相关信息的快照
     */
    @Column(columnDefinition = "longtext")
    private String tcDagContentNodeSpec;

    /**
     * 用于缓存
     */
    @Transient
    private JSONObject tcDagContentNodeSpecJson;

    @Transient
    private TcDagNode dagNode;
    @Transient
    private JSONObject outJsonWithCache;

    public long retryTimes() {
        return retryTimes == null ? 0 : retryTimes;
    }

    /**
     * 将dag或者node的快照更新进来，只有创建的时候才会用到
     */
    public TcDagInstNode updateTcDagOrNodeDetail(JSONObject nodeJson) {
        switch (type()) {
            case NODE:
                TcDagNode dagNode = BeanUtil.getBean(TcDagNodeRepository.class).findFirstById(defId());
                String formatType = nodeJson.getJSONObject("data").getString("format_type");
                String formatDetail = nodeJson.getJSONObject("data").getString("format_detail");
                Long maxRetryTimes = nodeJson.getJSONObject("data").getLong("maxRetryTimes");
                String retryExpression = nodeJson.getJSONObject("data").getString("retryExpression");
                Integer runTimeout = nodeJson.getJSONObject("data").getInteger("runTimeout");
                if (StringUtils.isNotEmpty(formatType)) {
                    dagNode.setFormatType(formatType);
                }
                if (StringUtils.isNotEmpty(formatDetail)) {
                    dagNode.setFormatDetail(formatDetail);
                }
                if (maxRetryTimes != null) {
                    dagNode.setMaxRetryTimes(maxRetryTimes);
                }
                if (retryExpression != null) {
                    dagNode.setRetryExpression(retryExpression);
                }
                if (runTimeout != null) {
                    dagNode.setRunTimeout(runTimeout);
                }
                tcDagOrNodeDetail = JSONObject.toJSONString(dagNode);
                break;
            case DAG:
                tcDagOrNodeDetail = JSONObject.toJSONString(
                        BeanUtil.getBean(TcDagRepository.class).findFirstById(defId()));
                break;
            default:
                break;
        }
        return this;
    }

    public TcDagInstNode updateNodeId() {
        nodeId();
        return this;
    }

    public String alias() {
        if (subDagInstId == null || subDagInstId == 0) {
            return dagNode().alias();
        } else {
            return subDagInst().dag().alias();
        }
    }

    /**
     * dag图中的nodeId
     */
    public JSONObject tcDagContentNodeSpecJson() {
        if (CollectionUtils.isEmpty(tcDagContentNodeSpecJson)) {
            tcDagContentNodeSpecJson = JSONObject.parseObject(tcDagContentNodeSpec);
        }
        return tcDagContentNodeSpecJson;
    }

    /**
     * dag图中的nodeId
     */
    public String nodeId() {
        if (StringUtils.isEmpty(nodeId)) {
            nodeId = tcDagContentNodeSpecJson().getString("id");
        }
        return nodeId;
    }

    /**
     * dag图中的data
     */
    public JSONObject dataJson() {
        return tcDagContentNodeSpecJson().getJSONObject("data");
    }

    /**
     * dag图中的type
     */
    public DagInstNodeType type() {
        return DagInstNodeType.valueOf(dataJson().getString("type"));
    }

    /**
     * dag图中的defId
     */
    public Long defId() {
        return dataJson().getLongValue("defId");
    }

    /**
     * dag图中的输入参数
     */
    public List<DagNodeInputParam> inputParamList() {
        List<DagNodeInputParam> dagNodeInputParamsFromDag = DagNodeInputParam.toList(
                dataJson().getJSONArray("inputParams")
        );
        List<DagNodeInputParam> dagNodeInputParamsFromNode = dagNode().inputParamList();

        for (DagNodeInputParam dagNodeInputParamFromDag : dagNodeInputParamsFromDag) {
            for (DagNodeInputParam dagNodeInputParamFromNode : dagNodeInputParamsFromNode) {
                if (dagNodeInputParamFromDag.getName().equals(dagNodeInputParamFromNode.getName())) {
                    dagNodeInputParamFromDag.setType(dagNodeInputParamFromNode.getType());
                }
            }
        }

        return dagNodeInputParamsFromDag;
    }

    /**
     * dag图中的输出参数
     */
    public List<DagNodeOutputParam> outputParamList() {
        return DagNodeOutputParam.toList(dataJson().getJSONArray("outputParams"));
    }

    public DagInstNodeStatus status() {
        return DagInstNodeStatus.valueOf(status);
    }

    public boolean isEnd() {
        switch (status()) {
            case EXCEPTION:
            case STOPPED:
            case SUCCESS:
            case SKIP:
            case SKIP_CAUSE_BY_STOPPED:
            case SKIP_CAUSE_BY_EXCEPTION:
                return true;
            default:
                return false;
        }
    }

    public TcDagInst dagInst() {
        return BeanUtil.getBean(TcDagInstRepository.class).findFirstById(dagInstId);
    }

    public TcDagInst subDagInst() {
        return BeanUtil.getBean(TcDagInstRepository.class).findFirstById(subDagInstId);
    }

    public Long taskId() {
        if (StringUtils.isEmpty(taskId)) {
            return null;
        }
        return Long.parseLong(taskId);
    }

    public TcDagInstNodeStd localNodeStd() {
        return BeanUtil.getBean(TcDagInstNodeStdRepository.class).findFirstById(id);
    }

    public Long stopTaskId() {
        if (StringUtils.isEmpty(stopTaskId)) {
            return null;
        }
        return Long.parseLong(stopTaskId);
    }

    public TcDagNode dagNode() {
        if (dagNode == null) {
            dagNode = JSONObject.parseObject(tcDagOrNodeDetail, TcDagNode.class);
        }
        return dagNode;
    }

    public List<TcDagInstEdge> inEdgeList() {
        TcDagInstEdgeRepository dagInstEdgeRepository = BeanUtil.getBean(TcDagInstEdgeRepository.class);
        List<TcDagInstEdge> edgeList = dagInstEdgeRepository.findAllByDagInstIdAndTarget(dagInstId, nodeId());
        return CollectionUtils.isEmpty(edgeList) ? new ArrayList<>() : edgeList;
    }

    public List<TcDagInstNode> sourceNodeList() {
        return inEdgeList()
                .stream()
                .map(TcDagInstEdge::sourceNode)
                //.filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    public JSONObject outJsonWithCache() throws Exception {
        if (CollectionUtils.isEmpty(outJsonWithCache)) {
            outJsonWithCache = outJson();
        }
        return outJsonWithCache;
    }

    public JSONObject outJson() throws Exception {
        JSONObject outJson;
        TcDagNode dagNode = dagNode();
        switch (dagNode.type()) {
            case LOCAL:
                outJson = BeanUtil.getBean(LocalActionNewService.class).stdout(taskId());
                break;
            default:
                throw new Exception("dagNode node have this type: " + dagNode.type());
        }
        return outJson;
    }

    public void saveAndFlush() {
        BeanUtil.getBean(TcDagInstNodeRepository.class).saveAndFlush(this);
    }

    @Override
    public void save() {
        BeanUtil.getBean(TcDagInstNodeRepository.class).save(this);
    }

    @Override
    public void flush() {
        BeanUtil.getBean(TcDagInstNodeRepository.class).flush();
    }

    public void statusSet(Exception e) {
        this.status = DagInstNodeStatus.EXCEPTION.toString();
        this.statusDetail = Throwables.getStackTraceAsString(e);
    }

    public void statusSet(DagInstNodeStatus status) {
        this.status = status.toString();
    }

    public void statusSet(DagInstNodeStatus status, String statusDetail) {
        this.status = status.toString();
        this.statusDetail = statusDetail;
    }
}
