package com.alibaba.tesla.dag.model.domain;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.BeanUtil;
import com.alibaba.tesla.dag.model.domain.dagnode.*;
import com.alibaba.tesla.dag.model.repository.TcDagNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagOptionsRepository;
import lombok.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import javax.persistence.*;
import java.util.List;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@EqualsAndHashCode(callSuper = true)
@Table(uniqueConstraints = {@UniqueConstraint(columnNames = {"appId", "name"})})
@Entity
@EntityListeners(AuditingEntityListener.class)
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
public class TcDagNode extends AbstractTcDag {

    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;

    @Column
    private Long gmtCreate;

    @Column
    private Long gmtModified;

    @Column
    private String appId;

    @Column
    private String name;

    @Column
    private String alias;

    @Column(columnDefinition = "longtext")
    private String description;

    @Column
    private Integer isShare;

    @Column
    private Integer isShow;

    @Column(columnDefinition = "longtext")
    private String inputParams;

    @Column(columnDefinition = "longtext")
    private String outputParams;

    @Column
    private String type;

    @Column(columnDefinition = "longtext")
    private String detail;

    @Column
    private String formatType;

    @Column(columnDefinition = "longtext")
    private String formatDetail;

    private String lastUpdateBy;

    private String creator;

    private String modifier;

    private Integer isSupportChatops;

    @Column(columnDefinition = "longtext")
    private String chatopsDetail;

    private Integer runTimeout;

    @Column
    private Long maxRetryTimes;

    @Column(columnDefinition = "longtext")
    private String retryExpression;

    public long maxRetryTimes() {
        return maxRetryTimes == null ? 0 : maxRetryTimes;
    }

    public String alias() {
        return StringUtils.isEmpty(alias) ? name : alias;
    }

    public boolean share() {
        return isShare != null && 1 == isShare;
    }

    public boolean show() {
        return isShow != null && 1 == isShow;
    }

    public List<DagNodeInputParam> inputParamList() {
        return DagNodeInputParam.toList(JSONArray.parseArray(inputParams));
    }

    public List<DagNodeOutputParam> outputParamList() {
        return DagNodeOutputParam.toList(JSONArray.parseArray(outputParams));
    }

    public DagNodeType type() {
        return DagNodeType.valueOf(type);
    }

    public DagNodeDetailInterface detail() {
        switch (type()) {
            case LOCAL:
                return JSONObject.parseObject(detail, DagNodeDetailLocal.class);
            case TASK:
                return JSONObject.parseObject(detail, DagNodeDetailTask.class);
            case API:
                return JSONObject.parseObject(detail, DagNodeDetailApi.class);
            case FAAS:
                return JSONObject.parseObject(detail, DagNodeDetailFaas.class);
            default:
                return null;
        }
    }

    public DagNodeFormatType formatType() {
        return StringUtils.isEmpty(formatType) ? DagNodeFormatType.CUSTOM : DagNodeFormatType.valueOf(formatType);
    }

    public void updateOptions() {
        updateOptions("zh_CN");
    }

    public void updateOptions(String locale) {
        TcDagOptions.builder()
            .locale(locale).name(format("dagNode:%s:alias", id)).value(alias)
            .build().upsert();
        TcDagOptions.builder()
            .locale(locale).name(format("dagNode:%s:description", id)).value(description)
            .build().upsert();
    }

    public void obtainOptions(String locale) {
        TcDagOptionsRepository dagOptionsRepository = BeanUtil.getBean(TcDagOptionsRepository.class);
        TcDagOptions dagOptions = dagOptionsRepository.findFirstByLocaleAndName(locale, format("dagNode:%s:alias", id));
        if (dagOptions != null) {
            alias = dagOptions.getValue();
        }
        dagOptions = dagOptionsRepository.findFirstByLocaleAndName(locale, format("dagNode:%s:description", id));
        if (dagOptions != null) {
            description = dagOptions.getValue();
        }
    }

    public void upsertByAppIdAndName() throws Exception {
        if (id == null) {
            TcDagNodeRepository dagNodeRepository = BeanUtil.getBean(TcDagNodeRepository.class);
            TcDagNode searchDagNode = dagNodeRepository.findFirstByAppIdAndName(appId, name);
            if (searchDagNode != null) {
                gmtCreate = searchDagNode.getGmtCreate();
                id = searchDagNode.getId();
            }
        }
        if (id == null) {
            setGmtCreate(System.currentTimeMillis() / 1000);
            setGmtModified(System.currentTimeMillis() / 1000);
        } else {
            setGmtModified(System.currentTimeMillis() / 1000);
        }
        saveAndFlush();
    }

    public void saveAndFlush() throws Exception {
        if (StringUtils.isEmpty(name)) {
            throw new Exception("name can not be null");
        }
        if (StringUtils.isEmpty(appId)) {
            throw new Exception("appId can not be null");
        }
        BeanUtil.getBean(TcDagNodeRepository.class).saveAndFlush(this);
    }

    @Override
    public void save() {
        BeanUtil.getBean(TcDagNodeRepository.class).save(this);
    }

    @Override
    public void flush() {
        BeanUtil.getBean(TcDagNodeRepository.class).flush();
    }

}
