package com.alibaba.tesla.dag.services;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.ApplicationProperties;
import com.alibaba.tesla.dag.algorithm.DAG;
import com.alibaba.tesla.dag.common.Tools;
import com.alibaba.tesla.dag.constant.DagConstant;
import com.alibaba.tesla.dag.dispatch.IDagInstDispatcher;
import com.alibaba.tesla.dag.model.domain.dag.DagInputParam;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeType;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeInputParam;
import com.alibaba.tesla.dag.notify.*;
import com.alibaba.tesla.dag.repository.dao.*;
import com.alibaba.tesla.dag.repository.domain.*;
import com.alibaba.tesla.dag.schedule.event.DagInstStatusEvent;
import com.alibaba.tesla.dag.schedule.status.DagInstEdgeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstStatus;
import com.alibaba.tesla.dag.util.DagUtil;
import com.alibaba.tesla.dag.util.DateUtil;
import com.alibaba.tesla.dag.util.IPUtil;
import com.alibaba.tesla.dag.util.MonitorUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author QianMo
 * @date 2021/04/22.
 */
@Slf4j
@Service
public class DagInstNewService {

    private static List<DagInstStatus> RUNNING_LIST = Arrays.asList(DagInstStatus.PRE_RUNNING,
            DagInstStatus.RUNNING, DagInstStatus.POST_RUNNING);

    @Autowired
    private DagInstDAO dagInstDAO;

    @Autowired
    private DagInstNodeDAO dagInstNodeDAO;

    @Autowired
    private DagDAO dagDAO;

    @Autowired
    private IDagInstNotify dagInstNotify;

    @Autowired
    private List<IDagInstDispatcher> handlerList;

    @Autowired
    private IDagInstNodeTaskNotify dagInstNodeTaskNotify;

    @Autowired
    private DagBatchDAO dagBatchDAO;

    @Autowired
    private DagNodeDAO dagNodeDAO;

    @Autowired
    private ApplicationProperties applicationProperties;

    @Autowired
    private ApplicationContext applicationContext;

    private static String getTag(String entityValue) {
        return StringUtils.substring(entityValue, 0, 128);
    }

    private static boolean isDelete(String global) {
        return StringUtils.equals(DagConstant.FAAS_DELETE_OPERATION, global);
    }

    private static List<DagInstNodeDO> topologySortByDag(List<DagInstNodeDO> dagInstNodeDOList, DAG dag) {
        List<DagInstNodeDO> result = new ArrayList<>(dagInstNodeDOList.size());

        List topologySortList = dag.topologySort();
        log.info(">>>dagInstNewService|topologySortByDag|topology={}", JSON.toJSONString(topologySortList));

        topologySortList.stream().forEach(sort ->
                result.add(findByNodeId(dagInstNodeDOList, sort.toString()))
        );

        return result;
    }

    private static DagInstNodeDO findByNodeId(List<DagInstNodeDO> dagInstNodeDOList, String nodeId) {
        return dagInstNodeDOList.stream().filter(dagInstNodeDO -> StringUtils.equals(dagInstNodeDO.getNodeId(),
                nodeId)).findFirst().orElse(null);
    }

    public Long submit(DagDO dagDO, String operator, String channel, String env, JSONObject commonParams, boolean isStandalone)
            throws Exception {
        log.info(">>>dagInstNewService|submit|enter|dagName={}, channel={}, env={}, commonParams={}, isStandalone={}", dagDO.getName(),
                channel, env, commonParams, isStandalone);

        boolean entityValueIsTargetKey = commonParams.getBooleanValue("entityValueIsTargetKey");

        JSONArray selectItemArray = commonParams.getJSONArray("selectItems");
        JSONObject itemConfigs = commonParams.getJSONObject("itemConfigs");

        JSONObject globalVariable = commonParams.getJSONObject("configs");
        if (Objects.isNull(globalVariable)) {
            globalVariable = new JSONObject();
        }
        String entityValue = commonParams.getString("entityValue");
        if (StringUtils.isNotEmpty(entityValue)) {
            List<String> hostList = IPUtil.toIpList(entityValue);
            if (CollectionUtils.isEmpty(hostList) || entityValueIsTargetKey) {
                hostList = Arrays.asList(StringUtils.split(entityValue, ","));
            }

            if (CollectionUtils.isNotEmpty(hostList)) {
                globalVariable.put(AbstractActionNewService.PARAMS_SPEC_TARGET_KEY, hostList);
                globalVariable.put(AbstractActionNewService.PARAMS_SPEC_TARGET_ARGS_KEY, hostList);
            }
        }

        dagDO.updateContent(selectItemArray, itemConfigs);

        DagInstDO dagInst = DagInstDO.builder()
                .appId(dagDO.getAppId())
                .dagId(dagDO.getId())
                .lockId(IPUtil.getLockId())
                .tcDagDetail(JSONObject.toJSONString(dagDO))
                .status(DagInstStatus.INIT.toString())
                .globalVariable(JSONObject.toJSONString(globalVariable))
                .globalParams(JSONObject.toJSONString(new JSONObject()))
                .channel(channel)
                .env(env)
                .tag(getTag(entityValue))
                .creator(operator)
                .isSub(false)
                .standaloneIp(isStandalone || applicationProperties.isJarUseMode() ? Tools.localIp : "")
                .build();

        dagInstDAO.insert(dagInst);
        dagInstNotify.sendDagInstDispatch(DagInstDispatch.builder().dagInstId(dagInst.getId())
                .dagInstStatus(DagInstStatus.INIT).build());

        return dagInst.getId();

    }

    /**
     * @param dagInstNodeDO   DAG类型的节点
     * @param parentDagInstDO 父流程实例
     * @return
     * @throws Exception
     */
    public Long submitSub(DagInstNodeDO dagInstNodeDO, DagInstDO parentDagInstDO)
            throws Exception {
        Long dagId = dagInstNodeDO.fetchDefId();
        log.info(">>>[SubmitSub]|dagInstId={}, nodeId={}, dagId={}", dagInstNodeDO.getDagInstId(), dagInstNodeDO.getNodeId(), dagId);

        DagDO dagDO = dagDAO.getDagById(dagId);
        if (Objects.isNull(dagDO)) {
            log.warn(">>>dagInstNewService|submitSub|dag does not exist|dagId={}", dagId);
            throw new Exception("dag does not exist!dagId=" + dagId);
        }

        JSONObject globalVariableJson = parentDagInstDO.fetchGlobalVariableJson();
        JSONObject globalParamsJson = parentDagInstDO.fetchGlobalParamsJson();

        DagInstDO subDagInst = DagInstDO.builder()
                .appId(dagDO.getAppId())
                .dagId(dagDO.getId())
                .lockId(IPUtil.getLockId())
                .status(DagInstStatus.INIT.toString())
                .globalVariable(JSONObject.toJSONString(globalVariableJson))
                .globalParams(JSONObject.toJSONString(globalParamsJson))
                .globalObject(parentDagInstDO.getGlobalObject())
                .channel(parentDagInstDO.getChannel())
                .env(parentDagInstDO.getEnv())
                .creator(parentDagInstDO.getCreator())
                .standaloneIp(parentDagInstDO.getStandaloneIp())
                .fatherDagInstNodeId(dagInstNodeDO.getId())
                .isSub(true)
                .build();

        subDagInst.setRelationNode(dagInstNodeDO);

        List<DagNodeInputParam> dagNodeInputParamList = dagInstNodeDO.fetchInputParamList();
        List<DagInputParam> subDagInputParamList = dagDO.fetchInputParamList();

        subDagInputParamList.addAll(
                dagNodeInputParamList.stream().map(DagNodeInputParam::toDagInputParam).collect(Collectors.toList())
        );
        dagDO.setInputParamList(subDagInputParamList);
        subDagInst.setTcDagDetail(JSONObject.toJSONString(dagDO));

        dagInstDAO.insert(subDagInst);
        dagInstNotify.sendDagInstDispatch(DagInstDispatch.builder().dagInstId(subDagInst.getId())
                .dagInstStatus(DagInstStatus.INIT).build());

        return subDagInst.getId();

    }

    public void doDagInstDispatch(DagInstDispatch dagInstDispatch) {
        long beginTime = System.currentTimeMillis();
        IDagInstDispatcher iDagInstDispatcher = handlerList.stream().filter(
                handler -> handler.registerType() == dagInstDispatch.getDagInstStatus())
                .findFirst().orElse(null);
        if (Objects.nonNull(iDagInstDispatcher)) {
            DagInstDO dagInstDO = dagInstDAO.getDagInstById(dagInstDispatch.getDagInstId(),
                    dagInstDispatch.getDagInstStatus());
            if (Objects.isNull(dagInstDO)) {
                log.info(">>>dagInstNewService|doDagInstDispatch|dagInstDO not exist|dagInstDispatch={}", dagInstDispatch);
                return;
            }

            DagDO dagDO = dagInstDO.fetchDagDO();
            if (Objects.isNull(dagDO)) {
                log.info(">>>dagInstNewService|doDagInstDispatch|dagDO not exist");
                return;
            }

            iDagInstDispatcher.dispatch(dagInstDO);
        }

        MonitorUtil.instMonitor.addCost(System.currentTimeMillis() - beginTime);
    }

    public void inspectDagInstNode(DagInstDO dagInstDO) {
        DagInstStatus nodeStatus = dagInstDO.fetchStatus();
        if (RUNNING_LIST.contains(nodeStatus)) {
            List<DagInstNodeDO> dagInstNodeList = dagInstNodeDAO.listByRunning(dagInstDO.getId());
            for (DagInstNodeDO dagInstNode : dagInstNodeList) {
                DagInstNodeTask dagInstNodeTask = DagInstNodeTask.builder().dagInstId(
                        dagInstNode.getDagInstId()).nodeId(dagInstNode.getNodeId()).nodeTaskType(NodeTaskType.INSPECT)
                        .build();
                dagInstNodeTaskNotify.sendDagInstNodeTask(dagInstNodeTask);
            }
        }
    }

    public void doDagInstDispatch(DagInstDO dagInstDO) {
        if (Objects.isNull(dagInstDO)) {
            return;
        }
        IDagInstDispatcher iDagInstDispatcher = handlerList.stream().filter(
                handler -> handler.registerType() == dagInstDO.fetchStatus())
                .findFirst().orElse(null);
        if (Objects.nonNull(iDagInstDispatcher)) {
            iDagInstDispatcher.dispatch(dagInstDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public DagInstDispatch initInst(DagInstDO dagInstDO, DagInstStatus dagInstStatus) {
        Long dagInstId = dagInstDO.getId();

        //获取当前的DAG的结构(点+边)
        JSONObject contentJson = JSONObject.parseObject(dagInstDO.fetchDagDO().getContent());

        JSONArray nodeArray = contentJson.getJSONArray(DagConstant.NODE_KEY_IN_DAG);
        List<DagInstNodeDO> dagInstNodeDOList = new ArrayList<>();

        //前置节点
        JSONObject preNode = contentJson.getJSONObject(DagConstant.PRE_NODE_ID);
        if (Objects.nonNull(preNode)) {
            preNode.put("id", DagConstant.PRE_NODE_ID);
            dagInstNodeDOList.add(getDagInstNodeDO(dagInstDO, preNode));
        }

        //后置节点
        JSONObject postNode = contentJson.getJSONObject(DagConstant.POST_NODE_ID);
        if (Objects.nonNull(postNode)) {
            postNode.put("id", DagConstant.POST_NODE_ID);
            dagInstNodeDOList.add(getDagInstNodeDO(dagInstDO, postNode));
        }

        for (int i = 0; i < nodeArray.size(); i++) {
            JSONObject node = nodeArray.getJSONObject(i);
            dagInstNodeDOList.add(getDagInstNodeDO(dagInstDO, node));
        }

        JSONArray edgeArray = contentJson.getJSONArray(DagConstant.EDGE_KEY_IN_DAG);
        List<DagInstEdgeDO> dagInstEdgeDOList = new ArrayList<>();
        for (int i = 0; i < edgeArray.size(); i++) {
            JSONObject edge = edgeArray.getJSONObject(i);
            dagInstEdgeDOList.add(getDagInstEdgeDO(dagInstDO, edge));
        }

        DAG dag = DagUtil.calcDAG(dagInstNodeDOList, dagInstEdgeDOList);

        dagInstNodeDOList = topologySortByDag(dagInstNodeDOList, dag);
        dagBatchDAO.batchInsertNode(dagInstNodeDOList);
        dagBatchDAO.batchInsertEdge(dagInstEdgeDOList);

        DagInstDispatch dagInstDispatch = DagInstDispatch.builder().dagInstId(dagInstId).build();

        //有前置节点则进入PRE_RUNNING,无则进入到RUNNING
        if (Objects.nonNull(preNode)) {
            freshInstStatus(dagInstDO, dagInstStatus, DagInstStatus.PRE_RUNNING);

            dagInstDispatch.setDagInstStatus(DagInstStatus.PRE_RUNNING);

        } else {
            freshInstStatus(dagInstDO, dagInstStatus, DagInstStatus.RUNNING);
            dagInstDispatch.setDagInstStatus(DagInstStatus.RUNNING);
        }

        return dagInstDispatch;
    }

    public void freshInstStatus(DagInstDO dagInstDO, DagInstStatus fromStatus, DagInstStatus toStatus) {
        dagInstDAO.updateStatus(dagInstDO.getId(), fromStatus, toStatus);
        doOnInstIsEnd(dagInstDO, toStatus);
        publishInstStatusEvent(dagInstDO, fromStatus, toStatus);
    }

    public void freshInstStatusWithDetail(DagInstDO dagInstDO, DagInstStatus fromStatus, DagInstStatus toStatus, String statusDetail) {
        dagInstDAO.updateStatusWithDetail(dagInstDO.getId(), toStatus, statusDetail);
        doOnInstIsEnd(dagInstDO, toStatus);
        publishInstStatusEvent(dagInstDO, fromStatus, toStatus);
    }

    private void publishInstStatusEvent(DagInstDO dagInstDO, DagInstStatus fromStatus, DagInstStatus toStatus) {
        if (applicationProperties.isJarUseMode()) {
            DagInstStatusEvent dagInstStatusEvent = new DagInstStatusEvent(this, dagInstDO.getId(), fromStatus, toStatus);
            log.info(">>>dagInstNewService|publishInstStatusEvent|dagInstStatusEvent={}", dagInstStatusEvent);
            applicationContext.publishEvent(dagInstStatusEvent);
        }
    }

    private DagInstNodeDO getDagInstNodeDO(DagInstDO dagInstDO, JSONObject node) {
        String nodeId = node.getString("id");
        return DagInstNodeDO.builder()
                .gmtCreate(DateUtil.currentSeconds())
                .gmtModified(DateUtil.currentSeconds())
                .gmtStart(999999999999999L)
                .lockId("0")
                .dagInstId(dagInstDO.getId())
                .tcDagContentNodeSpec(JSONObject.toJSONString(node))
                .status(DagInstNodeStatus.INIT.toString())
                .nodeId(nodeId)
                .tcDagOrNodeDetail(getDagOrNodeDetail(node)).build();
    }

    private DagInstEdgeDO getDagInstEdgeDO(DagInstDO dagInstDO, JSONObject edge) {
        return DagInstEdgeDO.builder()
                .gmtCreate(DateUtil.currentSeconds())
                .gmtModified(DateUtil.currentSeconds())
                .dagInstId(dagInstDO.getId())
                .source(edge.getString("source"))
                .target(edge.getString("target"))
                .label(edge.getString("label"))
                .shape(edge.getString("shape"))
                .style(edge.getString("style"))
                .data(edge.getString("data"))
                .status(DagInstEdgeStatus.INIT.toString())
                .build();
    }

    private String getDagOrNodeDetail(JSONObject nodeJson) {
        JSONObject dataJson = nodeJson.getJSONObject("data");
        DagInstNodeType type = DagInstNodeType.valueOf(dataJson.getString("type"));
        Long defId = dataJson.getLong("defId");
        if (type == DagInstNodeType.DAG) {
            DagDO dagDO = dagDAO.getDagById(defId);
            return JSONObject.toJSONString(dagDO);
        } else if (type == DagInstNodeType.NODE) {
            DagNodeDO dagNodeDO = dagNodeDAO.getDagNodeById(defId);
            String formatType = dataJson.getString("format_type");
            String formatDetail = dataJson.getString("format_detail");
            Long maxRetryTimes = dataJson.getLong("maxRetryTimes");
            String retryExpression = dataJson.getString("retryExpression");
            Long runTimeout = dataJson.getLong("runTimeout");
            if (StringUtils.isNotEmpty(formatType)) {
                dagNodeDO.setFormatType(formatType);
            }
            if (StringUtils.isNotEmpty(formatDetail)) {
                dagNodeDO.setFormatDetail(formatDetail);
            }
            if (Objects.nonNull(maxRetryTimes)) {
                dagNodeDO.setMaxRetryTimes(maxRetryTimes);
            }
            if (StringUtils.isNotEmpty(retryExpression)) {
                dagNodeDO.setRetryExpression(retryExpression);
            }
            if (Objects.nonNull(runTimeout)) {
                dagNodeDO.setRunTimeout(runTimeout);
            }

            return JSONObject.toJSONString(dagNodeDO);
        }

        return null;
    }

    //实例结束时
    private void doOnInstIsEnd(DagInstDO dagInstDO, DagInstStatus toStatus) {
        if (toStatus.isEnd()) {
            Long dagInstId = dagInstDO.fetchRelationInstId();
            String nodeId = dagInstDO.fetchRelationNodeId();

            //当前实例属于某个子Node，则通知
            if (Objects.nonNull(dagInstId) && StringUtils.isNotEmpty(nodeId)) {
                DagInstNodeTask dagInstNodeTask = DagInstNodeTask.builder().dagInstId(
                        dagInstId).nodeId(nodeId).nodeTaskType(NodeTaskType.INSPECT).build();
                dagInstNodeTaskNotify.sendDagInstNodeTask(dagInstNodeTask);
            }

            log.info(">>>dagInstNewService|doOnInstIsEnd|dagInstId={}, toStatus={}", dagInstDO.getId(), toStatus);
        }
    }

    public void freshGlobalData(Long dagInstId) {
        List<DagInstNodeDO> successNodeList = dagInstNodeDAO.listBySuccess(dagInstId);

        List<String> nodeIdList = successNodeList.stream().map(dagInstNodeDO -> dagInstNodeDO.getNodeId()).collect(Collectors.toList());

        log.info(">>>dagInstNewService|freshGlobalData|dagInstId={}, nodeIdList={}", dagInstId, nodeIdList);

        DagInstDO updateDagInst = DagInstDO.builder().id(dagInstId).build();
        for (DagInstNodeDO successNode : successNodeList) {
            updateInstGlobalObject(updateDagInst, successNode);
            updateInstGlobalParams(updateDagInst, successNode);
            updateInstGlobalResult(updateDagInst, successNode);
        }

        updateDagInst.setVersion(CollectionUtils.size(nodeIdList));

        int count = dagInstDAO.updateWithVersion(updateDagInst);
        log.info(">>>dagInstNewService|freshGlobalData|exit|dagInstId={}, count={}", dagInstId, count);
    }

    private void updateInstGlobalObject(DagInstDO dagInstDO, DagInstNodeDO dagInstNodeDO) {
        String globalObject = dagInstNodeDO.getGlobalObject();
        if (StringUtils.isEmpty(globalObject) || StringUtils.equals(DagConstant.FAAS_NULL_SERIALIZE, globalObject)) {
            return;
        }

        if (isDelete(globalObject)) {
            dagInstDO.setGlobalObject(DagConstant.FAAS_NULL_SERIALIZE);
        } else {
            dagInstDO.setGlobalObject(globalObject);
        }

    }

    private JSONObject mergeJsonObject(JSONObject j1, JSONObject j2) {
        JSONObject retJson = new JSONObject();
        retJson.putAll(j1);
        retJson.putAll(j2);
        for (String key : j1.keySet()) {
            if (!j2.containsKey(key)) {
                continue;
            }
            Object value1 = JSON.toJSON(j1.get(key));
            Object value2 = JSON.toJSON(j2.get(key));
            if (value1 instanceof JSONObject && value2 instanceof JSONObject) {
                retJson.put(key, mergeJsonObject((JSONObject) value1, (JSONObject) value2));
            }
        }
        return retJson;
    }

    private void updateInstGlobalParams(DagInstDO dagInstDO, DagInstNodeDO dagInstNodeDO) {
        String globalParams = dagInstNodeDO.getGlobalParams();
        JSONObject globalParamsJson;
        if (isDelete(globalParams)) {
            globalParamsJson = new JSONObject();

        } else {
            globalParamsJson = JSONObject.parseObject(globalParams);
            if (Objects.isNull(globalParamsJson)) {
                globalParamsJson = new JSONObject();
            }

            globalParamsJson = mergeJsonObject(dagInstDO.fetchGlobalParamsJson(), globalParamsJson);
        }

        dagInstDO.setGlobalParams(globalParamsJson.toJSONString());
        if (StringUtils.isNotEmpty(dagInstDO.getGlobalParams())) {
            log.info(">>>dagInstNewService|setGlobalParams|merge node globalParams to inst|dagInstId={}, nodeId={}, " +
                            "md5={}",
                    dagInstNodeDO.getDagInstId(), dagInstNodeDO.getNodeId(),
                    DigestUtils.md5DigestAsHex(dagInstDO.getGlobalParams().getBytes()));
        }
    }

    private void updateInstGlobalResult(DagInstDO dagInstDO, DagInstNodeDO dagInstNodeDO) {
        JSONObject globalResultJson = dagInstDO.fetchGlobalResultJson();
        globalResultJson.put(dagInstNodeDO.getNodeId(), JSONObject.parseObject(dagInstNodeDO.getGlobalResult()));
        dagInstDO.setGlobalResult(JSONObject.toJSONString(globalResultJson));
    }

    /**
     * 外部调用，不可删
     *
     * @param dagInstId
     * @return
     */
    public DagInstDO getDagInstById(Long dagInstId) {
        log.info(">>>dagInstNewService|getDagInstById|externalCall|dagInstId={}", dagInstId);
        return dagInstDAO.getDagInstById(dagInstId);
    }
}
