package com.alibaba.tesla.dag.services;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.Tools;
import com.alibaba.tesla.dag.constant.DagConstant;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeType;
import com.alibaba.tesla.dag.notify.DagInstDispatch;
import com.alibaba.tesla.dag.notify.DagInstNodeTask;
import com.alibaba.tesla.dag.notify.IDagInstNotify;
import com.alibaba.tesla.dag.repository.dao.DagInstDAO;
import com.alibaba.tesla.dag.repository.dao.DagInstEdgeDAO;
import com.alibaba.tesla.dag.repository.dao.DagInstNodeDAO;
import com.alibaba.tesla.dag.repository.domain.DagInstDO;
import com.alibaba.tesla.dag.repository.domain.DagInstEdgeDO;
import com.alibaba.tesla.dag.repository.domain.DagInstNodeDO;
import com.alibaba.tesla.dag.repository.domain.DagNodeDO;
import com.alibaba.tesla.dag.schedule.status.DagInstEdgeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstStatus;
import com.alibaba.tesla.dag.schedule.task.TaskStatus;
import com.alibaba.tesla.dag.util.DateUtil;
import com.alibaba.tesla.dag.util.MonitorUtil;
import com.google.common.base.Throwables;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.DigestUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author QianMo
 * @date 2021/04/22.
 */
@Slf4j
@Service
public class DagInstNodeNewService {

    @Autowired
    private DagInstNodeDAO dagInstNodeDAO;

    @Autowired
    private DagInstEdgeDAO dagInstEdgeDAO;

    @Autowired
    private DagInstDAO dagInstDAO;

    @Autowired
    private DagInstNewService dagInstNewService;

    @Autowired
    private List<AbstractActionNewService> actionNewServiceList;

    @Autowired
    private IDagInstNotify dagInstNotify;

    public void startDagInstNode(DagInstNodeTask dagInstNodeTask) {
        long beginTime = System.currentTimeMillis();

        DagInstNodeDO dagInstNodeDO = dagInstNodeDAO.getDagInstNode(dagInstNodeTask.getDagInstId(),
                dagInstNodeTask.getNodeId());

        if (Objects.isNull(dagInstNodeDO)) {
            log.info(">>>dagInstNodeNewService|startDagInstNode|Not Found|dagInstId={}, nodeId={}", dagInstNodeTask.getDagInstId(), dagInstNodeTask.getNodeId());
            return;
        }

        DagInstNodeStatus status = DagInstNodeStatus.valueOf(dagInstNodeDO.getStatus());
        log.info(">>>dagInstNodeNewService|startDagInstNode|dagInstId={}, nodeId={}, status={}", dagInstNodeTask.getDagInstId(), dagInstNodeTask.getNodeId(), status);

        try {
            switch (status) {
                case INIT:
                    start(dagInstNodeDO);
                    break;
                default:
                    log.warn(">>>dagInstNodeNewService|startDagInstNode|Status is Wrong|dagInstId={}, nodeId={}, status={}", dagInstNodeTask.getDagInstId(), dagInstNodeTask.getNodeId(), status);
                    break;
            }
        } catch (Exception e) {
            log.error(">>>dagInstNodeNewService|startDagInstNode|dagInstId={}, err={}", dagInstNodeTask.getDagInstId(), e.getMessage(), e);
            dagInstNodeDAO.updateStatusWithDetail(dagInstNodeDO.getId(), DagInstNodeStatus.EXCEPTION,
                    Throwables.getStackTraceAsString(e));

            DagInstDispatch dagInstDispatch = DagInstDispatch.builder().dagInstId(dagInstNodeDO.getDagInstId())
                    .nodeId(dagInstNodeDO.getNodeId())
                    .dagInstStatus(
                            DagInstStatus.RUNNING).build();
            dagInstNotify.sendDagInstDispatch(dagInstDispatch);
        }

        MonitorUtil.nodeStartMonitor.addCost(System.currentTimeMillis() - beginTime);
    }

    //启动节点，如果节点为DAG类型，则启动一个新的实例
    public void start(DagInstNodeDO dagInstNodeDO) throws Exception {
        DagInstDO dagInstDO = dagInstDAO.getDagInstById(dagInstNodeDO.getDagInstId());
        DagInstNodeDO updateInstNode = DagInstNodeDO.builder().id(dagInstNodeDO.getId()).build();
        updateInstNode.setGmtStart(DateUtil.currentSeconds());

        switch (dagInstNodeDO.fetchNodeType()) {
            case DAG:
                if (Objects.nonNull(dagInstNodeDO.getSubDagInstId()) && dagInstNodeDO.getSubDagInstId() > 0) {
                    log.warn(">>>dagInstNodeNewService|start|重复启动子DAG|subDagInstId={}",
                            dagInstNodeDO.getSubDagInstId());
                    return;
                }

                Long subDagInstId = dagInstNewService.submitSub(dagInstNodeDO, dagInstDO);
                log.info(">>>dagInstNodeNewService|SUCCESS|startDag|subDagInstId={}", subDagInstId);
                updateInstNode.setSubDagInstId(subDagInstId);

                break;
            case NODE:
                long beginTime = System.currentTimeMillis();
                if (StringUtils.isNotEmpty(dagInstNodeDO.getTaskId())) {
                    log.warn(">>>dagInstNodeNewService|start|重复启动作业|dagInstId={}, nodeId={}, taskId={}",
                            dagInstNodeDO.getDagInstId(), dagInstNodeDO.getNodeId(), dagInstNodeDO.getTaskId());
                    return;
                }
                DagNodeDO dagNode = dagInstNodeDO.fetchDagNode();

                DagNodeType dagNodeType = dagNode.fetchNodeType();
                AbstractActionNewService abstractActionNewService = actionNewServiceList.stream().filter(
                        actionNewService -> actionNewService.registerNodeType() == dagNodeType)
                        .findFirst().orElse(null);

                if (Objects.isNull(abstractActionNewService)) {
                    throw new Exception("no such type: " + dagNodeType);
                }

                Long taskId = abstractActionNewService.start(dagInstDO, dagInstNodeDO);
                log.info(">>>dagInstNodeNewService|SUCCESS|startJob|dagInstId={}, nodeId={}, taskId={}",
                        dagInstNodeDO.getDagInstId(), dagInstNodeDO.getNodeId(), taskId);

                MonitorUtil.taskMonitor.addCost(System.currentTimeMillis() - beginTime);
                updateInstNode.setTaskId(Long.toString(taskId));
                break;
            default:
                break;
        }
        updateInstNode.setStatus(DagInstNodeStatus.RUNNING.toString());

        dagInstNodeDAO.update(updateInstNode);
    }

    /**
     * 节点自查
     *
     * @param dagInstNodeTask
     * @throws Exception
     */
    public void inspectDagInstNode(DagInstNodeTask dagInstNodeTask) throws Exception {
        DagInstNodeDO dagInstNodeDO = dagInstNodeDAO.getDagInstNode(dagInstNodeTask.getDagInstId(),
                dagInstNodeTask.getNodeId());

        if (Objects.isNull(dagInstNodeDO)) {
            log.info(">>>dagInstNodeNewService|inspectDagInstNode|Not Found|dagInstNodeTask={}", dagInstNodeTask);
            return;
        }

        switch (dagInstNodeDO.fetchNodeType()) {
            //如果是DAG节点，则把实例里的数据复制到节点中来
            case DAG:
                Long subDagInstId = dagInstNodeDO.getSubDagInstId();
                DagInstDO subDagInst = dagInstDAO.getDagInstById(subDagInstId);
                DagInstStatus subDagInstStatus = subDagInst.fetchStatus();

                if (subDagInstStatus == DagInstStatus.EXCEPTION) {
                    dagInstNodeDAO.updateStatus(dagInstNodeDO.getId(), DagInstNodeStatus.EXCEPTION);
                } else if (subDagInstStatus == DagInstStatus.SUCCESS) {
                    DagInstNodeDO updateDagNodeInst = DagInstNodeDO.builder().id(dagInstNodeDO.getId()).build();
                    updateDagNodeInst.setGlobalResult(subDagInst.getGlobalResult());
                    updateDagNodeInst.setGlobalParams(subDagInst.getGlobalParams());

                    if (StringUtils.isNotEmpty(subDagInst.getGlobalParams())) {
                        log.info(">>>dagInstNodeNewService|setGlobalParams|copy globalParams from sub to " +
                                        "parent|dagInstId={}, subDagInst={}, md5={}",
                                dagInstNodeDO.getDagInstId(), subDagInstId,
                                DigestUtils.md5DigestAsHex(subDagInst.getGlobalParams().getBytes()));
                    }
                    updateDagNodeInst.setGlobalObject(subDagInst.getGlobalObject());
                    updateDagNodeInst.setStatus(DagInstNodeStatus.MERGE.toString());
                    updateDagNodeInst.setLockId(DateUtil.currentTimeMillis());
                    dagInstNodeDAO.update(updateDagNodeInst);
                    log.info(">>>dagInstNodeNewService|inspectDagInstNode|MERGE|dagInstId={}", dagInstNodeDO.getDagInstId());

                    calcEdge(dagInstNodeDO);

                    dagInstNewService.freshGlobalData(dagInstNodeDO.getDagInstId());

                    updateDagNodeInst = DagInstNodeDO.builder().id(dagInstNodeDO.getId()).build();
                    updateDagNodeInst.setStatus(DagInstNodeStatus.SUCCESS.toString());
                    dagInstNodeDAO.update(updateDagNodeInst);
                    log.info(">>>dagInstNodeNewService|inspectDagInstNode|SUCCESS|dagInstId={}", dagInstNodeDO.getDagInstId());
                } else {
                    log.info(">>>dagInstNodeNewService|inspectDagInstNode|兜底条件|subDagInstId={}, subDagInstStatus={}", subDagInstId, subDagInstStatus);
                }

                DagInstDispatch dagInstDispatch = DagInstDispatch.builder().dagInstId(dagInstNodeDO.getDagInstId())
                        .nodeId(dagInstNodeDO.getNodeId())
                        .dagInstStatus(
                                DagInstStatus.RUNNING).build();
                dagInstNotify.sendDagInstDispatch(dagInstDispatch);

                break;
            case NODE:
                long beginTime = System.currentTimeMillis();
                DagInstNodeStatus nodeStatus = dagInstNodeDO.fetchNodeStatus();
                if (nodeStatus == DagInstNodeStatus.RUNNING) {
                    String taskId = dagInstNodeDO.getTaskId();
                    if (StringUtils.isNotEmpty(taskId)) {
                        DagNodeDO dagNode = dagInstNodeDO.fetchDagNode();
                        DagNodeType dagNodeType = dagNode.fetchNodeType();
                        AbstractActionNewService abstractActionNewService = actionNewServiceList.stream().filter(
                                actionNewService -> actionNewService.registerNodeType() == dagNodeType)
                                .findFirst().orElse(null);

                        Long taskIdLong = Long.parseLong(taskId);
                        TaskStatus taskStatus = abstractActionNewService.status(taskIdLong);

                        MonitorUtil.statusMonitor.addCost(System.currentTimeMillis() - beginTime);
                        jobCallBack(dagInstNodeDO.getDagInstId(), dagInstNodeDO.getNodeId(), taskIdLong, taskStatus, "");
                    }
                }

                break;
            default:
                break;
        }
    }

    /**
     * 作业回调
     *
     * @param dagInstId
     * @param nodeId
     * @param taskStatus
     * @throws Exception
     */
    public void jobCallBack(Long dagInstId, String nodeId, Long taskId, TaskStatus taskStatus, String detail) throws Exception {
        log.info(">>>dagInstNodeNewService|jobCallBack|enter|dagInstId={}, nodeId={}, taskId={}, taskStatus={}, detail={}", dagInstId, nodeId,
                taskId, taskStatus, detail);

        DagInstNodeDO dagInstNodeDO = dagInstNodeDAO.getDagInstNode(dagInstId, nodeId);
        DagInstDO dagInstDO = dagInstDAO.getDagInstById(dagInstId);

        long beginTime = System.currentTimeMillis();

        if (taskStatus == TaskStatus.SUCCESS) {
            DagNodeDO dagNode = dagInstNodeDO.fetchDagNode();
            DagNodeType dagNodeType = dagNode.fetchNodeType();
            AbstractActionNewService abstractActionNewService = actionNewServiceList.stream().filter(
                    actionNewService -> actionNewService.registerNodeType() == dagNodeType)
                    .findFirst().orElse(null);

            if (Objects.isNull(abstractActionNewService)) {
                throw new Exception("no such type: " + dagNodeType);
            }

            DagInstNodeDO updateDagNodeInst = DagInstNodeDO.builder().id(dagInstNodeDO.getId()).build();

            JSONObject outJson = null;
            try {
                outJson = abstractActionNewService.stdout(taskId);
            }
            catch (Exception e){
                dagInstNodeDAO.updateStatusWithDetail(dagInstNodeDO.getId(), DagInstNodeStatus.EXCEPTION, e.getMessage());
            }

            if(Objects.nonNull(outJson)) {
                MonitorUtil.stdOutMonitor.addCost(System.currentTimeMillis() - beginTime);

                updateDagNodeInst.setGlobalObject(outJson.getString(DagConstant.OUTPUT_GLOBAL_OBJECT_KEY));
                updateDagNodeInst.setGlobalParams(outJson.getString(DagConstant.OUTPUT_GLOBAL_PARAMS_KEY));
                if (StringUtils.isNotEmpty(updateDagNodeInst.getGlobalParams())) {
                    log.info(">>>dagInstNodeNewService|setGlobalParams|set globalParams to node|dagInstId={}, " +
                                    "nodeId={}, md5={}",
                            dagInstId, nodeId,
                            DigestUtils.md5DigestAsHex(updateDagNodeInst.getGlobalParams().getBytes()));
                }
                updateDagNodeInst.setGlobalResult(AbstractActionNewService.getDataResultAndOutput(outJson).toJSONString());
                updateDagNodeInst.setStatus(DagInstNodeStatus.MERGE.toString());
                updateDagNodeInst.setLockId(DateUtil.currentTimeMillis());

                dagInstNodeDAO.update(updateDagNodeInst);
                log.info(">>>dagInstNodeNewService|jobCallBack|MERGE|dagInstId={}", dagInstNodeDO.getDagInstId());

                //注意！！！这里的顺序不能调整，freshGlobalData后才能计算边，边使用了上下文的值
                dagInstNewService.freshGlobalData(dagInstId);
                calcEdge(dagInstNodeDO);

                updateDagNodeInst = DagInstNodeDO.builder().id(dagInstNodeDO.getId()).build();
                updateDagNodeInst.setStatus(DagInstNodeStatus.SUCCESS.toString());
                dagInstNodeDAO.update(updateDagNodeInst);
                log.info(">>>dagInstNodeNewService|jobCallBack|SUCCESS|dagInstId={}", dagInstNodeDO.getDagInstId());
            }
        } else if (taskStatus == TaskStatus.EXCEPTION) {
            dagInstNodeDAO.updateStatusWithDetail(dagInstNodeDO.getId(), DagInstNodeStatus.EXCEPTION,
                    StringUtils.isEmpty(detail) ? "exception in running task" : detail);
        }

        DagInstDispatch dagInstDispatch = DagInstDispatch.builder().dagInstId(dagInstNodeDO.getDagInstId())
                .nodeId(dagInstNodeDO.getNodeId())
                .dagInstStatus(dagInstDO.fetchStatus()).build();
        dagInstNotify.sendDagInstDispatch(dagInstDispatch);

    }

    public void calcEdge(DagInstNodeDO dagInstNodeDO) {
        List<DagInstEdgeDO> edgeList = dagInstEdgeDAO.getList(dagInstNodeDO.getDagInstId()).stream().filter(
                dagInstEdgeDO -> StringUtils.equals(dagInstEdgeDO.getSource(), dagInstNodeDO.getNodeId())).collect(
                Collectors.toList());

        if (CollectionUtils.isNotEmpty(edgeList)) {
            DagInstDO dagInstDO = null;

            for (DagInstEdgeDO edge : edgeList) {
                try {
                    String expression = edge.fetchExpressionString();
                    Boolean isPass = true;
                    if (StringUtils.isNotEmpty(expression)) {
                        if (Objects.isNull(dagInstDO)) {
                            dagInstDO = dagInstDAO.getDagInstById(dagInstNodeDO.getDagInstId());
                        }

                        isPass = Tools.execExpression(edge.fetchKey(), dagInstDO.fetchExpressionParamsJson(), expression,
                                Boolean.class);
                        if (Objects.isNull(isPass)) {
                            isPass = false;
                        }
                    }

                    edge.setIsPass(isPass ? 1 : 0);
                    edge.setStatus(DagInstEdgeStatus.SUCCESS.toString());
                } catch (Exception e) {
                    edge.setStatus(DagInstEdgeStatus.EXCEPTION.toString());
                    edge.setException(Throwables.getStackTraceAsString(e));
                }

                edge.setGmtModified(DateUtil.currentSeconds());

                dagInstEdgeDAO.update(edge);
            }
        }
    }
}
