package com.alibaba.tesla.dag.services;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.model.domain.TcDagInst;
import com.alibaba.tesla.dag.model.domain.TcDagInstNode;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeFormatType;
import com.alibaba.tesla.dag.model.repository.TcDagInstNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagInstRepository;
import com.alibaba.tesla.dag.provider.DagInstProvider;
import com.alibaba.tesla.dag.schedule.status.DagInstNodeStatus;
import com.alibaba.tesla.dag.schedule.status.DagInstStatus;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.DependsOn;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Service
@DependsOn(value = "beanUtil")
public class DagInstService {

    @Autowired
    private DagInstProvider dagInstProvider;

    @Autowired
    private TcDagInstNodeRepository dagInstNodeRepository;

    @Autowired
    private TcDagInstRepository dagInstRepository;

    public Long start(String appId, String name, JSONObject globalVariable, String operator, boolean isStandalone) throws Exception {
        JSONObject commonParams = new JSONObject();
        commonParams.put("nameEn", name);
        commonParams.put("configs", globalVariable);
        return dagInstProvider.submit(appId, operator, "", "", commonParams, isStandalone);
    }

    public Long start(String appId, String name, JSONObject globalVariable, boolean isStandalone) throws Exception {
        return start(appId, name, globalVariable, "", isStandalone);
    }

    public Long start(String name, JSONObject globalVariable, boolean isStandalone) throws Exception {
        return start("tesla", name, globalVariable, isStandalone);
    }

    public Object getElementOut(Long dagInstNodeId, String elementKey) throws Exception {
        if (dagInstNodeRepository.findFirstById(dagInstNodeId) == null) {
            throw new Exception("dagInstNodeId: " + dagInstNodeId + " not exists");
        }
        TcDagInstNode dagInstNode = dagInstNodeRepository.findFirstById(dagInstNodeId);
        JSONObject outJson = dagInstNode.outJson();
        JSON json = AbstractActionNewService.getDataData(outJson, JSON.class);
        if (json instanceof JSONObject) {
            return ((JSONObject) json).get(elementKey);
        } else if (json instanceof JSONArray) {
            JSONArray dataArray = (JSONArray) json;
            List<JSONObject> formatDetailJsonList = JSONArray.parseArray(dagInstNode.dagNode().getFormatDetail())
                    .toJavaList(JSONObject.class);
            for (int index = 0; index < formatDetailJsonList.size(); index++) {
                JSONObject formatDetailJson = formatDetailJsonList.get(index);
                if (formatDetailJson.getString("type").equals(DagNodeFormatType.CUSTOM.name())) {
                    JSONObject dataJson = dataArray.getJSONObject(index);
                    if (dataJson.containsKey(elementKey)) {
                        return dataJson.get(elementKey);
                    }
                }
            }
            return null;
        } else {
            throw new Exception("dataData is not JSONObject or JSONArray; dataData: " + JSONObject.toJSONString(json));
        }
    }

    public Object getElementOut(Long dagInstNodeId, int elementIndex) throws Exception {
        if (dagInstNodeRepository.findFirstById(dagInstNodeId) == null) {
            throw new Exception("dagInstNodeId: " + dagInstNodeId + " not exists");
        }
        TcDagInstNode dagInstNode = dagInstNodeRepository.findFirstById(dagInstNodeId);
        JSONObject outJson = dagInstNode.outJson();
        return AbstractActionNewService.getDataData(outJson, JSONArray.class).get(elementIndex);
    }

    public TcDagInst get(long dagInstId) {
        return dagInstRepository.findFirstById(dagInstId);
    }

    public List<Long> listDagInstNodeId(long dagInstId) {
        return dagInstNodeRepository.findAllByDagInstId(dagInstId)
                .stream()
                .map(TcDagInstNode::getId)
                .collect(Collectors.toList());
    }

    public void updateGlobalVariable(Long dagInstId, JSONObject globalVariable) {
        if (CollectionUtils.isEmpty(globalVariable)) {
            return;
        }
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        if (dagInst == null) {
            return;
        }
        JSONObject globalVariableJson = dagInst.globalVariableJson();
        globalVariableJson.putAll(globalVariable);
        dagInstRepository.updateGlobalVariable(dagInstId, JSONObject.toJSONString(globalVariableJson, true));

        List<TcDagInstNode> nodeList = dagInstNodeRepository.findAllByDagInstId(dagInstId);
        for (TcDagInstNode node : nodeList) {
            Long subDagInstId = node.getSubDagInstId();
            if (subDagInstId == null || subDagInstId <= 0) {
                continue;
            }
            updateGlobalVariable(subDagInstId, globalVariable);
        }
    }

    public void rerunDagInstByStatus(Long dagInstId, List<DagInstNodeStatus> statusList) throws Exception {
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        if (!dagInst.status().isEnd()) {
            throw new Exception("can not rerun dagInst when it's status is " + dagInst.status().toString());
        }
        List<TcDagInstNode> dagInstNodeList = dagInstNodeRepository.findAllByDagInstIdAndStatusIn(
                dagInstId,
                statusList.stream().map(Enum::toString).collect(Collectors.toList())
        );

        if (CollectionUtils.isEmpty(dagInstNodeList)) {
            return;
        }

        for (TcDagInstNode node : dagInstNodeList) {
            Long subDagInstId = node.getSubDagInstId();
            if (subDagInstId == null || subDagInstId <= 0) {
                dagInstNodeRepository.updateStatusById(DagInstNodeStatus.INIT.toString(), node.getId());
            } else {
                rerunDagInstByStatus(subDagInstId, statusList);
            }
        }
        dagInstRepository.setStatus(DagInstStatus.RUNNING.toString(), dagInstId);
    }

    public void stop(Long dagInstId) {
        dagInstRepository.updateStatus(dagInstId, DagInstStatus.WAIT_STOP.toString());
    }

    public void rerunStoppedExceptionDagInst(Long dagInstId) throws Exception {
        rerunDagInstByStatus(
                dagInstId,
                Arrays.asList(
                        DagInstNodeStatus.STOPPED,
                        DagInstNodeStatus.SKIP_CAUSE_BY_STOPPED,
                        DagInstNodeStatus.STOPPED,
                        DagInstNodeStatus.SKIP_CAUSE_BY_STOPPED
                )
        );
    }

    public void rerunStoppedDagInst(Long dagInstId) throws Exception {
        rerunDagInstByStatus(
                dagInstId,
                Arrays.asList(
                        DagInstNodeStatus.STOPPED,
                        DagInstNodeStatus.SKIP_CAUSE_BY_STOPPED
                )
        );
    }

    public void rerunExceptionDagInst(Long dagInstId) throws Exception {
        rerunDagInstByStatus(
                dagInstId,
                Arrays.asList(
                        DagInstNodeStatus.EXCEPTION,
                        DagInstNodeStatus.SKIP_CAUSE_BY_EXCEPTION
                )
        );
    }

    public boolean isEnd(Long instanceId) {
        return dagInstRepository.findFirstById(instanceId).status().isEnd();
    }

    public JSONObject getNativeById(Long dagInstId, boolean withData) throws Exception {
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        JSONObject jsonObject = new JSONObject();
        JSONObject globalResultJson = dagInst.globalResultJson();
        if (withData) {
            globalResultJson = getTotalGlobalResult(dagInstId);
        }
        regularGlobalResult(globalResultJson);
        jsonObject.put("globalVariable", dagInst.globalVariableJson());
        jsonObject.put("globalResult", globalResultJson);
        jsonObject.put("globalParams", dagInst.globalParamsJson());
        jsonObject.put("status", dagInst.status());
        jsonObject.put("channel", dagInst.getChannel());
        jsonObject.put("env", dagInst.getEnv());

        return jsonObject;
    }

    private JSONObject getTotalGlobalResult(Long dagInstId) throws Exception {
        JSONObject jsonObject = new JSONObject();
        TcDagInst dagInst = dagInstRepository.findFirstById(dagInstId);
        List<TcDagInstNode> dagInstNodeList = dagInst.nodes();
        for (TcDagInstNode dagInstNode : dagInstNodeList) {
            if (dagInstNode.getSubDagInstId() == null) {
                JSONObject outJson = AbstractActionNewService.getData(dagInstNode.outJsonWithCache());
                jsonObject.put(dagInstNode.nodeId(), outJson);
            } else {
                jsonObject.put(dagInstNode.nodeId(), getTotalGlobalResult(dagInstNode.getSubDagInstId()));
            }
        }
        return jsonObject;
    }

    private void regularGlobalResult(JSONObject globalResult) {
        if (globalResult.containsKey("result") || globalResult.containsKey("data")) {
            globalResult.remove("output");
            globalResult.remove("chatops");
        }
        for (String key : globalResult.keySet()) {
            Object value = JSON.toJSON(globalResult.get(key));
            if (value instanceof JSONObject) {
                regularGlobalResult((JSONObject) value);
            }
        }
    }

}
