package com.alibaba.tesla.dag.services;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.common.Tools;
import com.alibaba.tesla.dag.local.AbstractLocalNodeBase;
import com.alibaba.tesla.dag.local.ClassService;
import com.alibaba.tesla.dag.local.LocalTaskDO;
import com.alibaba.tesla.dag.model.domain.dagnode.DagInstNodeRunRet;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeDetailLocal;
import com.alibaba.tesla.dag.repository.dao.DagInstDAO;
import com.alibaba.tesla.dag.repository.dao.DagInstNodeDAO;
import com.alibaba.tesla.dag.repository.dao.DagInstNodeStdDAO;
import com.alibaba.tesla.dag.repository.domain.DagInstDO;
import com.alibaba.tesla.dag.repository.domain.DagInstNodeDO;
import com.alibaba.tesla.dag.repository.domain.DagInstNodeStdDO;
import com.alibaba.tesla.dag.repository.domain.DagNodeDO;
import com.alibaba.tesla.dag.schedule.task.TaskStatus;
import com.alibaba.tesla.dag.util.DateUtil;
import com.google.common.base.Throwables;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class LocalTaskService {
    @Autowired
    private DagInstNodeStdDAO dagInstNodeStdDAO;

    @Autowired
    private DagInstNodeDAO dagInstNodeDAO;

    @Autowired
    private DagInstDAO dagInstDAO;

    @Autowired
    private ClassService classService;

    @Autowired
    private DagInstNodeNewService dagInstNodeNewService;

    @Value("${dag.local.thread-pool-size:100}")
    private int localThreadPoolSize;

    private ThreadPoolExecutor localTaskThreadPoolExecutor;
    private ScheduledExecutorService scheduledService = Executors.newScheduledThreadPool(5);
    private Map<Long, Future> localTaskId2FutureMap = new HashMap<>(1000);
    private Map<Long, AbstractLocalNodeBase> localTaskId2NodeMap = new HashMap<>(1000);

    @PostConstruct
    public void initScheduled() {
        log.info(">>>localTaskService|initScheduled|enter|localThreadPoolSize={}", localThreadPoolSize);

        localTaskThreadPoolExecutor = new ThreadPoolExecutor(localThreadPoolSize, localThreadPoolSize,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>());

        scheduledService.scheduleAtFixedRate(new TimeoutHandler(), 60, 5, TimeUnit.SECONDS);
        scheduledService.scheduleAtFixedRate(new ReBuilder(), 10, 10, TimeUnit.SECONDS);
    }

    public ThreadPoolExecutor getLocalTaskThreadPoolExecutor() {
        return localTaskThreadPoolExecutor;
    }


    private DagInstNodeDO queryByLocalTaskId(Long localTaskId) {
        DagInstNodeStdDO dagInstNodeStdDO = dagInstNodeStdDAO.getDagInstNodeStdById(localTaskId);
        if (Objects.isNull(dagInstNodeStdDO)) {
            log.warn(">>>localTaskService|run|local task does not exist|localTaskId={}", localTaskId);
            return null;
        }

        String currentStatus = dagInstNodeStdDO.getStatus();
        if (DagInstNodeStdDAO.UN_END_STATUS_LIST.contains(currentStatus)) {
            return dagInstNodeDAO.getDagInstNode(dagInstNodeStdDO.getDagInstNodeId());
        } else {
            log.warn(">>>localTaskService|queryByLocalTaskId|double check|dagInstId={}, nodeId={}, localTaskId={}, " +
                            "currentStatus={}",
                    dagInstNodeStdDO.getDagInstId(), dagInstNodeStdDO.getDagInstNodeId(), localTaskId, currentStatus);
            return null;
        }

    }

    private boolean isTimeout(Long localTaskId) {
        AbstractLocalNodeBase localNode = localTaskId2NodeMap.get(localTaskId);
        if (Objects.isNull(localNode) || localNode.realRunTimeout == 0) {
            return false;
        }

        long actTimestamp = DateUtil.currentSeconds() - localNode.actRunTimestamp;
        if (actTimestamp > localNode.realRunTimeout) {
            return true;
        }

        return false;
    }

    public void doLocalTask(LocalTaskDO localTaskDO) {
        Long dagInstId = localTaskDO.getDagInstId();
        String nodeId = localTaskDO.getNodeId();
        Long localTaskId = localTaskDO.getTaskId();

        try {
            if (localTaskId2FutureMap.containsKey(localTaskDO.getTaskId())) {
                log.warn(">>>localTaskService|doLocalTask|task is exist|dagInstId={}, nodeId={}, localTaskId={}",
                        dagInstId, nodeId, localTaskId);
                return;
            }
            Future future = localTaskThreadPoolExecutor.submit(new LocalTask(localTaskDO));
            localTaskId2FutureMap.put(localTaskDO.getTaskId(), future);
        } catch (Throwable e) {
            log.error(">>>localTaskService|doLocalTask|Err|dagInstId={}, nodeId={}, localTaskId={}, Err={}",
                    dagInstId, nodeId, localTaskId, e.toString(), e);
            dagInstNodeStdDAO.updateStatusWithDetail(localTaskId, TaskStatus.EXCEPTION,
                    Throwables.getStackTraceAsString(e));
            try {
                dagInstNodeNewService.jobCallBack(dagInstId, nodeId, localTaskId, TaskStatus.EXCEPTION,
                        Throwables.getStackTraceAsString(e));
            } catch (Exception e1) {
                log.error(">>>localTaskService|jobCallBack|Err|dagInstId={}, nodeId={}, localTaskId={}, Err={}",
                        dagInstId, nodeId, localTaskId, e1.toString(), e1);
            }
        }

    }

    public AbstractLocalNodeBase newInstance(Long localTaskId, DagInstNodeDO dagInstNode) throws Exception {
        if (localTaskId2NodeMap.containsKey(localTaskId)) {
            return localTaskId2NodeMap.get(localTaskId);
        }
        DagInstDO dagInstDO = dagInstDAO.getDagInstById(dagInstNode.getDagInstId());

        DagNodeDO dagNodeDO = dagInstNode.fetchDagNode();
        DagNodeDetailLocal dagNodeDetailLocal = (DagNodeDetailLocal) dagNodeDO.fetchDetailInterface();
        String dagNodeName = dagNodeDetailLocal.getName();
        JSONObject params = AbstractActionNewService.inputParams(dagInstDO, dagInstNode);

        Class nodeClass = classService.nodeMap.get(dagNodeName);
        AbstractLocalNodeBase localNode = (AbstractLocalNodeBase) nodeClass.getConstructor().newInstance();
        log.info(">>>localTaskService|newInstance|dagInstId={}, taskId={}, dagNodeName={}, nodeClass={}",
                dagInstDO.getId(),
                localTaskId, dagNodeName, nodeClass);

        localNode.dagInstId = dagInstDO.getId();
        localNode.dagInstNodeId = dagInstNode.getId();
        localNode.params = params;
        localNode.globalParams = dagInstDO.fetchGlobalParamsJson();
        localNode.lastGlobalVariableTimestamp = DateUtil.currentSeconds();
        localNode.globalVariable = dagInstDO.fetchGlobalVariableJson();
        localNode.globalResult = dagInstDO.fetchGlobalResultJson();
        localNode.nodeId = dagInstNode.getNodeId();
        localNode.fatherNodeId = dagInstDO.fetchRelationNodeId();
        localNode.retryTimes = Objects.isNull(dagInstNode.getRetryTimes()) ? 0 : dagInstNode.getRetryTimes();
        localNode.instanceCreateTimestamp = DateUtil.currentSeconds();
        localNode.realRunTimeout = Objects.isNull(dagNodeDO.getRunTimeout()) ? 0 : dagNodeDO.getRunTimeout();

        localTaskId2NodeMap.put(localTaskId, localNode);
        return localNode;
    }

    private class TimeoutHandler implements Runnable {

        @Override
        public void run() {
            try {
                List<Long> localTaskIdList = localTaskId2FutureMap.keySet().stream().collect(Collectors.toList());
                for (Long localTaskId : localTaskIdList) {
                    if (isTimeout(localTaskId)) {
                        log.info(">>>timeoutHandler|task is timeout|localTaskId={}", localTaskId);
                        dagInstNodeStdDAO.updateStatusWithDetail(localTaskId, TaskStatus.EXCEPTION, "timeout, killed " +
                                "by system");

                        Future future = localTaskId2FutureMap.remove(localTaskId);
                        if (Objects.nonNull(future)) {
                            future.cancel(true);
                        }

                        localTaskId2NodeMap.remove(localTaskId);
                    }
                }
            } catch (Exception e) {
                log.error(">>>timeoutHandler|scheduleAtFixedRate|Err={}", e.toString(), e);
            }
        }
    }

    private class ReBuilder implements Runnable {

        @Override
        public void run() {
            Long end = DateUtil.currentSeconds() - 30;
            Long begin = end - 60 * DateUtil.MINUTE;

            List<DagInstNodeStdDO> timeOutList = dagInstNodeStdDAO.listTimeOut(begin, end);

            if (CollectionUtils.isNotEmpty(timeOutList)) {
                for (DagInstNodeStdDO dagInstNodeStdDO : timeOutList) {
                    Long localTaskId = dagInstNodeStdDO.getId();
                    if (!localTaskId2FutureMap.containsKey(localTaskId)) {
                        DagInstNodeDO dagInstNodeDO = queryByLocalTaskId(localTaskId);
                        if (Objects.nonNull(dagInstNodeDO)) {
                            Long dagInstId = dagInstNodeDO.getDagInstId();
                            String nodeId = dagInstNodeDO.getNodeId();
                            log.info(">>>reBuilder|task is rebuild|dagInstId={}, nodeId={}, localTaskId={}",
                                    dagInstId, nodeId, localTaskId);
                            doLocalTask(LocalTaskDO.builder().dagInstId(dagInstId).nodeId(nodeId).taskId(localTaskId).build());
                        }
                    }
                }
            }
        }
    }

    private class LocalTask implements Runnable {
        private LocalTaskDO localTaskDO;

        public LocalTask(LocalTaskDO localTaskDO) {
            this.localTaskDO = localTaskDO;
        }

        @Override
        public void run() {
            log.info(">>>localTaskService|localTask run enter|localTaskDO={}", localTaskDO);
            Long dagInstId = localTaskDO.getDagInstId();
            String nodeId = localTaskDO.getNodeId();
            Long localTaskId = localTaskDO.getTaskId();
            try {
                DagInstNodeStdDO dagInstNodeStdDO = DagInstNodeStdDO.builder().id(localTaskId).build();
                DagInstNodeDO dagInstNode = dagInstNodeDAO.getDagInstNode(dagInstId, nodeId);

                AbstractLocalNodeBase localNode = newInstance(localTaskId, dagInstNode);
                localNode.actRunTimestamp = DateUtil.currentSeconds();
                dagInstNodeStdDO.setStatus(TaskStatus.RUNNING.toString());
                dagInstNodeStdDO.setIp(Tools.localIp);
                dagInstNodeStdDO.setInitGlobalParams(JSONObject.toJSONString(localNode.globalParams));

                dagInstNodeStdDAO.update(dagInstNodeStdDO);

                log.info(">>>localTask|run|enter|dagInstId={}, nodeId={}, localTaskId={}", dagInstId, nodeId,
                        localTaskId);
                DagInstNodeRunRet ret = localNode.run();
                dagInstNodeStdDO.setStatus(TaskStatus.SUCCESS.name());
                dagInstNodeStdDO.setStdout(JSONObject.toJSONString(ret));
                dagInstNodeStdDO.setGlobalParams(
                        localNode.isDeleteParams ? "__DEL__" : JSONObject.toJSONString(localNode.globalParams)
                );

                dagInstNodeStdDAO.update(dagInstNodeStdDO);
                log.info(">>>localTask|run|exit|dagInstId={}, nodeId={}, localTaskId={}", dagInstId, nodeId,
                        localTaskId);

                dagInstNodeNewService.jobCallBack(dagInstId, nodeId, localTaskId, TaskStatus.SUCCESS, "");
            } catch (Exception e) {
                log.error(">>>localTask|run|Err|dagInstId={}, nodeId={}, localTaskId={}, Err={}", dagInstId, nodeId,
                        localTaskId, e.toString(), e);
                dagInstNodeStdDAO.updateStatusWithDetail(localTaskId, TaskStatus.EXCEPTION,
                        Throwables.getStackTraceAsString(e));
                try {
                    dagInstNodeNewService.jobCallBack(dagInstId, nodeId, localTaskId, TaskStatus.EXCEPTION,
                            Throwables.getStackTraceAsString(e));
                } catch (Exception e1) {
                    log.error(">>>localTask|jobCallBack|Err|dagInstId={}, nodeId={}, localTaskId={}, Err={}",
                            dagInstId, nodeId, localTaskId, e1.toString(), e1);
                }
            } finally {
                localTaskId2FutureMap.remove(localTaskId);
                localTaskId2NodeMap.remove(localTaskId);
            }
        }
    }
}
