package com.alibaba.tesla.dag.services;

import java.util.List;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.tesla.dag.ApplicationProperties;
import com.alibaba.tesla.dag.common.Tools;
import com.alibaba.tesla.dag.model.domain.TcDag;
import com.alibaba.tesla.dag.model.domain.TcDagNode;
import com.alibaba.tesla.dag.model.domain.dag.DagInputParam;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeDetailApi;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeDetailFaas;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeDetailTask;
import com.alibaba.tesla.dag.model.domain.dagnode.DagNodeType;
import com.alibaba.tesla.dag.model.repository.TcDagNodeRepository;
import com.alibaba.tesla.dag.model.repository.TcDagRepository;

import com.google.common.base.Throwables;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import static java.lang.String.format;

/**
 * @author jinghua.yjh
 */
@Slf4j
@Service
public class RegisterService {

    @Autowired
    TcDagNodeRepository dagNodeRepository;

    @Autowired
    TcDagRepository dagRepository;

    @Autowired
    ApplicationProperties applicationProperties;

    public JSONObject dags(JSONArray dagJsonArray) {
        JSONObject metaJson = dags(dagJsonArray, 1, new JSONObject());
        dagRepository.deleteByNameNotInAndLastUpdateBy(
            metaJson.getJSONArray("name").toJavaList(String.class),
            metaJson.getString("updateBy")
        );
        return metaJson;
    }

    private JSONObject dags(JSONArray dagJsonArray, int index, JSONObject metaJson) {
        if (CollectionUtils.isEmpty(metaJson)) {
            metaJson = new JSONObject();
            metaJson.put("failed", new JSONArray());
            metaJson.put("name", new JSONArray());
            metaJson.put("detail", new StringBuilder());
        }
        StringBuilder sb = metaJson.getObject("detail", StringBuilder.class);
        JSONArray failed = metaJson.getJSONArray("failed");
        sb.append("\n##### NO.").append(index).append("#####");

        for (JSONObject dagJson : dagJsonArray.toJavaList(JSONObject.class)) {
            sb.append("\n").append(dagJson.getString("name")).append(": ");
            try {
                updateDag(dagJson, metaJson);
            } catch (Exception e) {
                sb.append(Throwables.getRootCause(e));
                log.error("updateDag: " + JSONObject.toJSONString(dagJson), e);
                failed.add(dagJson);
            }
        }

        if (failed.size() > 0 && failed.size() < dagJsonArray.size()) {
            dags(failed, index + 1, metaJson);
        }

        return metaJson;
    }

    public JSONObject nodes(List<JSONObject> nodeJsonList) throws Exception {
        JSONObject metaJson = new JSONObject();
        StringBuilder sb = new StringBuilder();
        JSONArray failed = new JSONArray();
        metaJson.put("failed", failed);
        metaJson.put("name", new JSONArray());
        metaJson.put("detail", sb);

        for (JSONObject nodeJson : nodeJsonList) {
            sb.append("\n").append(nodeJson.getString("name")).append(": ");
            updateNode(nodeJson, metaJson);
        }
        dagNodeRepository.deleteByNameNotInAndLastUpdateBy(
            metaJson.getJSONArray("name").toJavaList(String.class),
            metaJson.getString("updateBy")
        );
        return metaJson;
    }

    private void updateNode(JSONObject nodeJson, JSONObject metaJson) throws Exception {
        String appId = nodeJson.getString("app_id");
        String faasName = nodeJson.getString("faas_name");
        String moduleFile = nodeJson.getString("module_file");
        String updateBy = format("FAAS:%s:%s", appId, faasName);
        if (StringUtils.isNotEmpty(moduleFile)) {
            updateBy = updateBy + ":" + moduleFile;
        }
        String name = nodeJson.getString("name");
        String alias = nodeJson.getString("alias");
        StringBuilder sb = metaJson.getObject("detail", StringBuilder.class);

        metaJson.put("updateBy", updateBy);
        metaJson.getJSONArray("name").add(name);

        TcDagNode dagNode = dagNodeRepository.findFirstByAppIdAndName(appId, name);
        if (dagNode == null) {
            dagNode = TcDagNode.builder().gmtCreate(System.currentTimeMillis() / 1000).build();
        } else {
            if (StringUtils.isNotEmpty(dagNode.getLastUpdateBy()) && !dagNode.getLastUpdateBy().equals(updateBy)) {
                sb.append("conflict update by [").append(dagNode.getLastUpdateBy()).append("]");
                return;
            }
        }
        dagNode.setGmtModified(System.currentTimeMillis() / 1000);
        dagNode.setAppId(appId);
        dagNode.setName(name);
        dagNode.setAlias(StringUtils.isEmpty(alias) ? name : alias);
        dagNode.setDescription(nodeJson.getString("description"));
        dagNode.setIsShare(nodeJson.getInteger("is_share"));
        dagNode.setIsShow(nodeJson.getInteger("is_show"));
        dagNode.setInputParams(Tools.jsonObjectGetString(nodeJson, "input_params"));
        dagNode.setOutputParams(Tools.jsonObjectGetString(nodeJson, "output_params"));
        DagNodeType nodeType = nodeJson.getObject("type", DagNodeType.class);
        dagNode.setType(nodeType.name());
        switch (nodeType) {
            case API:
                dagNode.setDetail(DagNodeDetailApi.builder()
                    .timeout(nodeJson.getInteger("api_time_out"))
                    .urlPath(nodeJson.getString("api_url_path"))
                    .build().toJson());
                break;
            case FAAS:
                dagNode.setDetail(DagNodeDetailFaas.builder()
                    .appId(appId)
                    .faasName(faasName)
                    .name(nodeJson.getString("name"))
                    .callUrl(nodeJson.getString("call_url"))
                    .build().toJson());
                break;
            case TASK:
                dagNode.setDetail(DagNodeDetailTask.builder()
                    .appId(nodeJson.getString("task_app_id"))
                    .bizKey(nodeJson.getString("task_biz_key"))
                    .build().toJson());
                break;
            default:
                break;
        }

        dagNode.setFormatType(nodeJson.getString("format_type"));
        dagNode.setFormatDetail(Tools.jsonObjectGetString(nodeJson, "format_detail"));
        dagNode.setLastUpdateBy(updateBy);

        dagNode.upsertByAppIdAndName();
        dagNode.updateOptions();
        sb.append("OK");
    }

    private void updateDag(JSONObject dagJson, JSONObject metaJson) throws Exception {

        String appId = dagJson.getString("app_id");
        String faasName = dagJson.getString("faas_name");
        String moduleFile = dagJson.getString("module_file");
        String updateBy = format("FAAS:%s:%s", appId, faasName);
        if (StringUtils.isNotEmpty(moduleFile)) {
            updateBy = updateBy + ":" + moduleFile;
        }
        String name = dagJson.getString("name");
        String alias = dagJson.getString("alias");
        String exScheduleTaskId = dagJson.getString("ex_schedule_task_id");
        StringBuilder sb = metaJson.getObject("detail", StringBuilder.class);

        metaJson.put("updateBy", updateBy);
        metaJson.getJSONArray("name").add(name);

        TcDag dag = dagRepository.findFirstByAppIdAndName(appId, name);
        if (dag == null) {
            dag = TcDag.builder().gmtCreate(System.currentTimeMillis() / 1000).build();
        } else {
            if (StringUtils.isNotEmpty(dag.getLastUpdateBy()) && !dag.getLastUpdateBy().equals(updateBy)) {
                sb.append("conflict update by [").append(dag.getLastUpdateBy()).append("]");
                return;
            }
        }
        dag.setGmtModified(System.currentTimeMillis() / 1000);
        dag.setAppId(appId);
        dag.setName(name);
        dag.setAlias(StringUtils.isEmpty(alias) ? name : alias);
        dag.setExScheduleTaskId(exScheduleTaskId);
        dag.setContent(JSONObject.toJSONString(dagJson.getJSONObject("content")));
        dag.setLastUpdateBy(updateBy);
        dag.setHasFeedback(0);
        dag.setHasHistory(1);
        dag.setInputParams(dagJson.getJSONArray("input_params").toJavaList(DagInputParam.class));
        fillNodeInDag(dag);

        dag.upsertByAppIdAndName();
        dag.updateOptions();
        sb.append("OK");

    }

    private void fillNode(JSONObject nodeJson) throws Exception {
        JSONObject dataJson = nodeJson.getJSONObject("data");
        dataJson.put("tooltip", new JSONObject());
        if (dataJson.containsKey("defIdDetail")) {
            JSONObject defIdDetailJson = dataJson.getJSONObject("defIdDetail");
            String appId = defIdDetailJson.getString("app_id");
            String name = defIdDetailJson.getString("name");
            switch (dataJson.getString("type")) {
                case "NODE":
                    TcDagNode dagNode = dagNodeRepository.findFirstByAppIdAndName(appId, name);
                    if (dagNode == null) {
                        throw new Exception(format("can not find node by appId: %s and name: %s", appId, name));
                    }
                    dataJson.put("defId", dagNode.getId());
                    dataJson.put("label", dagNode.getAlias());
                    nodeJson.put("label", dagNode.getAlias());
                    dataJson.put("inputParams", dagNode.inputParamList());
                    dataJson.put("outputParams", dagNode.outputParamList());
                    break;
                case "DAG":
                    TcDag subDag = dagRepository.findFirstByAppIdAndName(appId, name);
                    if (subDag == null) {
                        throw new Exception(format("can not find subDag with appId: %s and name: %s", appId, name));
                    }
                    dataJson.put("defId", subDag.getId());
                    dataJson.put("label", subDag.getAlias());
                    nodeJson.put("label", subDag.getAlias());
                    dataJson.put("inputParams", subDag.inputParams());
                    break;
                default:
                    break;
            }
            dataJson.remove("defIdDetail");
        }
    }

    private void fillNodeInDag(TcDag dag) throws Exception {
        JSONObject contentJson = dag.contentJson();
        JSONArray nodeJsonArray = contentJson.getJSONArray("nodes");
        for (JSONObject nodeJson : nodeJsonArray.toJavaList(JSONObject.class)) {
            fillNode(nodeJson);
        }
        JSONObject preNodeJson = contentJson.getJSONObject("__pre_node__");
        if (!CollectionUtils.isEmpty(preNodeJson)) {
            fillNode(preNodeJson);
        }
        JSONObject postNodeJson = contentJson.getJSONObject("__post_node__");
        if (!CollectionUtils.isEmpty(postNodeJson)) {
            fillNode(postNodeJson);
        }
        dag.setContent(JSONObject.toJSONString(contentJson));
    }

    public int deleteNode(String appId, String name) {
        return dagNodeRepository.deleteByAppIdAndName(appId, name);
    }

    public int deleteDag(String appId, String name) {
        return dagRepository.deleteByAppIdAndName(appId, name);
    }
}
