package com.alibaba.tesla.common.base;


import com.alibaba.tesla.common.base.exception.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import static com.alibaba.tesla.common.base.constant.TeslaStatusCode.*;

import java.util.HashMap;
import java.util.Map;


/**
 * 用来生成Tesla result格式返回值的factory, 单例注入到Spring IoC容器中
 *
 * @author dongdong.ldd@alibaba-inc.com
 */
public class TeslaResultFactory {

    public static final String TESLA_CODE = "TESLA_CODE";

    public static TeslaBaseResult buildSucceedResult() {
        setTeslaCode(SUCCESS);
        return new TeslaBaseResult(StringUtils.EMPTY);
    }

    public static TeslaBaseResult buildSucceedResult(Object data) {
        TeslaBaseResult ret = buildSucceedResult();
        ret.setData(data);
        return ret;
    }

    public static TeslaBaseResult buildSucceedResult(Object data, String message) {
        TeslaBaseResult ret = buildSucceedResult();
        ret.setMessage(message);
        ret.setData(data);
        return ret;
    }

    public static TeslaBaseResult buildExceptionResult(TeslaBaseException e) {
        String errMessage = e.getErrMessage();
        if (e instanceof ServerException) {
            errMessage = String.format("后台服务错误：%s", errMessage);
        } else if (e instanceof ClientException) {
            if (e instanceof ClientUserArgsException) {
                errMessage = String.format("用户参数错误：%s", errMessage);
            } else if (e instanceof ClientUserPermException) {
                errMessage = String.format("无此操作的权限：%s", errMessage);
            } else if (e instanceof ClientForbiddenException) {
                errMessage = String.format("该操作被禁止：%s", errMessage);
            } else {
                errMessage = String.format("客户端错误：%s", errMessage);
            }
        } else {
            errMessage = String.format("服务异常：%s", errMessage);
        }
        setTeslaCode(e.getErrCode());
        return new TeslaBaseResult(e.getErrCode(), errMessage, e.getErrData());
    }

    /**
     * 构建参数错误返回数据
     *
     * @param data 参数错误对应关系，key=错误字段名，value=错误原因
     */
    public static TeslaBaseResult buildValidationErrorResult(Map<String, String> data) {
        setTeslaCode(VALIDATION_ERROR);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(USER_ARG_ERROR);
        ret.setMessage("Validation Error");
        ret.setData(data);
        return ret;
    }

    /**
     * 构建参数错误返回数据
     *
     * @param field        字段名称
     * @param errorMessage 错误信息
     */
    public static TeslaBaseResult buildValidationErrorResult(String field, String errorMessage) {
        setTeslaCode(VALIDATION_ERROR);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(USER_ARG_ERROR);
        ret.setMessage("Validation Error");
        Map<String, String> data = new HashMap<>(1);
        data.put(field, errorMessage);
        ret.setData(data);
        return ret;
    }


    /**
     * 构建客户端异常信息
     *
     * @param errorMessage 错误信息
     */
    public static TeslaBaseResult buildClientErrorResult(String errorMessage) {
        setTeslaCode(CLIENT_ERROR);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(CLIENT_ERROR);
        ret.setMessage(errorMessage);
        return ret;
    }

    /**
     * 构建错误返回结构
     *
     * @param data
     * @param msg
     * @return
     */
    public static TeslaBaseResult buildErrorResult(Object data, String msg) {
        setTeslaCode(SERVER_ERROR);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(SERVER_ERROR);
        ret.setMessage(msg);
        ret.setData(data);
        return ret;
    }


    /**
     * 构建找不到的错误返回数据
     */
    public static TeslaBaseResult buildNotFoundResult() {
        setTeslaCode(NOT_FOUND);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(NOT_FOUND);
        ret.setMessage("not found");
        return ret;
    }

    /**
     * 构建无权限的错误返回
     */
    public static TeslaBaseResult buildForbiddenResult(String message) {
        setTeslaCode(FORBIDDEN);
        TeslaBaseResult ret = new TeslaBaseResult();
        ret.setCode(FORBIDDEN);
        ret.setMessage(message);
        return ret;
    }

    /**
     * 构建无权限的错误返回
     */
    public static TeslaBaseResult buildForbiddenResult() {
        return buildForbiddenResult("forbidden");
    }

    public static TeslaBaseResult buildExceptionResult(Exception e) {
        setTeslaCode(SERVER_ERROR);
        TeslaBaseResult result = new TeslaBaseResult();
        result.setCode(SERVER_ERROR);
        result.setMessage(String.format("Server error： %s", e.toString()));
        result.setData(ExceptionUtils.getStackTrace(e));
        return result;
    }

    public static TeslaBaseResult buildResult(int code, String message, Object data) {
        setTeslaCode(code);
        return new TeslaBaseResult(code, message, data);
    }

    public static TeslaBaseResult buildResult(int code, String message) {
        setTeslaCode(code);
        return new TeslaBaseResult(code, message, null);
    }

    /**
     * 用于监控指标，在 request 中设置 TESLA_CODE attribute
     *
     * @param code 实际代码
     */
    private static void setTeslaCode(Integer code) {
        //todo
        //兼容mvcMvc和Flux
        //RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        //if (requestAttributes == null) {
        //    return;
        //}
        //HttpServletRequest request = ((ServletRequestAttributes)requestAttributes).getRequest();
        //if (request == null) {
        //    return;
        //}
        //request.setAttribute(TESLA_CODE, code);
    }
}
