/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.metrics;

import com.codahale.metrics.Clock;
import com.google.common.annotations.VisibleForTesting;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.agrona.concurrent.AtomicBuffer;
import org.agrona.concurrent.UnsafeBuffer;
import org.apache.cassandra.concurrent.TPC;
import org.apache.cassandra.concurrent.TPCUtils;
import org.apache.cassandra.io.compress.BufferType;
import org.apache.cassandra.metrics.Composable;
import org.apache.cassandra.metrics.Histogram;

public final class DecayingEstimatedHistogram
extends Histogram {
    static final String LEGACY_QUANTILE_FUNCTION_PROPERTY = "dse.metrics.legacy.quantile.function";
    private final ForwardDecayingReservoir reservoir;
    private final Recorder recorder;

    DecayingEstimatedHistogram(boolean considerZeroes, long maxTrackableValue, int updateTimeMillis, Clock clock) {
        BucketProperties bucketProperties = new BucketProperties(considerZeroes, maxTrackableValue);
        this.reservoir = new ForwardDecayingReservoir(bucketProperties, clock, considerZeroes, updateTimeMillis, false);
        this.recorder = new Recorder(bucketProperties, this.reservoir);
    }

    static Histogram.Reservoir makeCompositeReservoir(boolean considerZeroes, long maxTrackableValue, int updateIntervalMillis, Clock clock) {
        return new ForwardDecayingReservoir(new BucketProperties(considerZeroes, maxTrackableValue), clock, considerZeroes, updateIntervalMillis, true);
    }

    @Override
    public final void update(long value) {
        this.recorder.update(value);
    }

    public int size() {
        return this.reservoir.bucketProperties.size();
    }

    @Override
    public com.codahale.metrics.Snapshot getSnapshot() {
        return this.reservoir.getSnapshot();
    }

    @VisibleForTesting
    boolean isOverflowed() {
        return this.reservoir.isOverflowed();
    }

    @Override
    public void clear() {
        this.recorder.clear();
        this.reservoir.clear();
    }

    @Override
    public void aggregate() {
        this.reservoir.onReadAggregate();
    }

    @Override
    public boolean considerZeroes() {
        return this.reservoir.considerZeroes();
    }

    @Override
    public long maxTrackableValue() {
        return this.reservoir.maxTrackableValue();
    }

    @Override
    public long[] getOffsets() {
        return this.reservoir.getOffsets();
    }

    @Override
    public long getCount() {
        return this.reservoir.getCount();
    }

    @Override
    public Composable.Type getType() {
        return Composable.Type.SINGLE;
    }

    @VisibleForTesting
    public static class Snapshot
    extends com.codahale.metrics.Snapshot {
        private final long[] bucketOffsets;
        private final boolean isOverflowed;
        private final long[] buckets;
        private final long[] decayingBuckets;
        private final long count;
        private final long countWithDecay;
        private final long maxValue;
        private final boolean legacyQuantileFunction = Boolean.valueOf(System.getProperty("dse.metrics.legacy.quantile.function", "false"));

        public Snapshot(ForwardDecayingReservoir reservoir) {
            this.bucketOffsets = reservoir.getOffsets();
            this.isOverflowed = reservoir.isOverflowed;
            this.buckets = this.copyBuckets(reservoir.buckets, reservoir.considerZeroes);
            this.decayingBuckets = this.rescaleBuckets(this.copyBuckets(reservoir.decayingBuckets, reservoir.considerZeroes), reservoir.forwardDecayWeight());
            this.count = this.countBuckets(this.buckets);
            this.countWithDecay = this.countBuckets(this.decayingBuckets);
            this.maxValue = reservoir.maxTrackableValue();
        }

        private long[] copyBuckets(long[] buckets, boolean considerZeroes) {
            long[] ret = new long[this.bucketOffsets.length];
            int index = 0;
            if (considerZeroes) {
                ret[index++] = buckets[0];
                ret[index++] = buckets[1];
            } else {
                ret[index++] = buckets[0] + buckets[1];
            }
            for (int i = 2; i < buckets.length; ++i) {
                ret[index++] = buckets[i];
            }
            return ret;
        }

        private long[] rescaleBuckets(long[] buckets, double rescaleFactor) {
            int length = buckets.length;
            for (int i = 0; i < length; ++i) {
                buckets[i] = Math.round((double)buckets[i] / rescaleFactor);
            }
            return buckets;
        }

        private long countBuckets(long[] buckets) {
            long sum = 0L;
            for (int i = 0; i < buckets.length; ++i) {
                sum += buckets[i];
            }
            return sum;
        }

        long getCount() {
            return this.count;
        }

        public double getValue(double quantile) {
            assert (quantile >= 0.0 && quantile <= 1.0);
            if (this.isOverflowed) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            long qcount = (long)Math.ceil((double)this.countWithDecay * quantile);
            if (qcount == 0L) {
                return 0.0;
            }
            long elements = 0L;
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                if ((elements += this.decayingBuckets[i]) < qcount) continue;
                if (this.legacyQuantileFunction) {
                    return this.bucketOffsets[i];
                }
                double lowerBoundInclusive = this.bucketOffsets[i];
                double upperBoundExclusive = i == this.decayingBuckets.length - 1 ? (double)this.maxValue : (double)this.bucketOffsets[i + 1];
                double elementIdxInTheBucket = qcount - (elements - this.decayingBuckets[i]) - 1L;
                double numElementsInTheBucket = this.decayingBuckets[i];
                return lowerBoundInclusive + (upperBoundExclusive - lowerBoundInclusive) * elementIdxInTheBucket / numElementsInTheBucket;
            }
            return 0.0;
        }

        @VisibleForTesting
        public long[] getOffsets() {
            return this.bucketOffsets;
        }

        public long[] getValues() {
            return this.buckets;
        }

        public int size() {
            return this.decayingBuckets.length;
        }

        public long getMax() {
            if (this.isOverflowed) {
                return Long.MAX_VALUE;
            }
            for (int i = this.decayingBuckets.length - 1; i >= 0; --i) {
                if (this.decayingBuckets[i] <= 0L) continue;
                return this.bucketOffsets[i + 1] - 1L;
            }
            return 0L;
        }

        public double getMean() {
            if (this.isOverflowed) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            long elements = 0L;
            long sum = 0L;
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                long bCount = this.decayingBuckets[i];
                elements += bCount;
                long delta = this.bucketOffsets[i] - (i == 0 ? 0L : this.bucketOffsets[i - 1]);
                sum += bCount * (this.bucketOffsets[i] + (delta >> 1));
            }
            return (double)sum / (double)elements;
        }

        public long getMin() {
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                if (this.decayingBuckets[i] <= 0L) continue;
                return this.bucketOffsets[i];
            }
            return 0L;
        }

        public double getStdDev() {
            if (this.isOverflowed) {
                throw new IllegalStateException("Unable to compute when histogram overflowed");
            }
            if (this.countWithDecay <= 1L) {
                return 0.0;
            }
            double mean = this.getMean();
            double sum = 0.0;
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                long value = this.bucketOffsets[i];
                double diff = (double)value - mean;
                sum += diff * diff * (double)this.decayingBuckets[i];
            }
            return Math.sqrt(sum / (double)(this.countWithDecay - 1L));
        }

        public void dump(OutputStream output) {
            try (PrintWriter out = new PrintWriter(new OutputStreamWriter(output, StandardCharsets.UTF_8));){
                for (int i = 0; i < this.decayingBuckets.length; ++i) {
                    out.printf("%d%n", this.decayingBuckets[i]);
                }
            }
        }
    }

    private static final class Buffer {
        private final AtomicBuffer buffer;
        private final BucketProperties bucketProperties;
        private volatile boolean isOverflowed;

        Buffer(BucketProperties bucketProperties) {
            this.buffer = new UnsafeBuffer(BufferType.OFF_HEAP.allocate(bucketProperties.numBuckets << 3));
            this.bucketProperties = bucketProperties;
            this.isOverflowed = false;
        }

        void update(long value, boolean lazy) {
            if (value > this.bucketProperties.maxTrackableValue) {
                this.isOverflowed = true;
                return;
            }
            int index = this.bucketProperties.getIndex(value) << 3;
            if (lazy) {
                this.buffer.putLongOrdered(index, this.buffer.getLongVolatile(index) + 1L);
            } else {
                this.buffer.getAndAddLong(index, 1L);
            }
        }

        long getAndSet(int index, long value) {
            return this.buffer.getAndSetLong(index << 3, value);
        }

        long getLongVolatile(int index) {
            return this.buffer.getLongVolatile(index << 3);
        }

        void clear() {
            for (int i = 0; i < this.bucketProperties.size(); ++i) {
                this.getAndSet(i, 0L);
            }
        }
    }

    private static final class Recorder {
        private final BucketProperties bucketProperties;
        private final int numCores;
        private final Buffer[] buffers;
        private final ForwardDecayingReservoir reservoir;

        Recorder(BucketProperties bucketProperties, ForwardDecayingReservoir reservoir) {
            this.bucketProperties = bucketProperties;
            this.numCores = TPCUtils.getNumCores();
            this.buffers = new Buffer[this.numCores + 1];
            this.reservoir = reservoir;
            this.buffers[this.numCores] = new Buffer(bucketProperties);
            reservoir.add(this);
        }

        void update(long value) {
            this.reservoir.maybeSchedule();
            int coreId = TPCUtils.getCoreId();
            Buffer buffer = this.buffers[coreId];
            if (buffer == null) {
                assert (TPCUtils.isValidCoreId((int)coreId));
                buffer = this.buffers[coreId] = new Buffer(this.bucketProperties);
            }
            buffer.update(value, coreId < this.numCores);
        }

        boolean isOverFlowed() {
            for (Buffer buffer : this.buffers) {
                if (buffer == null || !buffer.isOverflowed) continue;
                return true;
            }
            return false;
        }

        long getValue(int index) {
            long ret = 0L;
            for (Buffer buffer : this.buffers) {
                if (buffer == null) continue;
                ret += buffer.getLongVolatile(index);
            }
            return ret;
        }

        void clear() {
            Arrays.stream(this.buffers).filter(Objects::nonNull).forEach(Buffer::clear);
        }
    }

    static final class ForwardDecayingReservoir
    implements Histogram.Reservoir {
        static final long HALF_TIME_IN_S = 60L;
        static final double MEAN_LIFETIME_IN_S = 60.0 / Math.log(2.0);
        static final long LANDMARK_RESET_INTERVAL_IN_MS = 1800000L;
        private final BucketProperties bucketProperties;
        private final Clock clock;
        private final boolean considerZeroes;
        private final int updateIntervalMillis;
        private long decayLandmark;
        private final CopyOnWriteArrayList<Recorder> recorders;
        private final long[] buckets;
        private final long[] decayingBuckets;
        private volatile Snapshot snapshot;
        private final boolean isComposite;
        private final AtomicBoolean scheduled;
        private volatile boolean isOverflowed;

        ForwardDecayingReservoir(BucketProperties bucketProperties, Clock clock, boolean considerZeroes, int updateIntervalMillis, boolean isComposite) {
            this.bucketProperties = bucketProperties;
            this.clock = clock;
            this.considerZeroes = considerZeroes;
            this.updateIntervalMillis = updateIntervalMillis;
            this.isOverflowed = false;
            this.buckets = new long[bucketProperties.numBuckets];
            this.decayingBuckets = new long[bucketProperties.numBuckets];
            this.recorders = new CopyOnWriteArrayList();
            this.decayLandmark = clock.getTime();
            this.snapshot = new Snapshot(this);
            this.scheduled = new AtomicBoolean(false);
            this.isComposite = isComposite;
            this.scheduleIfComposite();
        }

        void add(Recorder recorder) {
            this.recorders.add(recorder);
        }

        void maybeSchedule() {
            if (this.updateIntervalMillis <= 0 || this.scheduled.get()) {
                return;
            }
            if (this.scheduled.compareAndSet(false, true)) {
                this.schedule();
            }
        }

        private void schedule() {
            if (TPC.DEBUG_DONT_SCHEDULE_METRICS) {
                return;
            }
            TPC.bestTPCTimer().onTimeout(this::aggregate, (long)this.updateIntervalMillis, TimeUnit.MILLISECONDS);
        }

        void scheduleIfComposite() {
            if (this.updateIntervalMillis > 0 && this.isComposite && this.scheduled.compareAndSet(false, true)) {
                this.schedule();
            }
        }

        @Override
        public boolean considerZeroes() {
            return this.considerZeroes;
        }

        @Override
        public long maxTrackableValue() {
            return this.bucketProperties.maxTrackableValue;
        }

        void onReadAggregate() {
            if (this.updateIntervalMillis <= 0) {
                this.aggregate();
            } else {
                this.maybeSchedule();
            }
        }

        boolean isOverflowed() {
            this.onReadAggregate();
            return this.isOverflowed;
        }

        @Override
        public long getCount() {
            this.onReadAggregate();
            return this.snapshot.getCount();
        }

        @Override
        public Snapshot getSnapshot() {
            this.onReadAggregate();
            return this.snapshot;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @VisibleForTesting
        void aggregate() {
            assert (this.scheduled.get() || this.updateIntervalMillis <= 0) : "aggregation running without mutual exclusion guarantee and without synchronization on client side";
            try {
                long now = this.clock.getTime();
                this.rescaleIfNeeded(now);
                long weight = Math.round(ForwardDecayingReservoir.forwardDecayWeight(now, this.decayLandmark));
                this.isOverflowed = false;
                for (Recorder recorder : this.recorders) {
                    if (!recorder.isOverFlowed()) continue;
                    this.isOverflowed = true;
                    break;
                }
                for (int i = 0; i < this.buckets.length; ++i) {
                    long value = this.getRecordersSum(i);
                    long delta = value - this.buckets[i];
                    if (delta <= 0L) continue;
                    int n = i;
                    this.buckets[n] = this.buckets[n] + delta;
                    int n2 = i;
                    this.decayingBuckets[n2] = this.decayingBuckets[n2] + delta * weight;
                }
                this.snapshot = new Snapshot(this);
            }
            finally {
                this.scheduled.set(false);
                this.scheduleIfComposite();
            }
        }

        private long getRecordersSum(int index) {
            long ret = 0L;
            for (Recorder recorder : this.recorders) {
                ret += recorder.getValue(index);
            }
            return ret;
        }

        private boolean isCompatible(Histogram.Reservoir other) {
            if (!(other instanceof ForwardDecayingReservoir)) {
                return false;
            }
            ForwardDecayingReservoir otherFDReservoir = (ForwardDecayingReservoir)other;
            return otherFDReservoir.considerZeroes == this.considerZeroes && otherFDReservoir.buckets.length == this.buckets.length && otherFDReservoir.decayingBuckets.length == this.decayingBuckets.length;
        }

        @Override
        public void add(Histogram histogram) {
            if (!(histogram instanceof DecayingEstimatedHistogram)) {
                throw new IllegalArgumentException("Histogram is not compatible");
            }
            DecayingEstimatedHistogram decayingEstimatedHistogram = (DecayingEstimatedHistogram)histogram;
            if (!this.isCompatible(decayingEstimatedHistogram.reservoir)) {
                throw new IllegalArgumentException("Histogram reservoir is not compatible");
            }
            this.add(decayingEstimatedHistogram.recorder);
        }

        @Override
        public long[] getOffsets() {
            return this.bucketProperties.offsets;
        }

        void clear() {
            this.isOverflowed = false;
            for (int i = 0; i < this.bucketProperties.size(); ++i) {
                this.buckets[i] = 0L;
                this.decayingBuckets[i] = 0L;
            }
            this.snapshot = new Snapshot(this);
        }

        private void rescaleIfNeeded(long now) {
            if (this.needRescale(now)) {
                this.rescale(now);
            }
        }

        private void rescale(long now) {
            double rescaleFactor = ForwardDecayingReservoir.forwardDecayWeight(now, this.decayLandmark);
            this.decayLandmark = now;
            for (int i = 0; i < this.decayingBuckets.length; ++i) {
                this.decayingBuckets[i] = Math.round((double)this.decayingBuckets[i] / rescaleFactor);
            }
        }

        private boolean needRescale(long now) {
            return now - this.decayLandmark > 1800000L;
        }

        private double forwardDecayWeight() {
            return ForwardDecayingReservoir.forwardDecayWeight(this.clock.getTime(), this.decayLandmark);
        }

        private static double forwardDecayWeight(long now, long decayLandmark) {
            return Math.exp((double)(now - decayLandmark) / 1000.0 / MEAN_LIFETIME_IN_S);
        }
    }

    static final class BucketProperties {
        final boolean considerZeros;
        final long maxTrackableValue;
        final long[] offsets;
        final int numBuckets;
        static final int subBucketCount = 8;
        static final int subBucketHalfCount = 4;
        static final int unitMagnitude = 0;
        static final int subBucketCountMagnitude = 3;
        static final int subBucketHalfCountMagnitude = 2;
        final long subBucketMask;
        final int leadingZeroCountBase;

        BucketProperties(boolean considerZeros, long maxTrackableValue) {
            this.considerZeros = considerZeros;
            this.maxTrackableValue = maxTrackableValue;
            this.offsets = this.makeOffsets(considerZeros);
            this.numBuckets = this.offsets.length + (!considerZeros ? 1 : 0);
            this.subBucketMask = 7L;
            this.leadingZeroCountBase = 61;
        }

        private long[] makeOffsets(boolean considerZeroes) {
            ArrayList<Long> ret = new ArrayList<Long>();
            if (considerZeroes) {
                ret.add(0L);
            }
            for (int i2 = 1; i2 <= 8; ++i2) {
                ret.add(Long.valueOf(i2));
                if ((long)i2 >= this.maxTrackableValue) break;
            }
            long last = 8L;
            long unit = 2L;
            while (last < this.maxTrackableValue) {
                for (int i3 = 0; i3 < 4; ++i3) {
                    ret.add(last += unit);
                    if (last >= this.maxTrackableValue) break;
                }
                unit *= 2L;
            }
            return ret.stream().mapToLong(i -> i).toArray();
        }

        final int getIndex(long value) {
            if (value < 0L) {
                throw new ArrayIndexOutOfBoundsException("Histogram recorded value cannot be negative.");
            }
            int bucketIndex = this.leadingZeroCountBase - Long.numberOfLeadingZeros(value | this.subBucketMask);
            int subBucketIndex = (int)(value >>> bucketIndex + 0);
            int bucketBaseIndex = bucketIndex + 1 << 2;
            int offsetInBucket = subBucketIndex - 4;
            return bucketBaseIndex + offsetInBucket;
        }

        private int size() {
            return this.numBuckets;
        }
    }
}

