/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.metrics;

import com.google.common.collect.ImmutableMap;
import java.net.InetAddress;
import java.util.Comparator;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.HdrHistogram.Histogram;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.metrics.CassandraMetricsRegistry;
import org.apache.cassandra.metrics.DefaultNameFactory;
import org.apache.cassandra.metrics.Meter;
import org.apache.cassandra.metrics.Timer;
import org.apache.cassandra.net.DroppedMessages;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;

public class DroppedMessageMetrics {
    private static final int NUM_TOP_ELEMENTS = 3;
    private final DroppedMessages.Group droppedGroup;
    public final Meter dropped;
    public final Timer internalDroppedLatency;
    public final Timer crossNodeDroppedLatency;
    private volatile boolean isDroppedMessagesDataAvailable;
    private final ConcurrentMap<InetAddress, AtomicInteger> droppedByPeer;
    private final ConcurrentMap<String, AtomicInteger> droppedByKeyspaceTable;
    private volatile Histogram internalDroppedSizes;
    private volatile Histogram crossNodeDroppedSizes;

    public DroppedMessageMetrics(DroppedMessages.Group group) {
        this.droppedGroup = group;
        DefaultNameFactory factory = new DefaultNameFactory("DroppedMessage", group.toString());
        this.dropped = CassandraMetricsRegistry.Metrics.meter(factory.createMetricName("Dropped"));
        this.internalDroppedLatency = CassandraMetricsRegistry.Metrics.timer(factory.createMetricName("InternalDroppedLatency"));
        this.crossNodeDroppedLatency = CassandraMetricsRegistry.Metrics.timer(factory.createMetricName("CrossNodeDroppedLatency"));
        this.droppedByPeer = new ConcurrentHashMap<InetAddress, AtomicInteger>();
        this.droppedByKeyspaceTable = new ConcurrentHashMap<String, AtomicInteger>();
        this.internalDroppedSizes = DroppedMessageMetrics.newHistogram();
        this.crossNodeDroppedSizes = DroppedMessageMetrics.newHistogram();
    }

    public void onMessageDropped(Message<?> message) {
        assert (message != null);
        if (!this.isDroppedMessagesDataAvailable) {
            this.isDroppedMessagesDataAvailable = true;
        }
        this.dropped.mark();
        long timeTakenMillis = message.lifetimeMillis();
        if (message.isLocal()) {
            this.internalDroppedLatency.update(timeTakenMillis, TimeUnit.MILLISECONDS);
            this.internalDroppedSizes.recordValue(Math.max(0L, message.payloadSerializedSizeForThisVersion()));
            DroppedMessageMetrics.recordDroppedMessageWithKey(message.to(), this.droppedByPeer);
        } else {
            this.crossNodeDroppedLatency.update(timeTakenMillis, TimeUnit.MILLISECONDS);
            InetAddress peer = DroppedMessageMetrics.isCurrentNodeAddress(message.from()) ? message.to() : message.from();
            this.crossNodeDroppedSizes.recordValue(Math.max(0L, message.payloadSerializedSizeForThisVersion()));
            DroppedMessageMetrics.recordDroppedMessageWithKey(peer, this.droppedByPeer);
        }
        DroppedMessageMetrics.recordDroppedMessageKeyspaceTable(message, this.droppedByKeyspaceTable);
    }

    public DroppedMessages.DroppedMessageGroupStats getAndResetStats() {
        if (!this.isDroppedMessagesDataAvailable) {
            return null;
        }
        long internalLatencyMs = TimeUnit.NANOSECONDS.toMillis((long)this.internalDroppedLatency.getSnapshot().getMean());
        long crossNodeLatencyMs = TimeUnit.NANOSECONDS.toMillis((long)this.crossNodeDroppedLatency.getSnapshot().getMean());
        ImmutableMap droppedByPeerCopy = ImmutableMap.copyOf(this.droppedByPeer);
        ImmutableMap droppedByKeyspaceTableCopy = ImmutableMap.copyOf(this.droppedByKeyspaceTable);
        Object[] internalPercentiles = DroppedMessageMetrics.getHistogramPercentilesRow(String.format("%s (internal)", this.droppedGroup), this.internalDroppedSizes);
        Object[] crossNodePercentiles = DroppedMessageMetrics.getHistogramPercentilesRow(String.format("%s (cross-node)", this.droppedGroup), this.crossNodeDroppedSizes);
        this.resetInternalState();
        this.isDroppedMessagesDataAvailable = false;
        int internalDroppedCount = 0;
        int crossNodeDroppedCount = 0;
        PriorityQueue peersHeap = new PriorityQueue(Comparator.comparing(p -> (Integer)p.right));
        for (Map.Entry entry : droppedByPeerCopy.entrySet()) {
            InetAddress nodeAddress = (InetAddress)entry.getKey();
            int currCount = ((AtomicInteger)entry.getValue()).get();
            if (DroppedMessageMetrics.isCurrentNodeAddress(nodeAddress)) {
                internalDroppedCount += currCount;
            } else {
                crossNodeDroppedCount += currCount;
            }
            DroppedMessageMetrics.updatePairHeap(peersHeap, nodeAddress, currCount);
        }
        PriorityQueue keyspaceTablesHeap = new PriorityQueue(Comparator.comparing(p -> (Integer)p.right));
        for (Map.Entry entry : droppedByKeyspaceTableCopy.entrySet()) {
            String keyspaceTable = (String)entry.getKey();
            int currCount = ((AtomicInteger)entry.getValue()).get();
            DroppedMessageMetrics.updatePairHeap(keyspaceTablesHeap, keyspaceTable, currCount);
        }
        return new DroppedMessages.DroppedMessageGroupStats(this.droppedGroup, internalDroppedCount, crossNodeDroppedCount, internalLatencyMs, crossNodeLatencyMs, internalPercentiles, crossNodePercentiles, peersHeap, keyspaceTablesHeap);
    }

    private void resetInternalState() {
        this.droppedByPeer.clear();
        this.droppedByKeyspaceTable.clear();
        this.internalDroppedSizes = DroppedMessageMetrics.newHistogram();
        this.crossNodeDroppedSizes = DroppedMessageMetrics.newHistogram();
    }

    private static Object[] getHistogramPercentilesRow(String rowPrefix, Histogram histogram) {
        return new Object[]{rowPrefix, histogram.getValueAtPercentile(50.0), histogram.getValueAtPercentile(75.0), histogram.getValueAtPercentile(95.0), histogram.getValueAtPercentile(98.0), histogram.getValueAtPercentile(99.0), histogram.getMinValue(), histogram.getMaxValue()};
    }

    private static Histogram newHistogram() {
        return new Histogram(Integer.MAX_VALUE, 3);
    }

    private static <K> void recordDroppedMessageWithKey(K key, ConcurrentMap<K, AtomicInteger> droppedBy) {
        AtomicInteger dropped = (AtomicInteger)droppedBy.get(key);
        if (dropped == null) {
            AtomicInteger newDropped = new AtomicInteger();
            AtomicInteger existingDropped = droppedBy.putIfAbsent(key, newDropped);
            dropped = existingDropped != null ? existingDropped : newDropped;
        }
        dropped.incrementAndGet();
    }

    private static void recordDroppedMessageKeyspaceTable(Message<?> message, ConcurrentMap<String, AtomicInteger> droppedByKeyspaceTable) {
        Object payload = message.payload();
        if (payload instanceof ReadCommand) {
            ReadCommand readCommand = (ReadCommand)payload;
            TableMetadata tableMetadata = readCommand.metadata();
            DroppedMessageMetrics.recordDroppedMessageWithKey(DroppedMessageMetrics.getKeyspaceTableString(tableMetadata), droppedByKeyspaceTable);
        }
        if (payload instanceof Mutation) {
            Mutation mutation = (Mutation)payload;
            boolean recorded = false;
            for (PartitionUpdate update : mutation.getPartitionUpdates()) {
                if (recorded) break;
                recorded = true;
                TableMetadata tableMetadata = update.metadata();
                DroppedMessageMetrics.recordDroppedMessageWithKey(DroppedMessageMetrics.getKeyspaceTableString(tableMetadata), droppedByKeyspaceTable);
            }
        }
    }

    private static <K> void updatePairHeap(PriorityQueue<Pair<K, Integer>> heap, K currKey, int currCount) {
        if (heap.size() < 3 || (Integer)heap.peek().right < currCount) {
            heap.add(Pair.create(currKey, (Object)currCount));
        }
        if (heap.size() > 3) {
            heap.poll();
        }
    }

    private static boolean isCurrentNodeAddress(InetAddress address) {
        return FBUtilities.getBroadcastAddress().equals(address) || FBUtilities.getLocalAddress().equals(address);
    }

    private static String getKeyspaceTableString(TableMetadata tableMetadata) {
        return String.format("%s.%s", tableMetadata.keyspace, tableMetadata.name);
    }
}

