package io.streamlayer.sdk.twitter.view.internal.media;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import io.streamlayer.sdk.model.social.Tweet;
import io.streamlayer.sdk.model.social.TweetExtendedEntities;
import io.streamlayer.sdk.model.social.TweetMedia;
import io.streamlayer.sdk.model.social.TweetMediaVideoVariant;

public final class TweetMediaUtils {
    public static final String PHOTO_TYPE = "photo";
    public static final String VIDEO_TYPE = "video";
    public static final String GIF_TYPE = "animated_gif";
    private static final String CONTENT_TYPE_MP4 = "video/mp4";
    private static final String CONTENT_TYPE_HLS = "application/x-mpegURL";
    private static final int LOOP_VIDEO_IN_MILLIS = 6500;
    private static final String TIME_FORMAT_LONG = "%1$d:%2$02d:%3$02d";
    private static final String TIME_FORMAT_SHORT = "%1$d:%2$02d";

    private TweetMediaUtils() {
    }

    public static String getPlaybackTime(long timeMillis) {
        final int timeSeconds = (int) (timeMillis / 1000);
        final int seconds = timeSeconds % 60;
        final int minutes = (timeSeconds / 60) % 60;
        final int hours = timeSeconds / 3600;

        if (hours > 0) {
            return String.format(Locale.getDefault(), TIME_FORMAT_LONG, hours, minutes, seconds);
        } else {
            return String.format(Locale.getDefault(), TIME_FORMAT_SHORT, minutes, seconds);
        }
    }

    /**
     * This method gets the last photo entity out of the tweet, this is the photo to display inline
     *
     * @param tweet The Tweet
     * @return The last photo entity of Tweet
     */
    public static TweetMedia getPhotoEntity(Tweet tweet) {
        final List<TweetMedia> mediaEntityList = getAllMediaEntities(tweet);
        for (int i = mediaEntityList.size() - 1; i >= 0; i--) {
            final TweetMedia entity = mediaEntityList.get(i);
            if (entity.getType() != null && isPhotoType(entity)) {
                return entity;
            }
        }
        return null;
    }

    /**
     * This method gets the all the photos from the tweet, which are used to display inline
     *
     * @param tweet The Tweet
     * @return Photo entities of Tweet
     */
    public static List<TweetMedia> getPhotoEntities(Tweet tweet) {
        final List<TweetMedia> photoEntities = new ArrayList<>();
        final TweetExtendedEntities extendedEntities = tweet.getMetaData().getExtendedEntities();

        if (extendedEntities != null && extendedEntities.getMedia() != null
                && extendedEntities.getMedia().size() > 0) {
            for (int i = 0; i <= extendedEntities.getMedia().size() - 1; i++) {
                final TweetMedia entity = extendedEntities.getMedia().get(i);
                if (entity.getType() != null && isPhotoType(entity)) {
                    photoEntities.add(entity);
                }
            }
            return photoEntities;
        }

        return photoEntities;
    }

    /**
     * Returns true if there is a media entity with the type of "photo"
     *
     * @param tweet The Tweet entities
     * @return true if there is a media entity with the type of "photo"
     */
    public static boolean hasPhoto(Tweet tweet) {
        return getPhotoEntity(tweet) != null;
    }

    /**
     * This method gets the first video or animated gif entity out of the tweet, this is the video
     * to display inline
     *
     * @param tweet The Tweet
     * @return The last photo entity of Tweet
     */
    public static TweetMedia getVideoEntity(Tweet tweet) {
        for (TweetMedia mediaEntity : getAllMediaEntities(tweet)) {
            if (mediaEntity.getType() != null && isVideoType(mediaEntity)) {
                return mediaEntity;
            }
        }

        return null;
    }

    public static boolean isHlsVideo(TweetMediaVideoVariant variant) {
        return CONTENT_TYPE_HLS.equals(variant.getContentType());
    }

    public static boolean isPhotoType(TweetMedia mediaEntity) {
        return PHOTO_TYPE.equals(mediaEntity.getType());
    }

    public static boolean isVideoType(TweetMedia mediaEntity) {
        return VIDEO_TYPE.equals(mediaEntity.getType()) || GIF_TYPE.equals(mediaEntity.getType());
    }

    public static TweetMediaVideoVariant getSupportedVariant(TweetMedia mediaEntity) {
        for (TweetMediaVideoVariant variant : mediaEntity.getVideoInfo().getVariants()) {
            if (isVariantSupported(variant)) {
                return variant;
            }
        }
        return null;
    }

    public static boolean isLooping(TweetMedia mediaEntity) {
        return GIF_TYPE.equals(mediaEntity.getType()) ||
                VIDEO_TYPE.endsWith(mediaEntity.getType()) &&
                        mediaEntity.getVideoInfo().getDurationMillis() < LOOP_VIDEO_IN_MILLIS;
    }

    public static boolean showVideoControls(TweetMedia mediaEntity) {
        return !GIF_TYPE.equals(mediaEntity.getType());
    }

    public static boolean isVariantSupported(TweetMediaVideoVariant variant) {
        if (CONTENT_TYPE_HLS.equals(variant.getContentType())) {
            return true;
        } else return CONTENT_TYPE_MP4.equals(variant.getContentType());
    }

    public static List<TweetMedia> getAllMediaEntities(Tweet tweet) {
        final List<TweetMedia> entities = new ArrayList<>();
        if (tweet.getMetaData() != null && tweet.getMetaData().getEntities() != null) {
            entities.addAll(tweet.getMetaData().getEntities().getMedia());
        }

        if (tweet.getMetaData() != null && tweet.getMetaData().getExtendedEntities() != null) {
            entities.addAll(tweet.getMetaData().getExtendedEntities().getMedia());
        }

        return entities;
    }
}
