/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.resources.NamespaceResources;
import org.apache.pulsar.broker.service.Producer;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.shade.com.google.common.collect.Lists;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.Position;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.impl.ManagedLedgerImpl;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.impl.PositionImpl;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.proto.MLDataFormats;
import org.apache.pulsar.shade.org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.BacklogQuota;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.impl.BacklogQuotaImpl;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.FutureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BacklogQuotaManager {
    private static final Logger log = LoggerFactory.getLogger(BacklogQuotaManager.class);
    private final BacklogQuotaImpl defaultQuota;
    private final NamespaceResources namespaceResources;

    public BacklogQuotaManager(PulsarService pulsar) {
        double backlogQuotaGB = pulsar.getConfiguration().getBacklogQuotaDefaultLimitGB();
        this.defaultQuota = BacklogQuotaImpl.builder().limitSize(backlogQuotaGB > 0.0 ? (long)(backlogQuotaGB * 1.073741824E9) : pulsar.getConfiguration().getBacklogQuotaDefaultLimitBytes()).limitTime(pulsar.getConfiguration().getBacklogQuotaDefaultLimitSecond()).retentionPolicy(pulsar.getConfiguration().getBacklogQuotaDefaultRetentionPolicy()).build();
        this.namespaceResources = pulsar.getPulsarResources().getNamespaceResources();
    }

    public BacklogQuotaImpl getDefaultQuota() {
        return this.defaultQuota;
    }

    public BacklogQuotaImpl getBacklogQuota(NamespaceName namespace, BacklogQuota.BacklogQuotaType backlogQuotaType) {
        try {
            if (this.namespaceResources == null) {
                log.warn("Failed to read policies data from metadata store because namespaceResources is null.default backlog quota will be applied: namespace={}", (Object)namespace);
                return this.defaultQuota;
            }
            return this.namespaceResources.getPolicies(namespace).map(p -> (BacklogQuotaImpl)p.backlog_quota_map.getOrDefault((Object)backlogQuotaType, this.defaultQuota)).orElse(this.defaultQuota);
        }
        catch (MetadataStoreException e) {
            log.warn("Failed to read policies data from metadata store, will apply the default backlog quota: namespace={}", (Object)namespace, (Object)e);
            return this.defaultQuota;
        }
    }

    public void handleExceededBacklogQuota(PersistentTopic persistentTopic, BacklogQuota.BacklogQuotaType backlogQuotaType, boolean preciseTimeBasedBacklogQuotaCheck) {
        BacklogQuota quota = persistentTopic.getBacklogQuota(backlogQuotaType);
        log.info("Backlog quota type {} exceeded for topic [{}]. Applying [{}] policy", new Object[]{backlogQuotaType, persistentTopic.getName(), quota.getPolicy()});
        block0 : switch (quota.getPolicy()) {
            case consumer_backlog_eviction: {
                switch (backlogQuotaType) {
                    case destination_storage: {
                        this.dropBacklogForSizeLimit(persistentTopic, quota);
                        break block0;
                    }
                    case message_age: {
                        this.dropBacklogForTimeLimit(persistentTopic, quota, preciseTimeBasedBacklogQuotaCheck);
                        break block0;
                    }
                }
                break;
            }
            case producer_exception: 
            case producer_request_hold: {
                this.disconnectProducers(persistentTopic);
                break;
            }
        }
    }

    private void dropBacklogForSizeLimit(PersistentTopic persistentTopic, BacklogQuota quota) {
        double reductionFactor = 0.9;
        double targetSize = reductionFactor * (double)quota.getLimitSize();
        ManagedLedgerImpl mLedger = (ManagedLedgerImpl)persistentTopic.getManagedLedger();
        long backlogSize = mLedger.getEstimatedBacklogSize();
        if (log.isDebugEnabled()) {
            log.debug("[{}] target size is [{}] for quota limit [{}], backlog size is [{}]", new Object[]{persistentTopic.getName(), targetSize, targetSize / reductionFactor, backlogSize});
        }
        ManagedCursor previousSlowestConsumer = null;
        while ((double)backlogSize > targetSize) {
            ManagedCursor slowestConsumer = mLedger.getSlowestConsumer();
            if (slowestConsumer == null) {
                if (!log.isDebugEnabled()) break;
                log.debug("[{}] slowest consumer null.", (Object)persistentTopic.getName());
                break;
            }
            double messageSkipFactor = ((double)backlogSize - targetSize) / (double)backlogSize;
            if (slowestConsumer == previousSlowestConsumer) {
                log.info("[{}] Cursors not progressing, target size is [{}] for quota limit [{}], backlog size is [{}]", new Object[]{persistentTopic.getName(), targetSize, targetSize / reductionFactor, backlogSize});
                break;
            }
            long entriesInBacklog = slowestConsumer.getNumberOfEntriesInBacklog(false);
            int messagesToSkip = (int)(messageSkipFactor * (double)entriesInBacklog);
            try {
                if (messagesToSkip == 0) {
                    if (!log.isDebugEnabled()) break;
                    log.debug("no messages to skip for [{}]", (Object)slowestConsumer);
                    break;
                }
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Skipping [{}] messages on slowest consumer [{}] having backlog entries : [{}]", new Object[]{persistentTopic.getName(), messagesToSkip, slowestConsumer.getName(), entriesInBacklog});
                }
                slowestConsumer.skipEntries(messagesToSkip, ManagedCursor.IndividualDeletedEntries.Include);
            }
            catch (Exception e) {
                log.error("[{}] Error skipping [{}] messages from slowest consumer [{}]", new Object[]{persistentTopic.getName(), messagesToSkip, slowestConsumer.getName(), e});
            }
            backlogSize = mLedger.getEstimatedBacklogSize();
            previousSlowestConsumer = slowestConsumer;
            if (!log.isDebugEnabled()) continue;
            log.debug("[{}] Updated unconsumed size = [{}]. skipFactor: [{}]", new Object[]{persistentTopic.getName(), backlogSize, messageSkipFactor});
        }
    }

    private void dropBacklogForTimeLimit(PersistentTopic persistentTopic, BacklogQuota quota, boolean preciseTimeBasedBacklogQuotaCheck) {
        if (preciseTimeBasedBacklogQuotaCheck) {
            double reductionFactor = 0.9;
            int target = (int)(reductionFactor * (double)quota.getLimitTime());
            if (log.isDebugEnabled()) {
                log.debug("[{}] target backlog expire time is [{}]", (Object)persistentTopic.getName(), (Object)target);
            }
            persistentTopic.getSubscriptions().forEach((__, subscription) -> subscription.getExpiryMonitor().expireMessages(target));
        } else {
            Long currentMillis = ((ManagedLedgerImpl)persistentTopic.getManagedLedger()).getClock().millis();
            ManagedLedgerImpl mLedger = (ManagedLedgerImpl)persistentTopic.getManagedLedger();
            try {
                PositionImpl nextPosition;
                ManagedCursor slowestConsumer;
                Position oldestPosition;
                MLDataFormats.ManagedLedgerInfo.LedgerInfo ledgerInfo;
                while ((ledgerInfo = mLedger.getLedgerInfo((oldestPosition = (slowestConsumer = mLedger.getSlowestConsumer()).getMarkDeletedPosition()).getLedgerId()).get()).getTimestamp() > 0L && currentMillis - ledgerInfo.getTimestamp() > (long)quota.getLimitTime() && !(nextPosition = mLedger.getNextValidPosition(PositionImpl.get(ledgerInfo.getLedgerId(), ledgerInfo.getEntries() - 1L))).equals(oldestPosition)) {
                    slowestConsumer.resetCursor(nextPosition);
                }
            }
            catch (Exception e) {
                log.error("[{}] Error resetting cursor for slowest consumer [{}]", new Object[]{persistentTopic.getName(), mLedger.getSlowestConsumer().getName(), e});
            }
        }
    }

    private void disconnectProducers(PersistentTopic persistentTopic) {
        ArrayList futures = Lists.newArrayList();
        Map<String, Producer> producers = persistentTopic.getProducers();
        producers.values().forEach(producer -> {
            log.info("Producer [{}] has exceeded backlog quota on topic [{}]. Disconnecting producer", (Object)producer.getProducerName(), (Object)persistentTopic.getName());
            futures.add(producer.disconnect());
        });
        ((CompletableFuture)FutureUtil.waitForAll(futures).thenRun(() -> log.info("All producers on topic [{}] are disconnected", (Object)persistentTopic.getName()))).exceptionally(exception -> {
            log.error("Error in disconnecting producers on topic [{}] [{}]", (Object)persistentTopic.getName(), exception);
            return null;
        });
    }
}

