/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.shade.org.apache.pulsar.common.util;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.FutureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GracefulExecutorServicesTerminationHandler {
    private static final Logger log = LoggerFactory.getLogger(GracefulExecutorServicesTerminationHandler.class);
    private static final long SHUTDOWN_THREAD_COMPLETION_TIMEOUT_NANOS = Duration.ofMillis(100L).toNanos();
    private final List<ExecutorService> executors;
    private final CompletableFuture<Void> future;
    private final Duration shutdownTimeout;
    private final Duration terminationTimeout;
    private final CountDownLatch shutdownThreadCompletedLatch = new CountDownLatch(1);

    GracefulExecutorServicesTerminationHandler(Duration shutdownTimeout, Duration terminationTimeout, List<ExecutorService> executorServices) {
        this.shutdownTimeout = shutdownTimeout;
        this.terminationTimeout = terminationTimeout;
        this.executors = Collections.unmodifiableList(new ArrayList<ExecutorService>(executorServices));
        this.future = new CompletableFuture();
        log.info("Starting termination handler for {} executors.", (Object)this.executors.size());
        for (ExecutorService executor : this.executors) {
            if (executor.isShutdown()) continue;
            throw new IllegalStateException(String.format("Executor %s should have been shutdown before entering the termination handler.", executor));
        }
        if (this.haveExecutorsBeenTerminated()) {
            this.markShutdownCompleted();
        } else if (shutdownTimeout.isZero() || shutdownTimeout.isNegative()) {
            this.terminateExecutors();
            this.markShutdownCompleted();
        } else {
            Thread shutdownWaitingThread = new Thread(this::awaitShutdown, this.getClass().getSimpleName());
            shutdownWaitingThread.setDaemon(false);
            shutdownWaitingThread.setUncaughtExceptionHandler((thread, exception) -> log.error("Uncaught exception in shutdown thread {}", (Object)thread, (Object)exception));
            shutdownWaitingThread.start();
            FutureUtil.whenCancelledOrTimedOut(this.future, () -> {
                shutdownWaitingThread.interrupt();
                this.waitUntilShutdownWaitingThreadIsCompleted();
            });
        }
    }

    public CompletableFuture<Void> getFuture() {
        return this.future;
    }

    private boolean haveExecutorsBeenTerminated() {
        return this.executors.stream().allMatch(ExecutorService::isTerminated);
    }

    private void markShutdownCompleted() {
        log.info("Shutdown completed.");
        this.future.complete(null);
    }

    private void awaitShutdown() {
        try {
            this.awaitTermination(this.shutdownTimeout);
            this.terminateExecutors();
            this.markShutdownCompleted();
        }
        catch (Exception e) {
            log.error("Error in termination handler", (Throwable)e);
            this.future.completeExceptionally(e);
        }
        finally {
            this.shutdownThreadCompletedLatch.countDown();
        }
    }

    private boolean awaitTermination(Duration timeout) {
        if (!timeout.isZero() && !timeout.isNegative()) {
            long awaitUntilNanos = System.nanoTime() + timeout.toNanos();
            while (!Thread.currentThread().isInterrupted() && System.nanoTime() < awaitUntilNanos) {
                int activeExecutorsCount = this.executors.size();
                for (ExecutorService executor : this.executors) {
                    long remainingTimeNanos = awaitUntilNanos - System.nanoTime();
                    if (remainingTimeNanos <= 0L) continue;
                    try {
                        if (!executor.isTerminated() && !executor.awaitTermination(remainingTimeNanos, TimeUnit.NANOSECONDS)) continue;
                        --activeExecutorsCount;
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                }
                if (activeExecutorsCount != 0) continue;
                return true;
            }
        }
        return this.haveExecutorsBeenTerminated();
    }

    private void terminateExecutors() {
        for (ExecutorService executor : this.executors) {
            if (executor.isTerminated()) continue;
            log.info("Shutting down forcefully executor {}", (Object)executor);
            executor.shutdownNow();
        }
        if (!Thread.currentThread().isInterrupted() && !this.awaitTermination(this.terminationTimeout)) {
            for (ExecutorService executor : this.executors) {
                if (executor.isTerminated()) continue;
                log.warn("Executor {} didn't shutdown after waiting for termination.", (Object)executor);
                for (Runnable runnable : executor.shutdownNow()) {
                    log.info("Execution in progress for runnable instance of {}: {}", runnable.getClass(), (Object)runnable);
                }
            }
        }
    }

    private void waitUntilShutdownWaitingThreadIsCompleted() {
        try {
            this.shutdownThreadCompletedLatch.await(this.terminationTimeout.toNanos() + SHUTDOWN_THREAD_COMPLETION_TIMEOUT_NANOS, TimeUnit.NANOSECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }
}

