/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.loadbalance;

import com.google.common.base.Charsets;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LinuxInfoUtils {
    private static final Logger log = LoggerFactory.getLogger(LinuxInfoUtils.class);
    private static final String CGROUPS_CPU_USAGE_PATH = "/sys/fs/cgroup/cpu/cpuacct.usage";
    private static final String CGROUPS_CPU_LIMIT_QUOTA_PATH = "/sys/fs/cgroup/cpu/cpu.cfs_quota_us";
    private static final String CGROUPS_CPU_LIMIT_PERIOD_PATH = "/sys/fs/cgroup/cpu/cpu.cfs_period_us";
    private static final String PROC_STAT_PATH = "/proc/stat";
    private static final String NIC_PATH = "/sys/class/net/";
    private static final int ARPHRD_ETHER = 1;
    private static final String NIC_SPEED_TEMPLATE = "/sys/class/net/%s/speed";

    public static boolean isLinux() {
        return SystemUtils.IS_OS_LINUX;
    }

    public static boolean isCGroupEnabled() {
        try {
            return Files.exists(Paths.get(CGROUPS_CPU_USAGE_PATH, new String[0]), new LinkOption[0]);
        }
        catch (Exception e) {
            log.warn("[LinuxInfo] Failed to check cgroup CPU usage file: {}", (Object)e.getMessage());
            return false;
        }
    }

    public static double getTotalCpuLimit(boolean isCGroupsEnabled) {
        if (isCGroupsEnabled) {
            try {
                long quota = LinuxInfoUtils.readLongFromFile(Paths.get(CGROUPS_CPU_LIMIT_QUOTA_PATH, new String[0]));
                long period = LinuxInfoUtils.readLongFromFile(Paths.get(CGROUPS_CPU_LIMIT_PERIOD_PATH, new String[0]));
                if (quota > 0L) {
                    return 100.0 * (double)quota / (double)period;
                }
            }
            catch (IOException e) {
                log.warn("[LinuxInfo] Failed to read CPU quotas from cgroups", (Throwable)e);
            }
        }
        return 100 * Runtime.getRuntime().availableProcessors();
    }

    public static double getCpuUsageForCGroup() {
        try {
            return LinuxInfoUtils.readLongFromFile(Paths.get(CGROUPS_CPU_USAGE_PATH, new String[0]));
        }
        catch (IOException e) {
            log.error("[LinuxInfo] Failed to read CPU usage from {}", (Object)CGROUPS_CPU_USAGE_PATH, (Object)e);
            return -1.0;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static ResourceUsage getCpuUsageForEntireHost() {
        try (Stream<String> stream = Files.lines(Paths.get(PROC_STAT_PATH, new String[0]));){
            Optional<String> first = stream.findFirst();
            if (!first.isPresent()) {
                log.error("[LinuxInfo] Failed to read CPU usage from /proc/stat, because of empty values.");
                ResourceUsage resourceUsage2 = ResourceUsage.empty();
                return resourceUsage2;
            }
            String[] words = first.get().split("\\s+");
            long total = Arrays.stream(words).filter(s -> !s.contains("cpu")).mapToLong(Long::parseLong).sum();
            long idle = Long.parseLong(words[4]);
            ResourceUsage resourceUsage = ResourceUsage.builder().usage(total - idle).idle(idle).total(total).build();
            return resourceUsage;
        }
        catch (IOException e) {
            log.error("[LinuxInfo] Failed to read CPU usage from /proc/stat", (Throwable)e);
            return ResourceUsage.empty();
        }
    }

    private static boolean isPhysicalNic(Path nicPath) {
        try {
            if (nicPath.toRealPath(new LinkOption[0]).toString().contains("/virtual/")) {
                return false;
            }
            String type = LinuxInfoUtils.readTrimStringFromFile(nicPath.resolve("type"));
            return Integer.parseInt(type) == 1;
        }
        catch (Exception e) {
            log.warn("[LinuxInfo] Failed to read {} NIC type, the detail is: {}", (Object)nicPath, (Object)e.getMessage());
            return false;
        }
    }

    public static double getTotalNicLimit(List<String> nics, NICUnit nicUnit) {
        return nicUnit.convertBy(nics.stream().mapToDouble(nicPath -> {
            try {
                return LinuxInfoUtils.readDoubleFromFile(LinuxInfoUtils.getReplacedNICPath(NIC_SPEED_TEMPLATE, nicPath));
            }
            catch (IOException e) {
                log.error("[LinuxInfo] Failed to get total nic limit.", (Throwable)e);
                return 0.0;
            }
        }).sum());
    }

    public static double getTotalNicUsage(List<String> nics, NICUsageType type, UsageUnit unit) {
        return unit.convertBy(nics.stream().mapToDouble(nic -> {
            try {
                return LinuxInfoUtils.readDoubleFromFile(LinuxInfoUtils.getReplacedNICPath(type.template, nic));
            }
            catch (IOException e) {
                log.error("[LinuxInfo] Failed to read {} bytes for NIC {} ", new Object[]{type, nic, e});
                return 0.0;
            }
        }).sum());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> getPhysicalNICs() {
        try (Stream<Path> stream = Files.list(Paths.get(NIC_PATH, new String[0]));){
            List<String> list = stream.filter(LinuxInfoUtils::isPhysicalNic).map(path -> path.getFileName().toString()).collect(Collectors.toList());
            return list;
        }
        catch (IOException e) {
            log.error("[LinuxInfo] Failed to find NICs", (Throwable)e);
            return Collections.emptyList();
        }
    }

    public static boolean checkHasNicSpeeds() {
        List<String> physicalNICs = LinuxInfoUtils.getPhysicalNICs();
        if (CollectionUtils.isEmpty(physicalNICs)) {
            return false;
        }
        double totalNicLimit = LinuxInfoUtils.getTotalNicLimit(physicalNICs, NICUnit.Kbps);
        return totalNicLimit > 0.0;
    }

    private static Path getReplacedNICPath(String template, String nic) {
        return Paths.get(String.format(template, nic), new String[0]);
    }

    private static String readTrimStringFromFile(Path path) throws IOException {
        return new String(Files.readAllBytes(path), Charsets.UTF_8).trim();
    }

    private static long readLongFromFile(Path path) throws IOException {
        return Long.parseLong(LinuxInfoUtils.readTrimStringFromFile(path));
    }

    private static double readDoubleFromFile(Path path) throws IOException {
        return Double.parseDouble(LinuxInfoUtils.readTrimStringFromFile(path));
    }

    public static class ResourceUsage {
        private final long total;
        private final long idle;
        private final long usage;

        public static ResourceUsage empty() {
            return ResourceUsage.builder().total(-1L).idle(-1L).usage(-1L).build();
        }

        public boolean isEmpty() {
            return this.total == -1L && this.idle == -1L && this.usage == -1L;
        }

        ResourceUsage(long total, long idle, long usage) {
            this.total = total;
            this.idle = idle;
            this.usage = usage;
        }

        public static ResourceUsageBuilder builder() {
            return new ResourceUsageBuilder();
        }

        public long getTotal() {
            return this.total;
        }

        public long getIdle() {
            return this.idle;
        }

        public long getUsage() {
            return this.usage;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof ResourceUsage)) {
                return false;
            }
            ResourceUsage other = (ResourceUsage)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (this.getTotal() != other.getTotal()) {
                return false;
            }
            if (this.getIdle() != other.getIdle()) {
                return false;
            }
            return this.getUsage() == other.getUsage();
        }

        protected boolean canEqual(Object other) {
            return other instanceof ResourceUsage;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $total = this.getTotal();
            result = result * 59 + (int)($total >>> 32 ^ $total);
            long $idle = this.getIdle();
            result = result * 59 + (int)($idle >>> 32 ^ $idle);
            long $usage = this.getUsage();
            result = result * 59 + (int)($usage >>> 32 ^ $usage);
            return result;
        }

        public String toString() {
            return "LinuxInfoUtils.ResourceUsage(total=" + this.getTotal() + ", idle=" + this.getIdle() + ", usage=" + this.getUsage() + ")";
        }

        public static class ResourceUsageBuilder {
            private long total;
            private long idle;
            private long usage;

            ResourceUsageBuilder() {
            }

            public ResourceUsageBuilder total(long total) {
                this.total = total;
                return this;
            }

            public ResourceUsageBuilder idle(long idle) {
                this.idle = idle;
                return this;
            }

            public ResourceUsageBuilder usage(long usage) {
                this.usage = usage;
                return this;
            }

            public ResourceUsage build() {
                return new ResourceUsage(this.total, this.idle, this.usage);
            }

            public String toString() {
                return "LinuxInfoUtils.ResourceUsage.ResourceUsageBuilder(total=" + this.total + ", idle=" + this.idle + ", usage=" + this.usage + ")";
            }
        }
    }

    public static enum NICUsageType {
        TX("/sys/class/net/%s/statistics/tx_bytes"),
        RX("/sys/class/net/%s/statistics/rx_bytes");

        private final String template;

        private NICUsageType(String template) {
            this.template = template;
        }
    }

    public static enum UsageUnit {
        Kbps(0);

        private final int convertUnit;

        public double convertBy(double usageBytes) {
            return (double)this.convertUnit * usageBytes;
        }

        private UsageUnit(int convertUnit) {
            this.convertUnit = convertUnit;
        }
    }

    public static enum NICUnit {
        Kbps(1024);

        private final int convertUnit;

        public double convertBy(double usageBytes) {
            return (double)this.convertUnit * usageBytes;
        }

        private NICUnit(int convertUnit) {
            this.convertUnit = convertUnit;
        }
    }
}

