/*
 * Decompiled with CFR 0.152.
 */
package io.strimzi.kafka.oauth.server.authorizer;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.strimzi.kafka.oauth.common.BearerTokenWithPayload;
import io.strimzi.kafka.oauth.common.HttpException;
import io.strimzi.kafka.oauth.common.JSONUtil;
import io.strimzi.kafka.oauth.common.LogUtil;
import io.strimzi.kafka.oauth.server.authorizer.Semaphores;
import io.strimzi.kafka.oauth.services.ServiceException;
import io.strimzi.kafka.oauth.services.Services;
import io.strimzi.kafka.oauth.validator.DaemonThreadFactory;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import org.apache.kafka.common.security.oauthbearer.OAuthBearerToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressFBWarnings(value={"THROWS_METHOD_THROWS_CLAUSE_BASIC_EXCEPTION"})
class GrantsHandler
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(GrantsHandler.class);
    private final HashMap<String, Info> grantsCache = new HashMap();
    private final Semaphores<JsonNode> semaphores = new Semaphores();
    private final ExecutorService refreshWorker;
    private final ScheduledExecutorService gcWorker;
    private final ScheduledExecutorService refreshScheduler;
    private final long gcPeriodMillis;
    private final Function<String, JsonNode> authorizationGrantsProvider;
    private final int httpRetries;
    private final long grantsMaxIdleMillis;
    private long lastGcRunTimeMillis;

    @Override
    public void close() {
        this.shutDownExecutorService("grants refresh scheduler", this.refreshScheduler);
        this.shutDownExecutorService("grants refresh worker", this.refreshWorker);
        this.shutDownExecutorService("gc worker", this.gcWorker);
    }

    private void shutDownExecutorService(String name, ExecutorService service) {
        try {
            log.trace("Shutting down {} [{}]", (Object)name, (Object)service);
            service.shutdownNow();
            if (!service.awaitTermination(10L, TimeUnit.SECONDS)) {
                log.debug("[IGNORED] Failed to cleanly shutdown {} within 10 seconds", (Object)name);
            }
        }
        catch (Throwable t) {
            log.warn("[IGNORED] Failed to cleanly shutdown {}: ", (Object)name, (Object)t);
        }
    }

    @SuppressFBWarnings(value={"MC_OVERRIDABLE_METHOD_CALL_IN_CONSTRUCTOR"})
    GrantsHandler(int grantsRefreshPeriodSeconds, int grantsRefreshPoolSize, int grantsMaxIdleTimeSeconds, Function<String, JsonNode> httpGrantsProvider, int httpRetries, int gcPeriodSeconds) {
        this.authorizationGrantsProvider = httpGrantsProvider;
        this.httpRetries = httpRetries;
        if (grantsMaxIdleTimeSeconds <= 0) {
            throw new IllegalArgumentException("grantsMaxIdleTimeSeconds <= 0");
        }
        this.grantsMaxIdleMillis = (long)grantsMaxIdleTimeSeconds * 1000L;
        DaemonThreadFactory daemonThreadFactory = new DaemonThreadFactory();
        if (grantsRefreshPeriodSeconds > 0) {
            this.refreshWorker = Executors.newFixedThreadPool(grantsRefreshPoolSize, (ThreadFactory)daemonThreadFactory);
            this.refreshScheduler = Executors.newSingleThreadScheduledExecutor((ThreadFactory)daemonThreadFactory);
            this.refreshScheduler.scheduleAtFixedRate(this::performRefreshGrantsRun, grantsRefreshPeriodSeconds, grantsRefreshPeriodSeconds, TimeUnit.SECONDS);
        } else {
            this.refreshWorker = null;
            this.refreshScheduler = null;
        }
        if (gcPeriodSeconds <= 0) {
            throw new IllegalArgumentException("gcPeriodSeconds <= 0");
        }
        this.gcPeriodMillis = (long)gcPeriodSeconds * 1000L;
        this.gcWorker = Executors.newSingleThreadScheduledExecutor((ThreadFactory)daemonThreadFactory);
        this.gcWorker.scheduleAtFixedRate(this::gcGrantsCacheRunnable, gcPeriodSeconds, gcPeriodSeconds, TimeUnit.SECONDS);
    }

    private void gcGrantsCacheRunnable() {
        long timePassedSinceGc = System.currentTimeMillis() - this.lastGcRunTimeMillis;
        if (timePassedSinceGc < this.gcPeriodMillis - 1000L) {
            log.debug("Skipped queued gc run (last run {} ms ago)", (Object)timePassedSinceGc);
            return;
        }
        this.lastGcRunTimeMillis = System.currentTimeMillis();
        this.gcGrantsCache();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void gcGrantsCache() {
        int afterSize;
        int beforeSize;
        long start = System.currentTimeMillis();
        HashSet userIds = new HashSet(Services.getInstance().getSessions().map(OAuthBearerToken::principalName));
        log.trace("Grants gc: active users: {}", userIds);
        HashMap<String, Info> hashMap = this.grantsCache;
        synchronized (hashMap) {
            beforeSize = this.grantsCache.size();
            this.grantsCache.keySet().retainAll(userIds);
            afterSize = this.grantsCache.size();
        }
        log.debug("Grants gc: active users count: {}, grantsCache size before: {}, grantsCache size after: {}, gc duration: {} ms", new Object[]{userIds.size(), beforeSize, afterSize, System.currentTimeMillis() - start});
    }

    private JsonNode fetchAndSaveGrants(String userId, Info grantsInfo) {
        ObjectNode grants = null;
        try {
            log.debug("Fetching grants from Keycloak for user {}", (Object)userId);
            grants = this.fetchGrantsWithRetry(grantsInfo.getAccessToken());
            if (grants == null) {
                log.debug("Received null grants for user: {}, token: {}", (Object)userId, (Object)LogUtil.mask((String)grantsInfo.getAccessToken()));
                grants = JSONUtil.newObjectNode();
            }
        }
        catch (HttpException e) {
            if (e.getStatus() == 403) {
                grants = JSONUtil.newObjectNode();
            }
            log.warn("Unexpected status while fetching authorization data - will retry next time: {}", (Object)e.getMessage());
        }
        if (grants != null) {
            log.debug("Saving non-null grants for user: {}, token: {}", (Object)userId, (Object)LogUtil.mask((String)grantsInfo.getAccessToken()));
            grantsInfo.setGrants((JsonNode)grants);
        }
        return grants;
    }

    private JsonNode fetchGrantsWithRetry(String token) {
        int i = 0;
        while (true) {
            ++i;
            try {
                if (i > 1) {
                    log.debug("Grants request attempt no. {}", (Object)i);
                }
                return this.authorizationGrantsProvider.apply(token);
            }
            catch (Exception e) {
                int status;
                if (e instanceof HttpException && (403 == (status = ((HttpException)((Object)e)).getStatus()) || 401 == status)) {
                    throw e;
                }
                if (!log.isInfoEnabled()) continue;
                log.info("Failed to fetch grants on try no. {}", (Object)i, (Object)e);
                if (i <= this.httpRetries) continue;
                log.debug("Failed to fetch grants after {} tries", (Object)i);
                throw e;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void performRefreshGrantsRun() {
        try {
            HashMap<String, Info> workmap;
            log.debug("Refreshing authorization grants ... [{}]", (Object)this);
            HashMap<String, Info> hashMap = this.grantsCache;
            synchronized (hashMap) {
                workmap = new HashMap<String, Info>(this.grantsCache);
            }
            Set<Map.Entry<String, Info>> entries = workmap.entrySet();
            ArrayList<Future> scheduled = new ArrayList<Future>(entries.size());
            long now = System.currentTimeMillis();
            for (Map.Entry<String, Info> ent : entries) {
                String userId = ent.getKey();
                Info grantsInfo = ent.getValue();
                if (grantsInfo.getLastUsed() < now - this.grantsMaxIdleMillis) {
                    log.debug("Skipping refreshing grants for user '{}' due to max idle time.", (Object)userId);
                    this.removeUserFromCacheIfExpiredOrIdle(userId);
                }
                scheduled.add(new Future(userId, grantsInfo, this.refreshWorker.submit(() -> {
                    JsonNode newGrants;
                    if (log.isTraceEnabled()) {
                        log.trace("Fetch grants for user: {}, token: {}", (Object)userId, (Object)LogUtil.mask((String)grantsInfo.getAccessToken()));
                    }
                    try {
                        newGrants = this.fetchGrantsWithRetry(grantsInfo.getAccessToken());
                    }
                    catch (HttpException e) {
                        if (403 == e.getStatus()) {
                            newGrants = JSONUtil.newObjectNode();
                        }
                        throw e;
                    }
                    JsonNode oldGrants = grantsInfo.getGrants();
                    if (!GrantsHandler.semanticGrantsEquals(newGrants, oldGrants)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Grants have changed for user: {}; before: {}; after: {}", new Object[]{userId, oldGrants, newGrants});
                        }
                        grantsInfo.setGrants(newGrants);
                    }
                    return newGrants;
                })));
            }
            for (Future f : scheduled) {
                try {
                    f.get();
                }
                catch (ExecutionException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof HttpException) {
                        log.debug("[IGNORED] Failed to fetch grants for user: {}", (Object)cause.getMessage());
                        if (401 == ((HttpException)cause).getStatus()) {
                            this.grantsCache.remove(f.getUserId());
                            log.debug("Removed user from grants cache: {}", (Object)f.getUserId());
                            Services.getInstance().getSessions().removeAllWithMatchingAccessToken(f.getGrantsInfo().accessToken);
                            continue;
                        }
                    }
                    log.warn("[IGNORED] Failed to fetch grants for user: {}", (Object)e.getMessage(), (Object)e);
                }
                catch (Throwable e) {
                    if (!log.isWarnEnabled()) continue;
                    log.warn("[IGNORED] Failed to fetch grants for user: {}, token: {} - {}", new Object[]{f.getUserId(), LogUtil.mask((String)f.getGrantsInfo().accessToken), e.getMessage(), e});
                }
            }
        }
        catch (Throwable t) {
            log.error("{}", (Object)t.getMessage(), (Object)t);
        }
        finally {
            log.debug("Done refreshing grants");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeUserFromCacheIfExpiredOrIdle(String userId) {
        HashMap<String, Info> hashMap = this.grantsCache;
        synchronized (hashMap) {
            Info info = this.grantsCache.get(userId);
            if (info != null) {
                boolean isIdle;
                long now = System.currentTimeMillis();
                boolean bl = isIdle = info.getLastUsed() < now - this.grantsMaxIdleMillis;
                if (isIdle || info.isExpiredAt(now)) {
                    log.debug("Removed user from grants cache due to {}: {}", (Object)(isIdle ? "'idle'" : "'expired'"), (Object)userId);
                    this.grantsCache.remove(userId);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Info getGrantsInfoFromCache(BearerTokenWithPayload token) {
        Info grantsInfo;
        HashMap<String, Info> hashMap = this.grantsCache;
        synchronized (hashMap) {
            grantsInfo = this.grantsCache.computeIfAbsent(token.principalName(), k -> new Info(token.value(), token.lifetimeMs()));
        }
        grantsInfo.updateTokenIfExpiresLater(token);
        return grantsInfo;
    }

    JsonNode fetchGrantsForUserOrWaitForDelivery(String userId, Info grantsInfo) {
        Semaphores.SemaphoreResult<JsonNode> semaphore = this.semaphores.acquireSemaphore(userId);
        if (semaphore.acquired()) {
            try {
                log.debug("Acquired semaphore for '{}'", (Object)userId);
                JsonNode grants = this.fetchAndSaveGrants(userId, grantsInfo);
                semaphore.future().complete(grants);
                JsonNode jsonNode = grants;
                return jsonNode;
            }
            catch (Throwable t) {
                semaphore.future().completeExceptionally(t);
                throw t;
            }
            finally {
                this.semaphores.releaseSemaphore(userId);
                log.debug("Released semaphore for '{}'", (Object)userId);
            }
        }
        try {
            log.debug("Waiting on another thread to get grants for '{}'", (Object)userId);
            return semaphore.future().get();
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof ServiceException) {
                throw (ServiceException)cause;
            }
            throw new ServiceException("ExecutionException waiting for grants result: ", (Throwable)e);
        }
        catch (InterruptedException e) {
            throw new ServiceException("InterruptedException waiting for grants result: ", (Throwable)e);
        }
    }

    private static boolean semanticGrantsEquals(JsonNode grants1, JsonNode grants2) {
        if (grants1 == grants2) {
            return true;
        }
        if (grants1 == null) {
            throw new IllegalArgumentException("Invalid grants: null");
        }
        if (grants2 == null) {
            return false;
        }
        if (!grants1.isArray()) {
            throw new IllegalArgumentException("Invalid grants: not a JSON array");
        }
        if (!grants2.isArray()) {
            throw new IllegalArgumentException("Invalid grants: not a JSON array");
        }
        return JSONUtil.asSetOfNodes((ArrayNode)((ArrayNode)grants1)).equals(JSONUtil.asSetOfNodes((ArrayNode)((ArrayNode)grants2)));
    }

    static class Future
    implements java.util.concurrent.Future<JsonNode> {
        private final java.util.concurrent.Future<JsonNode> delegate;
        private final String userId;
        private final Info grantsInfo;

        @SuppressFBWarnings(value={"EI_EXPOSE_REP2"})
        public Future(String userId, Info grantsInfo, java.util.concurrent.Future<JsonNode> future) {
            this.userId = userId;
            this.grantsInfo = grantsInfo;
            this.delegate = future;
        }

        @SuppressFBWarnings(value={"EI_EXPOSE_REP"})
        public Info getGrantsInfo() {
            return this.grantsInfo;
        }

        public String getUserId() {
            return this.userId;
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            return this.delegate.cancel(mayInterruptIfRunning);
        }

        @Override
        public boolean isCancelled() {
            return this.delegate.isCancelled();
        }

        @Override
        public boolean isDone() {
            return this.delegate.isDone();
        }

        @Override
        public JsonNode get() throws InterruptedException, ExecutionException {
            return this.delegate.get();
        }

        @Override
        public JsonNode get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            return this.delegate.get(timeout, unit);
        }
    }

    static class Info {
        private volatile String accessToken;
        private volatile JsonNode grants;
        private volatile long expiresAt;
        private volatile long lastUsed;

        Info(String accessToken, long expiresAt) {
            this.accessToken = accessToken;
            this.expiresAt = expiresAt;
            this.lastUsed = System.currentTimeMillis();
        }

        synchronized void updateTokenIfExpiresLater(BearerTokenWithPayload token) {
            this.lastUsed = System.currentTimeMillis();
            if (token.lifetimeMs() > this.expiresAt) {
                this.accessToken = token.value();
                this.expiresAt = token.lifetimeMs();
            }
        }

        String getAccessToken() {
            return this.accessToken;
        }

        JsonNode getGrants() {
            return this.grants;
        }

        void setGrants(JsonNode newGrants) {
            this.grants = newGrants;
        }

        long getLastUsed() {
            return this.lastUsed;
        }

        boolean isExpiredAt(long timestamp) {
            return this.expiresAt < timestamp;
        }
    }
}

