package io.substrait.type;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import io.substrait.extension.DefaultExtensionCatalog;
import io.substrait.extension.SimpleExtension;
import java.io.File;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class YamlRead {
  private static final Logger LOGGER = LoggerFactory.getLogger(YamlRead.class);

  private static final List<String> FUNCTIONS =
      Collections.unmodifiableList(
          Arrays.asList(
              "boolean",
              "aggregate_generic",
              "arithmetic_decimal",
              "arithmetic",
              "comparison",
              "datetime",
              "string"));

  public static void main(String[] args) throws Exception {
    try {
      System.out.println(
          "Read: " + YamlRead.class.getResource(DefaultExtensionCatalog.FUNCTIONS_BOOLEAN));
      List<SimpleExtension.Function> signatures = loadFunctions();

      signatures.forEach(f -> System.out.println(f.key()));
    } catch (Exception ex) {
      throw ex;
    }
  }

  public static List<SimpleExtension.Function> loadFunctions() {
    return loadFunctions(
        FUNCTIONS.stream()
            .map(c -> String.format("/src/substrait/extensions/functions_%s.yaml", c))
            .collect(java.util.stream.Collectors.toList()));
  }

  public static List<SimpleExtension.Function> loadFunctions(List<String> files) {
    return files.stream().flatMap(YamlRead::parse).collect(Collectors.toList());
  }

  private static Stream<SimpleExtension.Function> parse(String name) {

    try {
      ObjectMapper mapper =
          new ObjectMapper(new YAMLFactory())
              .enable(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY)
              .registerModule(Deserializers.MODULE);
      SimpleExtension.ExtensionSignatures doc =
          mapper.readValue(new File(name), SimpleExtension.ExtensionSignatures.class);

      LOGGER.atDebug().log(
          "Parsed {} functions in file {}.",
          Optional.ofNullable(doc.scalars()).map(List::size).orElse(0)
              + Optional.ofNullable(doc.aggregates()).map(List::size).orElse(0),
          name);

      return doc.resolve(name);
    } catch (RuntimeException ex) {
      throw ex;
    } catch (Exception ex) {
      throw new IllegalStateException("Failure while parsing file " + name, ex);
    }
  }
}
