/*
 * Copyright (c) 2020 Simer JS Plaha (simer.j@gmail.com - @simerplaha)
 *
 * This file is a part of SwayDB.
 *
 * SwayDB is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * SwayDB is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with SwayDB. If not, see <https://www.gnu.org/licenses/>.
 *
 * Additional permission under the GNU Affero GPL version 3 section 7:
 * If you modify this Program or any covered work, only by linking or
 * combining it with separate works, the licensors of this Program grant
 * you additional permission to convey the resulting work.
 */

package swaydb.java

import java.time.Duration
import java.util.Optional

import swaydb.Apply
import swaydb.data.util.Java._

import scala.compat.java8.DurationConverters._

/**
 * Returns types for [[swaydb.java.PureFunction]]
 */
sealed trait Return[+V]
object Return {

  /**
   * Return types for [[swaydb.java.PureFunction]] used in [[swaydb.java.Map]]
   */
  sealed trait Map[V] extends Return[V]

  /**
   * Return types for [[swaydb.java.PureFunction]] used in [[swaydb.java.Set]]
   */
  sealed trait Set[V] extends Return[V]

  def nothing[V](): Nothing[V] =
    Nothing[V]()

  def remove[V](): Remove[V] =
    Remove[V]()

  def expire[V](after: Duration): Expire[V] =
    Expire[V](after)

  def update[V](value: V, expireAfter: Duration): Update[V] =
    Update[V](value, Optional.of(expireAfter))

  def update[V](value: V): Update[V] =
    Update[V](value, Optional.empty())

  private[java] def toScalaMap[V](returnValue: Return.Map[V]): Apply.Map[V] =
    returnValue match {
      case Nothing() =>
        Apply.Nothing

      case Remove() =>
        Apply.Remove

      case Expire(expireAfter) =>
        Apply.Expire(expireAfter.toScala)

      case Update(value, expireAfter) =>
        new Apply.Update(value, expireAfter.asScalaMap(_.toScala.fromNow))
    }

  private[java] def toScalaSet(returnValue: Return.Set[java.lang.Void]): Apply.Set[scala.Nothing] =
    returnValue match {
      case Nothing() =>
        Apply.Nothing

      case Remove() =>
        Apply.Remove

      case Expire(expireAfter) =>
        Apply.Expire(expireAfter.toScala)
    }

  final case class Nothing[V]() extends Map[V] with Set[V]
  final case class Remove[V]() extends Map[V] with Set[V]
  final case class Expire[V](expireAfter: Duration) extends Map[V] with Set[V]

  //For Maps only.
  final case class Update[V](value: V, expireAfter: Optional[Duration]) extends Map[V]
}
