/*
 * Copyright (c) 2014-2024 Takari, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-v10.html
 */
package io.takari.jpgp.key;

import java.io.FileInputStream;
import java.io.IOException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;

/**
 * Loads PGP secret key from the exported key file, which normally ends with the ".asc"
 * extension and has a "-----BEGIN PGP PRIVATE KEY BLOCK-----" header.
 *
 * @author Kohsuke Kawaguchi
 */
public class FilePrivateKeySource {

    public PGPSecretKey load(String keyFile) throws IOException {
        try (FileInputStream inputStream = new FileInputStream(keyFile)) {
            PGPObjectFactory pgpObjectFactory =
                    new PGPObjectFactory(PGPUtil.getDecoderStream(inputStream), new JcaKeyFingerprintCalculator());
            Object pgpObject = pgpObjectFactory.nextObject();
            if (!(pgpObject instanceof PGPSecretKeyRing)) {
                throw new IOException(keyFile + " doesn't contain PGP private key");
            }
            PGPSecretKeyRing keyRing = (PGPSecretKeyRing) pgpObject;
            return keyRing.getSecretKey();
        }
    }
}
