package io.takari.maven.plugins.plugin;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.xml.PrettyPrintXMLWriter;
import org.codehaus.plexus.util.xml.XMLWriter;

import io.takari.maven.plugins.plugin.model.MojoDescriptor;
import io.takari.maven.plugins.plugin.model.MojoParameter;
import io.takari.maven.plugins.plugin.model.MojoRequirement;
import io.takari.maven.plugins.plugin.model.PluginDependency;
import io.takari.maven.plugins.plugin.model.PluginDescriptor;

// originally copied from org.apache.maven.tools.plugin.generator.PluginDescriptorGenerator
class PluginDescriptorWriter {
  private static final String encoding = "UTF-8";

  public void writeDescriptor(OutputStream outputStream, PluginDescriptor pluginDescriptor) throws IOException {
    OutputStreamWriter writer = new OutputStreamWriter(outputStream, encoding);

    XMLWriter w = new PrettyPrintXMLWriter(writer, encoding, null);

    w.writeMarkup("\n<!-- Generated by takari-plugin-tools -->\n\n");
    w.startElement("plugin");
    element(w, "name", pluginDescriptor.getName());
    element(w, "description", pluginDescriptor.getDescription());
    element(w, "groupId", pluginDescriptor.getGroupId());
    element(w, "artifactId", pluginDescriptor.getArtifactId());
    element(w, "version", pluginDescriptor.getVersion());
    element(w, "goalPrefix", pluginDescriptor.getGoalPrefix());
    element(w, "isolatedRealm", String.valueOf(pluginDescriptor.isIsolatedRealm()));
    element(w, "inheritedByDefault", String.valueOf(pluginDescriptor.isInheritedByDefault()));
    writeMojos(w, pluginDescriptor);
    writeDependencies(w, pluginDescriptor);
    w.endElement();

    writer.flush();
  }

  private void writeMojos(XMLWriter w, PluginDescriptor pluginDescriptor) {
    w.startElement("mojos");
    List<MojoDescriptor> descriptors = new ArrayList<>(pluginDescriptor.getMojos());
    Sorting.sortDescriptors(descriptors);
    for (MojoDescriptor descriptor : descriptors) {
      writeMojoDescriptor(descriptor, w);
    }
    w.endElement();
  }

  private void writeDependencies(XMLWriter w, PluginDescriptor pluginDescriptor) {
    w.startElement("dependencies");
    List<PluginDependency> deps = pluginDescriptor.getDependencies();
    for (PluginDependency dep : deps) {
      w.startElement("dependency");
      element(w, "groupId", dep.getGroupId());
      element(w, "artifactId", dep.getArtifactId());
      element(w, "type", dep.getType());
      element(w, "version", dep.getVersion());
      w.endElement();
    }
    w.endElement();
  }

  private void element(XMLWriter w, String name, String value) {
    if (StringUtils.isNotEmpty(value)) {
      w.startElement(name);
      w.writeText(value != null ? value : "");
      w.endElement();
    }
  }

  /**
   * @param mojoDescriptor not null
   * @param w not null
   * @param helpDescriptor will clean html content from description fields
   */
  protected void writeMojoDescriptor(MojoDescriptor mojoDescriptor, XMLWriter w) {
    w.startElement("mojo");

    element(w, "goal", mojoDescriptor.getGoal());
    element(w, "description", mojoDescriptor.getDescription());

    element(w, "requiresDependencyResolution", mojoDescriptor.getRequiresDependencyResolution());
    element(w, "requiresDependencyCollection", mojoDescriptor.getRequiresDependencyCollection());
    element(w, "requiresDirectInvocation", String.valueOf(mojoDescriptor.isRequiresDirectInvocation()));
    element(w, "requiresProject", String.valueOf(mojoDescriptor.isRequiresProject()));
    element(w, "requiresReports", String.valueOf(mojoDescriptor.isRequiresReports()));
    element(w, "aggregator", String.valueOf(mojoDescriptor.isAggregator()));
    element(w, "requiresOnline", String.valueOf(mojoDescriptor.isRequiresOnline()));
    element(w, "inheritedByDefault", String.valueOf(mojoDescriptor.isInheritedByDefault()));
    element(w, "phase", mojoDescriptor.getPhase());
    element(w, "threadSafe", String.valueOf(mojoDescriptor.isThreadSafe()));
    element(w, "implementation", mojoDescriptor.getImplementation());
    element(w, "language", mojoDescriptor.getLanguage());
    element(w, "configurator", mojoDescriptor.getConfigurator());
    element(w, "instantiationStrategy", mojoDescriptor.getInstantiationStrategy());

    // Strategy for handling repeated reference to mojo in
    // the calculated (decorated, resolved) execution stack
    element(w, "executionStrategy", mojoDescriptor.getExecutionStrategy());

    // if (StringUtils.isNotEmpty(mojoDescriptor.getExecutePhase())) {
    // element(w, "executePhase", mojoDescriptor.getExecutePhase());
    // }

    // if (StringUtils.isNotEmpty(mojoDescriptor.getExecuteGoal())) {
    // element(w, "executeGoal", mojoDescriptor.getExecuteGoal());
    // }

    // if (StringUtils.isNotEmpty(mojoDescriptor.getExecuteLifecycle())) {
    // element(w, "executeLifecycle", mojoDescriptor.getExecuteLifecycle());
    // }

    // if (StringUtils.isNotEmpty(mojoDescriptor.getComponentComposer())) {
    // w.startElement("composer");
    // w.writeText(mojoDescriptor.getComponentComposer());
    // w.endElement();
    // }

    element(w, "since", mojoDescriptor.getSince());
    element(w, "deprecated", mojoDescriptor.getDeprecated());

    // ----------------------------------------------------------------------
    // Parameters
    // ----------------------------------------------------------------------

    Set<MojoParameter> configuration = new LinkedHashSet<MojoParameter>();

    w.startElement("parameters");
    List<MojoParameter> parameters = new ArrayList<>(mojoDescriptor.getParameters());
    Sorting.sortParameters(parameters);
    for (MojoParameter parameter : parameters) {
      w.startElement("parameter");
      element(w, "name", parameter.getName());
      element(w, "alias", parameter.getAlias());
      element(w, "type", parameter.getType());
      element(w, "since", parameter.getSince());
      element(w, "deprecated", parameter.getDeprecated());
      element(w, "implementation", parameter.getImplementation());
      element(w, "required", Boolean.toString(parameter.isRequired()));
      element(w, "editable", Boolean.toString(parameter.isEditable()));
      element(w, "description", parameter.getDescription());
      w.endElement();

      if (StringUtils.isNotEmpty(parameter.getDefaultValue()) || StringUtils.isNotEmpty(parameter.getExpression())) {
        configuration.add(parameter);
      }
    }

    w.endElement();

    // ----------------------------------------------------------------------
    // Configuration
    // ----------------------------------------------------------------------

    if (!configuration.isEmpty()) {
      w.startElement("configuration");

      for (MojoParameter parameter : configuration) {
        w.startElement(parameter.getName());

        String type = parameter.getType();
        if (StringUtils.isNotEmpty(type)) {
          w.addAttribute("implementation", type);
        }

        if (parameter.getDefaultValue() != null) {
          w.addAttribute("default-value", parameter.getDefaultValue());
        }

        if (StringUtils.isNotEmpty(parameter.getExpression())) {
          w.writeText(parameter.getExpression());
        }

        w.endElement();
      }

      w.endElement();
    }

    List<MojoRequirement> requirements = new ArrayList<>(mojoDescriptor.getRequirements());
    if (!requirements.isEmpty()) {
      Sorting.sortRequirements(requirements);
      w.startElement("requirements");
      for (MojoRequirement requirement : requirements) {
        w.startElement("requirement");
        element(w, "role", requirement.getRole());
        element(w, "role-hint", requirement.getRoleHint());
        element(w, "field-name", requirement.getFieldName());
        w.endElement();
      }
      w.endElement();
    }

    w.endElement();
  }
}
