/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.common;

import io.temporal.common.RetryOptions;
import io.temporal.failure.ApplicationFailure;
import io.temporal.internal.common.AsyncBackoffThrottler;
import io.temporal.internal.common.BackoffThrottler;
import io.temporal.internal.common.CheckedExceptionWrapper;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Retryer {
    private static final Logger log = LoggerFactory.getLogger(Retryer.class);

    public static <T extends Throwable> void retry(RetryOptions options, Optional<Duration> expiration, RetryableProc<T> r) throws T {
        Retryer.retryWithResult(options, expiration, () -> {
            r.apply();
            return null;
        });
    }

    public static <R, T extends Throwable> R retryWithResult(RetryOptions options, Optional<Duration> expiration, RetryableFunc<R, T> r) throws T {
        int attempt = 0;
        long startTime = System.currentTimeMillis();
        BackoffThrottler throttler = new BackoffThrottler(options.getInitialInterval(), options.getMaximumInterval(), options.getBackoffCoefficient());
        while (true) {
            try {
                ++attempt;
                throttler.throttle();
                R result = r.apply();
                throttler.success();
                return result;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return null;
            }
            catch (Exception e) {
                throttler.failure();
                if (options.getDoNotRetry() != null) {
                    String type = e instanceof ApplicationFailure ? ((ApplicationFailure)e).getType() : e.getClass().getName();
                    for (String exceptionToNotRetry : options.getDoNotRetry()) {
                        if (!exceptionToNotRetry.equals(type)) continue;
                        Retryer.rethrow(e);
                    }
                }
                long elapsed = System.currentTimeMillis() - startTime;
                int maxAttempts = options.getMaximumAttempts();
                if (maxAttempts > 0 && attempt >= maxAttempts || expiration.isPresent() && elapsed >= expiration.get().toMillis()) {
                    Retryer.rethrow(e);
                }
                log.warn("Retrying after failure", (Throwable)e);
                continue;
            }
            break;
        }
    }

    public static <R> CompletableFuture<R> retryWithResultAsync(RetryOptions options, Optional<Duration> expiration, Supplier<CompletableFuture<R>> function) {
        int attempt = 0;
        long startTime = System.currentTimeMillis();
        AsyncBackoffThrottler throttler = new AsyncBackoffThrottler(options.getInitialInterval(), options.getMaximumInterval(), options.getBackoffCoefficient());
        CompletableFuture unwrappedExceptionResult = new CompletableFuture();
        CompletableFuture<R> result = Retryer.retryWithResultAsync(options, expiration, function, attempt + 1, startTime, throttler);
        CompletionStage ignored = result.handle((r, e) -> {
            if (e == null) {
                unwrappedExceptionResult.complete(r);
            } else {
                unwrappedExceptionResult.completeExceptionally(CheckedExceptionWrapper.unwrap(e));
            }
            return null;
        });
        return unwrappedExceptionResult;
    }

    private static <R> CompletableFuture<R> retryWithResultAsync(RetryOptions options, Optional<Duration> expiration, Supplier<CompletableFuture<R>> function, int attempt, long startTime, AsyncBackoffThrottler throttler) {
        return ((CompletableFuture)((CompletableFuture)throttler.throttle().thenCompose(ignore -> {
            try {
                CompletableFuture result = (CompletableFuture)function.get();
                if (result == null) {
                    return CompletableFuture.completedFuture(null);
                }
                return result.handle((r, e) -> {
                    if (e == null) {
                        throttler.success();
                        return r;
                    }
                    throttler.failure();
                    throw CheckedExceptionWrapper.wrap(e);
                });
            }
            catch (Throwable e2) {
                throttler.failure();
                throw CheckedExceptionWrapper.wrap(e2);
            }
        })).handle((r, e) -> Retryer.failOrRetry(options.toBuilder().validateBuildWithDefaults(), expiration, function, attempt, startTime, throttler, r, e))).thenCompose(pair -> {
            if (pair.getException() != null) {
                throw CheckedExceptionWrapper.wrap(pair.getException());
            }
            return pair.getValue();
        });
    }

    private static <R> ValueExceptionPair<R> failOrRetry(RetryOptions options, Optional<Duration> expiration, Supplier<CompletableFuture<R>> function, int attempt, long startTime, AsyncBackoffThrottler throttler, R r, Throwable e) {
        int maxAttempts;
        if (e == null) {
            return new ValueExceptionPair<R>(CompletableFuture.completedFuture(r), null);
        }
        if (e instanceof CompletionException) {
            e = e.getCause();
        }
        if (e instanceof Error) {
            return new ValueExceptionPair(null, e);
        }
        e = CheckedExceptionWrapper.unwrap(e);
        long elapsed = System.currentTimeMillis() - startTime;
        if (options.getDoNotRetry() != null) {
            String type = e instanceof ApplicationFailure ? ((ApplicationFailure)e).getType() : e.getClass().getName();
            for (String exceptionToNotRetry : options.getDoNotRetry()) {
                if (!exceptionToNotRetry.equals(type)) continue;
                return new ValueExceptionPair(null, e);
            }
        }
        if ((maxAttempts = options.getMaximumAttempts()) > 0 && attempt >= maxAttempts || expiration.isPresent() && elapsed >= expiration.get().toMillis()) {
            return new ValueExceptionPair(null, e);
        }
        log.debug("Retrying after failure", e);
        CompletableFuture<R> next = Retryer.retryWithResultAsync(options, expiration, function, attempt + 1, startTime, throttler);
        return new ValueExceptionPair<R>(next, null);
    }

    private static <T extends Throwable> void rethrow(Exception e) throws T {
        if (e instanceof RuntimeException) {
            throw (RuntimeException)e;
        }
        Exception toRethrow = e;
        throw toRethrow;
    }

    private Retryer() {
    }

    private static class ValueExceptionPair<V> {
        private final CompletableFuture<V> value;
        private final Throwable exception;

        public ValueExceptionPair(CompletableFuture<V> value, Throwable exception) {
            this.value = value;
            this.exception = exception;
        }

        public CompletableFuture<V> getValue() {
            return this.value;
        }

        public Throwable getException() {
            return this.exception;
        }
    }

    public static interface RetryableFunc<R, E extends Throwable> {
        public R apply() throws E;
    }

    public static interface RetryableProc<E extends Throwable> {
        public void apply() throws E;
    }
}

