/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.worker;

import com.uber.m3.tally.Scope;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.temporal.internal.BackoffThrottler;
import io.temporal.internal.common.InternalUtils;
import io.temporal.internal.worker.ExecutorThreadFactory;
import io.temporal.internal.worker.PollerOptions;
import io.temporal.internal.worker.ShutdownableTaskExecutor;
import io.temporal.internal.worker.SuspendableWorker;
import io.temporal.internal.worker.Throttler;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Poller<T>
implements SuspendableWorker {
    private final String identity;
    private final ShutdownableTaskExecutor<T> taskExecutor;
    private final PollTask<T> pollTask;
    private final PollerOptions pollerOptions;
    private static final Logger log = LoggerFactory.getLogger(Poller.class);
    private ThreadPoolExecutor pollExecutor;
    private final Scope metricsScope;
    private final AtomicReference<CountDownLatch> suspendLatch = new AtomicReference();
    private BackoffThrottler pollBackoffThrottler;
    private Throttler pollRateThrottler;
    private final Thread.UncaughtExceptionHandler uncaughtExceptionHandler = new PollerUncaughtExceptionHandler();

    public Poller(String identity, PollTask<T> pollTask, ShutdownableTaskExecutor<T> taskExecutor, PollerOptions pollerOptions, Scope metricsScope) {
        Objects.requireNonNull(identity, "identity cannot be null");
        Objects.requireNonNull(pollTask, "poll service should not be null");
        Objects.requireNonNull(taskExecutor, "taskExecutor should not be null");
        Objects.requireNonNull(pollerOptions, "pollerOptions should not be null");
        Objects.requireNonNull(metricsScope, "metricsScope should not be null");
        this.identity = identity;
        this.pollTask = pollTask;
        this.taskExecutor = taskExecutor;
        this.pollerOptions = pollerOptions;
        this.metricsScope = metricsScope;
    }

    @Override
    public void start() {
        if (log.isInfoEnabled()) {
            log.info("start(): " + this);
        }
        if (this.pollerOptions.getMaximumPollRatePerSecond() > 0.0) {
            this.pollRateThrottler = new Throttler("poller", this.pollerOptions.getMaximumPollRatePerSecond(), this.pollerOptions.getMaximumPollRateIntervalMilliseconds());
        }
        this.pollExecutor = new ThreadPoolExecutor(this.pollerOptions.getPollThreadCount(), this.pollerOptions.getPollThreadCount(), 1L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(this.pollerOptions.getPollThreadCount()));
        this.pollExecutor.setThreadFactory(new ExecutorThreadFactory(this.pollerOptions.getPollThreadNamePrefix(), this.pollerOptions.getUncaughtExceptionHandler()));
        this.pollBackoffThrottler = new BackoffThrottler(this.pollerOptions.getPollBackoffInitialInterval(), this.pollerOptions.getPollBackoffMaximumInterval(), this.pollerOptions.getPollBackoffCoefficient());
        for (int i = 0; i < this.pollerOptions.getPollThreadCount(); ++i) {
            this.pollExecutor.execute(new PollLoopTask(new PollExecutionTask()));
            this.metricsScope.counter("temporal_poller_start").inc(1L);
        }
    }

    @Override
    public boolean isStarted() {
        return this.pollExecutor != null;
    }

    @Override
    public boolean isShutdown() {
        return this.pollExecutor.isShutdown() && this.taskExecutor.isShutdown();
    }

    @Override
    public boolean isTerminated() {
        return this.pollExecutor.isTerminated() && this.taskExecutor.isTerminated();
    }

    @Override
    public void shutdown() {
        log.info("shutdown");
        if (!this.isStarted()) {
            return;
        }
        this.pollExecutor.shutdownNow();
        try {
            this.pollExecutor.awaitTermination(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.taskExecutor.shutdown();
    }

    @Override
    public void shutdownNow() {
        log.info("shutdownNow poller={}", (Object)this.pollerOptions.getPollThreadNamePrefix());
        if (!this.isStarted()) {
            return;
        }
        this.pollExecutor.shutdownNow();
        this.taskExecutor.shutdownNow();
    }

    @Override
    public void awaitTermination(long timeout, TimeUnit unit) {
        if (!this.isStarted()) {
            return;
        }
        long timeoutMillis = unit.toMillis(timeout);
        timeoutMillis = InternalUtils.awaitTermination(this.pollExecutor, timeoutMillis);
        InternalUtils.awaitTermination(this.taskExecutor, timeoutMillis);
    }

    @Override
    public void suspendPolling() {
        log.info("suspendPolling");
        this.suspendLatch.set(new CountDownLatch(1));
    }

    @Override
    public void resumePolling() {
        log.info("resumePolling");
        CountDownLatch existing = this.suspendLatch.getAndSet(null);
        if (existing != null) {
            existing.countDown();
        }
    }

    @Override
    public boolean isSuspended() {
        return this.suspendLatch.get() != null;
    }

    public String toString() {
        return "Poller{options=" + this.pollerOptions + ", identity=" + this.identity + '}';
    }

    private final class PollerUncaughtExceptionHandler
    implements Thread.UncaughtExceptionHandler {
        private PollerUncaughtExceptionHandler() {
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            if (!Poller.this.pollExecutor.isShutdown() || !this.shouldIgnoreDuringShutdown(e)) {
                this.logPollErrors(t, e);
            } else {
                this.logPollExceptionsSuppressedDuringShutdown(t, e);
            }
        }

        private void logPollErrors(Thread t, Throwable e) {
            StatusRuntimeException te;
            if (e instanceof StatusRuntimeException && (te = (StatusRuntimeException)e).getStatus().getCode() == Status.Code.DEADLINE_EXCEEDED) {
                log.warn("Failure in thread {}", (Object)t.getName(), (Object)e);
                return;
            }
            log.error("Failure in thread {}", (Object)t.getName(), (Object)e);
        }

        private void logPollExceptionsSuppressedDuringShutdown(Thread t, Throwable e) {
            log.trace("Failure in thread {} is suppressed, considered normal during shutdown", (Object)t.getName(), (Object)e);
        }

        private boolean shouldIgnoreDuringShutdown(Throwable ex) {
            return ex instanceof RejectedExecutionException || ex instanceof InterruptedException || ex.getCause() instanceof InterruptedException;
        }
    }

    private class PollExecutionTask
    implements ThrowingRunnable {
        private PollExecutionTask() {
        }

        @Override
        public void run() throws Exception {
            Object task = Poller.this.pollTask.poll();
            if (task == null) {
                return;
            }
            Poller.this.taskExecutor.process(task);
        }
    }

    private class PollLoopTask
    implements Runnable {
        private final ThrowingRunnable task;

        PollLoopTask(ThrowingRunnable task) {
            this.task = task;
        }

        @Override
        public void run() {
            try {
                CountDownLatch suspender;
                Poller.this.pollBackoffThrottler.throttle();
                if (Poller.this.pollRateThrottler != null) {
                    Poller.this.pollRateThrottler.throttle();
                }
                if ((suspender = (CountDownLatch)Poller.this.suspendLatch.get()) != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("poll task suspending latchCount=" + suspender.getCount());
                    }
                    suspender.await();
                }
                if (this.shouldTerminate()) {
                    return;
                }
                this.task.run();
                Poller.this.pollBackoffThrottler.success();
            }
            catch (Throwable e) {
                if (e instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
                Poller.this.pollBackoffThrottler.failure();
                Poller.this.uncaughtExceptionHandler.uncaughtException(Thread.currentThread(), e);
            }
            finally {
                if (!this.shouldTerminate()) {
                    Poller.this.pollExecutor.execute(this);
                } else {
                    log.info("poll loop is terminated");
                }
            }
        }

        private boolean shouldTerminate() {
            return Poller.this.pollExecutor.isShutdown() || Thread.currentThread().isInterrupted();
        }
    }

    static interface ThrowingRunnable {
        public void run() throws Throwable;
    }

    public static interface PollTask<TT> {
        public TT poll();
    }
}

