/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.testservice;

import io.grpc.Status;
import io.temporal.api.nexus.v1.Endpoint;
import io.temporal.api.nexus.v1.EndpointSpec;
import io.temporal.internal.testservice.TestNexusEndpointStore;
import java.util.List;
import java.util.SortedMap;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class TestNexusEndpointStoreImpl
implements TestNexusEndpointStore {
    private static final Pattern ENDPOINT_NAME_REGEX = Pattern.compile("^[a-zA-Z][a-zA-Z0-9\\-]*[a-zA-Z0-9]$");
    private final SortedMap<String, Endpoint> endpointsById = new ConcurrentSkipListMap<String, Endpoint>();
    private final ConcurrentMap<String, Endpoint> endpointsByName = new ConcurrentHashMap<String, Endpoint>();

    @Override
    public Endpoint createEndpoint(EndpointSpec spec) {
        this.validateEndpointSpec(spec);
        String id = UUID.randomUUID().toString();
        Endpoint endpoint = Endpoint.newBuilder().setId(id).setVersion(1L).setSpec(spec).build();
        if (this.endpointsByName.putIfAbsent(spec.getName(), endpoint) != null) {
            throw Status.ALREADY_EXISTS.withDescription("Nexus endpoint already registered with name: " + spec.getName()).asRuntimeException();
        }
        if (this.endpointsById.putIfAbsent(id, endpoint) != null) {
            throw Status.ALREADY_EXISTS.withDescription("Nexus endpoint already exists with ID: " + id).asRuntimeException();
        }
        return endpoint;
    }

    @Override
    public Endpoint updateEndpoint(String id, long version, EndpointSpec spec) {
        this.validateEndpointSpec(spec);
        Endpoint prev = (Endpoint)this.endpointsById.get(id);
        if (prev == null) {
            throw Status.NOT_FOUND.withDescription("Could not find Nexus endpoint with ID: " + id).asRuntimeException();
        }
        if (prev.getVersion() != version) {
            throw Status.INVALID_ARGUMENT.withDescription("Error updating Nexus endpoint: version mismatch. Expected: " + prev.getVersion() + " Received: " + version).asRuntimeException();
        }
        Endpoint updated = Endpoint.newBuilder((Endpoint)prev).setVersion(version + 1L).setSpec(spec).build();
        if (!prev.getSpec().getName().equals(spec.getName()) && this.endpointsByName.putIfAbsent(spec.getName(), updated) != null) {
            throw Status.ALREADY_EXISTS.withDescription("Error updating Nexus endpoint: endpoint already registered with updated name: " + spec.getName()).asRuntimeException();
        }
        this.endpointsByName.remove(prev.getSpec().getName());
        this.endpointsById.put(id, updated);
        return updated;
    }

    @Override
    public void deleteEndpoint(String id, long version) {
        Endpoint existing = (Endpoint)this.endpointsById.get(id);
        if (existing == null) {
            throw Status.NOT_FOUND.withDescription("Could not find Nexus endpoint with ID: " + id).asRuntimeException();
        }
        if (existing.getVersion() != version) {
            throw Status.INVALID_ARGUMENT.withDescription("Error deleting Nexus endpoint: version mismatch. Expected " + existing.getVersion() + " Received: " + version).asRuntimeException();
        }
        this.endpointsById.remove(id);
        this.endpointsByName.remove(existing.getSpec().getName());
    }

    @Override
    public Endpoint getEndpoint(String id) {
        Endpoint endpoint = (Endpoint)this.endpointsById.get(id);
        if (endpoint == null) {
            throw Status.NOT_FOUND.withDescription("Could not find Nexus endpoint with ID: " + id).asRuntimeException();
        }
        return endpoint;
    }

    @Override
    public Endpoint getEndpointByName(String name) {
        Endpoint endpoint = (Endpoint)this.endpointsByName.get(name);
        if (endpoint == null) {
            throw Status.NOT_FOUND.withDescription("Could not find Nexus endpoint with name: " + name).asRuntimeException();
        }
        return endpoint;
    }

    @Override
    public List<Endpoint> listEndpoints(long pageSize, byte[] nextPageToken, String name) {
        if (name != null && !name.isEmpty()) {
            return this.endpointsById.values().stream().filter(ep -> ep.getSpec().getName().equals(name)).limit(1L).collect(Collectors.toList());
        }
        if (nextPageToken.length > 0) {
            return this.endpointsById.tailMap(new String(nextPageToken)).values().stream().skip(1L).limit(pageSize).collect(Collectors.toList());
        }
        return this.endpointsById.values().stream().limit(pageSize).collect(Collectors.toList());
    }

    @Override
    public void validateEndpointSpec(EndpointSpec spec) {
        if (spec.getName().isEmpty()) {
            throw Status.INVALID_ARGUMENT.withDescription("Nexus endpoint name cannot be empty").asRuntimeException();
        }
        if (!ENDPOINT_NAME_REGEX.matcher(spec.getName()).matches()) {
            throw Status.INVALID_ARGUMENT.withDescription("Nexus endpoint name (" + spec.getName() + ") does not match expected pattern: " + ENDPOINT_NAME_REGEX.pattern()).asRuntimeException();
        }
        if (!spec.hasTarget()) {
            throw Status.INVALID_ARGUMENT.withDescription("Nexus endpoint spec must have a target").asRuntimeException();
        }
        if (!spec.getTarget().hasWorker()) {
            throw Status.INVALID_ARGUMENT.withDescription("Test server only supports Nexus endpoints with worker targets").asRuntimeException();
        }
    }

    @Override
    public void close() {
    }
}

