/*
 * Decompiled with CFR 0.152.
 */
package io.trino.benchto.driver.execution;

import com.google.common.collect.Lists;
import io.trino.benchto.driver.Benchmark;
import io.trino.benchto.driver.BenchmarkProperties;
import io.trino.benchto.driver.FailedBenchmarkExecutionException;
import io.trino.benchto.driver.execution.BenchmarkExecutionDriver;
import io.trino.benchto.driver.execution.BenchmarkExecutionResult;
import io.trino.benchto.driver.listeners.benchmark.BenchmarkStatusReporter;
import io.trino.benchto.driver.loader.BenchmarkLoader;
import io.trino.benchto.driver.macro.MacroService;
import io.trino.benchto.driver.utils.TimeUtils;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ExecutionDriver {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss:SSS");
    private static final Logger LOG = LoggerFactory.getLogger(BenchmarkExecutionDriver.class);
    @Autowired
    private BenchmarkProperties properties;
    @Autowired
    private BenchmarkStatusReporter benchmarkStatusReporter;
    @Autowired
    private BenchmarkLoader benchmarkLoader;
    @Autowired
    private BenchmarkExecutionDriver benchmarkExecutionDriver;
    @Autowired
    private MacroService macroService;
    private final ZonedDateTime startTime = TimeUtils.nowUtc();

    public void execute() {
        List<Benchmark> benchmarks = this.loadBenchmarks();
        if (benchmarks.isEmpty()) {
            LOG.warn("No benchmarks selected, exiting...");
            return;
        }
        this.executeBeforeAllMacros();
        try {
            this.executeBenchmarks(benchmarks);
        }
        finally {
            try {
                this.executeAfterAllMacros();
            }
            catch (RuntimeException e) {
                LOG.error("Exception during execution of after-all macros", (Throwable)e);
            }
        }
    }

    private void executeBeforeAllMacros() {
        this.runOptionalMacros(this.properties.getBeforeAllMacros(), "before all");
    }

    private void executeAfterAllMacros() {
        this.runOptionalMacros(this.properties.getAfterAllMacros(), "after all");
    }

    private void runOptionalMacros(Optional<List<String>> macros, String kind) {
        if (macros.isPresent()) {
            LOG.info("Running {} macros: {}", (Object)kind, macros.get());
            this.macroService.runBenchmarkMacros(macros.get());
        }
    }

    private List<Benchmark> loadBenchmarks() {
        List<String> executionSequenceIds = this.benchmarkExecutionSequenceIds();
        LOG.info("Running benchmarks(executionSequenceIds={}) with properties: {}", executionSequenceIds, (Object)this.properties);
        String firstSequenceId = (String)executionSequenceIds.stream().findFirst().orElseThrow();
        List<Benchmark> baseBenchmarks = this.benchmarkLoader.loadBenchmarks(firstSequenceId);
        LOG.info("Loaded {} benchmarks", (Object)baseBenchmarks.size());
        return executionSequenceIds.stream().flatMap(sequenceId -> baseBenchmarks.stream().map(benchmark -> new Benchmark.BenchmarkBuilder((Benchmark)benchmark, (String)sequenceId).build())).collect(Collectors.toList());
    }

    private List<String> benchmarkExecutionSequenceIds() {
        return this.properties.getExecutionSequenceId().orElse(List.of(TimeUtils.nowUtc().format(DATE_TIME_FORMATTER)));
    }

    private void executeBenchmarks(List<Benchmark> benchmarks) {
        Map groups = benchmarks.stream().collect(Collectors.groupingBy(Benchmark::getName, LinkedHashMap::new, Collectors.toList()));
        List<BenchmarkExecutionResult> benchmarkExecutionResults = this.executeBenchmarkGroups(groups, benchmarks.size());
        List<BenchmarkExecutionResult> failedBenchmarkResults = benchmarkExecutionResults.stream().filter(benchmarkExecutionResult -> !benchmarkExecutionResult.isSuccessful()).collect(Collectors.toList());
        this.benchmarkStatusReporter.awaitAllFutures(10L, TimeUnit.MINUTES);
        if (!failedBenchmarkResults.isEmpty()) {
            throw new FailedBenchmarkExecutionException(failedBenchmarkResults, benchmarkExecutionResults.size());
        }
    }

    private List<BenchmarkExecutionResult> executeBenchmarkGroups(Map<String, List<Benchmark>> groups, int numberOfBenchmarks) {
        ArrayList benchmarkExecutionResults = Lists.newArrayList();
        int benchmarkOrdinalNumber = 1;
        for (Map.Entry<String, List<Benchmark>> group : groups.entrySet()) {
            for (Benchmark benchmark : group.getValue()) {
                if (this.isTimeLimitEnded()) {
                    LOG.warn("Time limit for running benchmarks has run out");
                    return benchmarkExecutionResults;
                }
                this.executeHealthCheck(benchmark);
            }
            benchmarkExecutionResults.addAll(this.benchmarkExecutionDriver.execute(group.getValue(), benchmarkOrdinalNumber, numberOfBenchmarks, this.getExecutionTimeLimit()));
            benchmarkOrdinalNumber += group.getValue().size();
            this.benchmarkStatusReporter.processCompletedFutures();
        }
        return benchmarkExecutionResults;
    }

    private boolean isTimeLimitEnded() {
        Optional<Duration> timeLimit = this.properties.getTimeLimit();
        return timeLimit.isPresent() && timeLimit.get().compareTo(Duration.between(this.startTime, TimeUtils.nowUtc())) < 0;
    }

    private Optional<ZonedDateTime> getExecutionTimeLimit() {
        Optional<Duration> timeLimit = this.properties.getTimeLimit();
        return timeLimit.map(this.startTime::plus);
    }

    private void executeHealthCheck(Benchmark benchmark) {
        Optional<List<String>> macros = this.properties.getHealthCheckMacros();
        if (macros.isPresent()) {
            LOG.info("Running health check macros: {}", macros.get());
            this.macroService.runBenchmarkMacros(macros.get(), benchmark);
        }
    }
}

