/*
 * Decompiled with CFR 0.152.
 */
package io.trino.benchto.driver.loader;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import io.trino.benchto.driver.Benchmark;
import io.trino.benchto.driver.BenchmarkExecutionException;
import io.trino.benchto.driver.BenchmarkProperties;
import io.trino.benchto.driver.Query;
import io.trino.benchto.driver.loader.BenchmarkByActiveVariablesFilter;
import io.trino.benchto.driver.loader.BenchmarkDescriptor;
import io.trino.benchto.driver.loader.QueryLoader;
import io.trino.benchto.driver.service.BenchmarkServiceClient;
import io.trino.benchto.driver.utils.CartesianProductUtils;
import io.trino.benchto.driver.utils.NaturalOrderComparator;
import io.trino.benchto.driver.utils.YamlUtils;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import javax.annotation.PostConstruct;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

@Component
public class BenchmarkLoader {
    private static final Logger LOGGER = LoggerFactory.getLogger(BenchmarkLoader.class);
    private static final Pattern VALUE_SUBSTITUTION_PATTERN = Pattern.compile(".*\\$\\{.+}.*");
    private static final String BENCHMARK_FILE_SUFFIX = "yaml";
    private static final int DEFAULT_RUNS = 3;
    private static final int DEFAULT_CONCURRENCY = 1;
    private static final int DEFAULT_SUITE_PREWARM_RUNS = 0;
    private static final int DEFAULT_BENCHMARK_PREWARM_RUNS = 2;
    @Autowired
    private Environment environment;
    @Autowired
    private BenchmarkProperties properties;
    @Autowired
    private BenchmarkServiceClient benchmarkServiceClient;
    @Autowired
    private QueryLoader queryLoader;
    @Autowired
    private Configuration freemarkerConfiguration;
    private Map<Object, Object> overrides;

    @PostConstruct
    public void setup() throws IOException {
        this.overrides = this.properties.getOverridesPath().isPresent() ? YamlUtils.loadYamlFromPath(this.properties.getOverridesPath().get()) : Collections.emptyMap();
    }

    public List<Benchmark> loadBenchmarks(String sequenceId) {
        try {
            List<Path> benchmarkFiles = this.findBenchmarkFiles();
            benchmarkFiles = benchmarkFiles.stream().filter(this.activeBenchmarks()).collect(Collectors.toList());
            this.verifyNoDuplicateBenchmarks(benchmarkFiles);
            benchmarkFiles.forEach(path -> LOGGER.info("Benchmark file to be read: {}", path));
            List<Benchmark> allBenchmarks = this.loadBenchmarks(sequenceId, benchmarkFiles);
            LOGGER.debug("All benchmarks: {}", allBenchmarks);
            List<Benchmark> includedBenchmarks = allBenchmarks.stream().filter(new BenchmarkByActiveVariablesFilter(this.properties)).collect(Collectors.toList());
            LinkedHashSet excludedBenchmarks = Sets.newLinkedHashSet(allBenchmarks);
            excludedBenchmarks.removeAll(includedBenchmarks);
            String formatString = this.createFormatString(allBenchmarks);
            LOGGER.info("Excluded Benchmarks:");
            this.printFormattedBenchmarksInfo(formatString, excludedBenchmarks);
            this.fillUniqueBenchmarkNames(includedBenchmarks);
            Object freshBenchmarks = ImmutableList.of();
            if (this.properties.isFrequencyCheckEnabled()) {
                freshBenchmarks = this.filterFreshBenchmarks(includedBenchmarks);
                LOGGER.info("Recently tested benchmarks:");
                this.printFormattedBenchmarksInfo(formatString, (Collection<Benchmark>)freshBenchmarks);
            }
            LOGGER.info("Selected Benchmarks:");
            includedBenchmarks.removeAll((Collection<?>)freshBenchmarks);
            this.printFormattedBenchmarksInfo(formatString, includedBenchmarks);
            Preconditions.checkState((allBenchmarks.size() == includedBenchmarks.size() + excludedBenchmarks.size() + freshBenchmarks.size() ? 1 : 0) != 0);
            return includedBenchmarks;
        }
        catch (IOException e) {
            throw new BenchmarkExecutionException("Could not load benchmarks", e);
        }
    }

    private List<Path> findBenchmarkFiles() throws IOException {
        LOGGER.info("Searching for benchmarks in classpath ...");
        this.verifyNoNestedBenchmarkDirs();
        ImmutableList.Builder benchmarkFilesBuilder = ImmutableList.builder();
        for (Path benchmarkFilesPath : this.properties.benchmarksFilesDirs()) {
            Files.walk(benchmarkFilesPath, new FileVisitOption[0]).filter(file -> Files.isRegularFile(file, new LinkOption[0]) && file.toString().endsWith(BENCHMARK_FILE_SUFFIX)).forEach(arg_0 -> ((ImmutableList.Builder)benchmarkFilesBuilder).add(arg_0));
        }
        ImmutableList benchmarkFiles = benchmarkFilesBuilder.build();
        benchmarkFiles.forEach(path -> LOGGER.info("Benchmark found: {}", (Object)path.toString()));
        return benchmarkFiles;
    }

    private void verifyNoNestedBenchmarkDirs() {
        for (Path benchmarkFilesPath : this.properties.benchmarksFilesDirs()) {
            for (Path otherBenchmarkFilesPath : this.properties.benchmarksFilesDirs()) {
                if (benchmarkFilesPath.equals(otherBenchmarkFilesPath) || !benchmarkFilesPath.startsWith(otherBenchmarkFilesPath)) continue;
                throw new BenchmarkExecutionException("Benchmark directories contain nested paths");
            }
        }
    }

    private void verifyNoDuplicateBenchmarks(List<Path> benchmarkFiles) {
        HashSet<String> benchmarkNames = new HashSet<String>();
        for (Path benchmarkFile : benchmarkFiles) {
            String benchmarkName = this.benchmarkName(benchmarkFile);
            if (benchmarkNames.add(benchmarkName)) continue;
            throw new BenchmarkExecutionException("Benchmark with name \"" + benchmarkName + "\" in multiple locations");
        }
    }

    private List<Benchmark> loadBenchmarks(String sequenceId, List<Path> benchmarkFiles) {
        return benchmarkFiles.stream().flatMap(file -> this.loadBenchmarks(sequenceId, (Path)file).stream()).sorted((left, right) -> NaturalOrderComparator.forStrings().compare(left.getName(), right.getName())).collect(Collectors.toList());
    }

    private List<Benchmark> loadBenchmarks(String sequenceId, Path benchmarkFile) {
        try {
            Map<Object, Object> yaml = YamlUtils.loadYamlFromPath(benchmarkFile);
            yaml = this.mergeTopLevelVariables(yaml);
            Preconditions.checkArgument((boolean)yaml.containsKey("datasource"), (String)"Mandatory variable %s not present in file %s", (Object)"datasource", (Object)benchmarkFile);
            Preconditions.checkArgument((boolean)yaml.containsKey("query-names"), (String)"Mandatory variable %s not present in file %s", (Object)"query-names", (Object)benchmarkFile);
            String defaultName = this.benchmarkName(benchmarkFile);
            List<BenchmarkDescriptor> benchmarkDescriptors = this.createBenchmarkDescriptors(defaultName, yaml);
            ArrayList benchmarks = Lists.newArrayListWithCapacity((int)benchmarkDescriptors.size());
            for (BenchmarkDescriptor benchmarkDescriptor : benchmarkDescriptors) {
                String benchmarkName = benchmarkDescriptor.getName();
                List<Query> queries = this.queryLoader.loadFromFiles(benchmarkDescriptor.getQueryNames());
                Benchmark benchmark = new Benchmark.BenchmarkBuilder(benchmarkName, sequenceId, queries).withDataSource(benchmarkDescriptor.getDataSource()).withEnvironment(this.properties.getEnvironmentName()).withRuns(benchmarkDescriptor.getRuns().orElse(3)).withSuitePrewarmRuns(benchmarkDescriptor.getSuitePrewarmRuns().orElse(0)).withBenchmarkPrewarmRuns(benchmarkDescriptor.getBenchmarkPrewarmRuns().orElse(2)).withConcurrency(benchmarkDescriptor.getConcurrency().orElse(1)).withFrequency(benchmarkDescriptor.getFrequency().map(Duration::ofDays)).withThroughputTest(benchmarkDescriptor.getThroughputTest()).withBeforeBenchmarkMacros(benchmarkDescriptor.getBeforeBenchmarkMacros()).withAfterBenchmarkMacros(benchmarkDescriptor.getAfterBenchmarkMacros()).withBeforeExecutionMacros(benchmarkDescriptor.getBeforeExecutionMacros()).withAfterExecutionMacros(benchmarkDescriptor.getAfterExecutionMacros()).withQueryResults(benchmarkDescriptor.getResults()).withVariables(benchmarkDescriptor.getVariables()).build();
                benchmarks.add(benchmark);
            }
            return benchmarks;
        }
        catch (IOException e) {
            throw new BenchmarkExecutionException("Could not load benchmark: " + benchmarkFile, e);
        }
    }

    private Map<Object, Object> mergeTopLevelVariables(Map<Object, Object> baseYaml) {
        Object value;
        Object key;
        ImmutableMap.Builder result = ImmutableMap.builder();
        for (Map.Entry<Object, Object> entry : baseYaml.entrySet()) {
            key = entry.getKey();
            value = entry.getValue();
            if (!(value instanceof Map) && this.overrides.containsKey(key)) {
                result.put(key, this.overrides.get(key));
                continue;
            }
            result.put(key, value);
        }
        for (Map.Entry<Object, Object> entry : this.overrides.entrySet()) {
            key = entry.getKey();
            value = entry.getValue();
            if (baseYaml.containsKey(key)) continue;
            result.put(key, value);
        }
        return result.build();
    }

    private List<BenchmarkDescriptor> createBenchmarkDescriptors(String defaultName, Map<Object, Object> yaml) {
        List<Map<String, String>> variablesCombinations = this.extractVariableMapList(yaml);
        Map<String, String> globalVariables = this.extractGlobalVariables(yaml);
        globalVariables.putIfAbsent("name", defaultName);
        for (Map<String, String> variablesMap : variablesCombinations) {
            for (Map.Entry<String, String> globalVariableEntry : globalVariables.entrySet()) {
                variablesMap.putIfAbsent(globalVariableEntry.getKey(), globalVariableEntry.getValue());
            }
            this.evaluateValueExpressions(variablesMap);
        }
        return variablesCombinations.stream().map(BenchmarkDescriptor::new).collect(Collectors.toList());
    }

    private void evaluateValueExpressions(Map<String, String> variablesMap) {
        for (Map.Entry<String, String> variableEntry : variablesMap.entrySet()) {
            String variableValue = variableEntry.getValue();
            try {
                if (!VALUE_SUBSTITUTION_PATTERN.matcher(variableValue).matches()) continue;
                Template valueTemplate = new Template(UUID.randomUUID().toString(), variableValue, this.freemarkerConfiguration);
                String evaluatedValue = FreeMarkerTemplateUtils.processTemplateIntoString((Template)valueTemplate, variablesMap);
                if (VALUE_SUBSTITUTION_PATTERN.matcher(evaluatedValue).matches()) {
                    throw new BenchmarkExecutionException("Recursive value substitution is not supported, invalid " + variableEntry.getKey() + ": " + variableValue);
                }
                variableEntry.setValue(evaluatedValue);
            }
            catch (TemplateException | IOException e) {
                throw new BenchmarkExecutionException("Could not evaluate value " + variableValue, e);
            }
        }
    }

    private String benchmarkName(Path benchmarkFile) {
        Path benchmarkFilesDir = this.properties.benchmarksFilesDirs().stream().filter(benchmarkFile::startsWith).findFirst().get();
        String relativePath = benchmarkFilesDir.relativize(benchmarkFile).toString();
        return FilenameUtils.removeExtension((String)relativePath);
    }

    private Map<String, String> extractGlobalVariables(Map<Object, Object> yaml) {
        return yaml.entrySet().stream().filter(entry -> !entry.getKey().toString().equals("variables")).collect(Collectors.toMap(entry -> entry.getKey().toString(), entry -> entry.getValue() == null ? null : entry.getValue().toString()));
    }

    private List<Map<String, String>> extractVariableMapList(Map<Object, Object> yaml) {
        Map variableMaps = (Map)yaml.getOrDefault("variables", Maps.newHashMap());
        List<Map<String, String>> variableMapList = variableMaps.values().stream().map(YamlUtils::stringifyMultimap).flatMap(variableMap -> CartesianProductUtils.cartesianProduct(variableMap).stream()).collect(Collectors.toList());
        if (variableMapList.isEmpty()) {
            variableMapList.add(Maps.newHashMap());
        }
        return variableMapList;
    }

    private Predicate<Path> activeBenchmarks() {
        Optional<List<String>> activeBenchmarks = this.properties.getActiveBenchmarks();
        return activeBenchmarks.map(this::benchmarkNameIn).orElseGet(() -> path -> true);
    }

    private Predicate<Path> benchmarkNameIn(List<String> activeBenchmarks) {
        ImmutableList names = ImmutableList.copyOf(activeBenchmarks);
        return arg_0 -> this.lambda$benchmarkNameIn$11((List)names, arg_0);
    }

    private void fillUniqueBenchmarkNames(List<Benchmark> benchmarks) {
        List<BenchmarkServiceClient.GenerateUniqueNamesRequestItem> namesRequestItems = benchmarks.stream().map(benchmark -> BenchmarkServiceClient.GenerateUniqueNamesRequestItem.generateUniqueNamesRequestItem(benchmark.getName(), benchmark.getNonReservedKeywordVariables())).collect(Collectors.toList());
        List<String> uniqueBenchmarkNames = this.benchmarkServiceClient.generateUniqueBenchmarkNames(namesRequestItems);
        Preconditions.checkState((uniqueBenchmarkNames.size() == benchmarks.size() ? 1 : 0) != 0);
        for (int i = 0; i < uniqueBenchmarkNames.size(); ++i) {
            benchmarks.get(i).setUniqueName(uniqueBenchmarkNames.get(i));
        }
    }

    private List<Benchmark> filterFreshBenchmarks(List<Benchmark> benchmarks) {
        List benchmarksWithFrequencySet = benchmarks.stream().filter(benchmark -> benchmark.getFrequency().isPresent()).collect(Collectors.toList());
        if (benchmarksWithFrequencySet.isEmpty()) {
            return ImmutableList.of();
        }
        List<String> benchmarkUniqueNames = benchmarksWithFrequencySet.stream().map(Benchmark::getUniqueName).collect(Collectors.toList());
        List<Duration> successfulExecutionAges = this.benchmarkServiceClient.getBenchmarkSuccessfulExecutionAges(benchmarkUniqueNames);
        return IntStream.range(0, benchmarksWithFrequencySet.size()).mapToObj(i -> {
            Benchmark benchmark = (Benchmark)benchmarksWithFrequencySet.get(i);
            if (((Duration)successfulExecutionAges.get(i)).compareTo(benchmark.getFrequency().get()) <= 0) {
                return benchmark;
            }
            return null;
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private void printFormattedBenchmarksInfo(String formatString, Collection<Benchmark> benchmarks) {
        LOGGER.info(String.format(formatString, "Benchmark Name", "Data Source", "Runs", "SuitePrewarms", "BenchmarkPrewarms", "Concurrency", "Throughput Test"));
        benchmarks.stream().map(benchmark -> String.format(formatString, benchmark.getName(), benchmark.getDataSource(), "" + benchmark.getRuns(), "" + benchmark.getSuitePrewarmRuns(), "" + benchmark.getBenchmarkPrewarmRuns(), "" + benchmark.getConcurrency(), "" + benchmark.isThroughputTest())).distinct().forEach(arg_0 -> ((Logger)LOGGER).info(arg_0));
    }

    private String createFormatString(Collection<Benchmark> benchmarks) {
        int nameMaxLength = benchmarks.stream().mapToInt(benchmark -> benchmark.getName().length()).max().orElse(10);
        int dataSourceMaxLength = benchmarks.stream().mapToInt(benchmark -> benchmark.getDataSource().length()).max().orElse(10);
        int indent = 3;
        return "\t| %-" + (nameMaxLength + indent) + "s | %-" + Math.max(dataSourceMaxLength + indent, 11) + "s | %-4s | %-8s | %-11s | %-15s |";
    }

    private /* synthetic */ boolean lambda$benchmarkNameIn$11(List names, Path benchmarkFile) {
        String benchmarkName = this.benchmarkName(benchmarkFile);
        return names.contains(benchmarkName);
    }
}

