/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 * See License.txt in the project root for license information.
 */

package io.trino.hadoop.$internal.com.microsoft.azure.datalake.store.retrypolicies;

/**
 * implements different retry decisions based on the error.
 *
 * <UL>
 *     <LI>For nonretryable errors (3xx, most 4xx, and some 5xx return codes), do no retry.</LI>
 *     <LI>For throttling error, do a retry with exponential backoff</LI>
 *     <LI>for all other errors, do a retry with linear backoff</LI>
 * </UL>
 */
public class ExponentialBackoffPolicyforMSI implements RetryPolicy {

    private int retryCount = 0;
    private int maxRetries = 4;
    private int exponentialRetryInterval = 1000;
    private int exponentialFactor = 4;
    private long lastAttemptStartTime = System.nanoTime();

    public ExponentialBackoffPolicyforMSI() {
    }

    /**
     * Implements Exponential backoff policy, with error condition checks specific to MSI
     * @param maxRetries maximum number of retries
     * @param exponentialRetryInterval (starting) interval to use for exponential backoff retries (in milliseconds)
     * @param exponentialFactor factor to multiply the retry interval by
     */
    public ExponentialBackoffPolicyforMSI(int maxRetries, int exponentialRetryInterval, int exponentialFactor) {
        this.maxRetries = maxRetries;
        this.exponentialRetryInterval = exponentialRetryInterval;
        this.exponentialFactor = exponentialFactor;
    }


    public boolean shouldRetry(int httpResponseCode, Exception lastException) {

        // Non-retryable error
        if (      (httpResponseCode >= 300 && httpResponseCode < 500   // 3xx and 4xx, except specific ones below
                                           && httpResponseCode != 404  // 404 is required for MSI; for some error conditions
                                                                       // they return a 404 that goes away on retry
                                           && httpResponseCode != 408
                                           && httpResponseCode != 429
                                           && httpResponseCode != 401)
               || (httpResponseCode == 501) // Not Implemented
               || (httpResponseCode == 505) // Version Not Supported
               ) {
            return false;
        }

        // Retryable error, retry with exponential backoff
        if ( lastException!=null || httpResponseCode >=500    // exception or 5xx, + specific ones below
                                 || httpResponseCode == 404   // see comment above for 404
                                 || httpResponseCode == 408
                                 || httpResponseCode == 429
                                 || httpResponseCode == 401) {
            if (retryCount < maxRetries) {
                int timeSpent = (int)((System.nanoTime() - lastAttemptStartTime) / 1000000);
                wait(exponentialRetryInterval - timeSpent);
                exponentialRetryInterval *= exponentialFactor;
                retryCount++;
                lastAttemptStartTime = System.nanoTime();
                return true;
            } else {
                return false;  // max # of retries exhausted
            }
        }

        // these are not errors - this method should never have been called with this
        if (httpResponseCode >= 100 && httpResponseCode <300)
        {
            return false;
        }

        // Dont know what happened - we should never get here
        return false;
    }

    private void wait(int milliseconds) {
        if (milliseconds <= 0) {
            return;
        }

        try {
            Thread.sleep(milliseconds);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();   // http://www.ibm.com/developerworks/library/j-jtp05236/
        }
    }
}
