/*
 * Copyright 2012 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package io.trino.hadoop.$internal.io.netty.example.file;

import io.trino.hadoop.$internal.io.netty.bootstrap.ServerBootstrap;
import io.trino.hadoop.$internal.io.netty.channel.ChannelFuture;
import io.trino.hadoop.$internal.io.netty.channel.ChannelInitializer;
import io.trino.hadoop.$internal.io.netty.channel.ChannelOption;
import io.trino.hadoop.$internal.io.netty.channel.ChannelPipeline;
import io.trino.hadoop.$internal.io.netty.channel.EventLoopGroup;
import io.trino.hadoop.$internal.io.netty.channel.nio.NioEventLoopGroup;
import io.trino.hadoop.$internal.io.netty.channel.socket.SocketChannel;
import io.trino.hadoop.$internal.io.netty.channel.socket.nio.NioServerSocketChannel;
import io.trino.hadoop.$internal.io.netty.handler.codec.LineBasedFrameDecoder;
import io.trino.hadoop.$internal.io.netty.handler.codec.string.StringDecoder;
import io.trino.hadoop.$internal.io.netty.handler.codec.string.StringEncoder;
import io.trino.hadoop.$internal.io.netty.handler.logging.LogLevel;
import io.trino.hadoop.$internal.io.netty.handler.logging.LoggingHandler;
import io.trino.hadoop.$internal.io.netty.handler.ssl.SslContext;
import io.trino.hadoop.$internal.io.netty.handler.ssl.SslContextBuilder;
import io.trino.hadoop.$internal.io.netty.handler.ssl.util.SelfSignedCertificate;
import io.trino.hadoop.$internal.io.netty.handler.stream.ChunkedWriteHandler;
import io.trino.hadoop.$internal.io.netty.util.CharsetUtil;

/**
 * Server that accept the path of a file an echo back its content.
 */
public final class FileServer {

    static final boolean SSL = System.getProperty("ssl") != null;
    // Use the same default port with the telnet example so that we can use the telnet client example to access it.
    static final int PORT = Integer.parseInt(System.getProperty("port", SSL? "8992" : "8023"));

    public static void main(String[] args) throws Exception {
        // Configure SSL.
        final SslContext sslCtx;
        if (SSL) {
            SelfSignedCertificate ssc = new SelfSignedCertificate();
            sslCtx = SslContextBuilder.forServer(ssc.certificate(), ssc.privateKey()).build();
        } else {
            sslCtx = null;
        }

        // Configure the server.
        EventLoopGroup bossGroup = new NioEventLoopGroup(1);
        EventLoopGroup workerGroup = new NioEventLoopGroup();
        try {
            ServerBootstrap b = new ServerBootstrap();
            b.group(bossGroup, workerGroup)
             .channel(NioServerSocketChannel.class)
             .option(ChannelOption.SO_BACKLOG, 100)
             .handler(new LoggingHandler(LogLevel.INFO))
             .childHandler(new ChannelInitializer<SocketChannel>() {
                 @Override
                 public void initChannel(SocketChannel ch) throws Exception {
                     ChannelPipeline p = ch.pipeline();
                     if (sslCtx != null) {
                         p.addLast(sslCtx.newHandler(ch.alloc()));
                     }
                     p.addLast(
                             new StringEncoder(CharsetUtil.UTF_8),
                             new LineBasedFrameDecoder(8192),
                             new StringDecoder(CharsetUtil.UTF_8),
                             new ChunkedWriteHandler(),
                             new FileServerHandler());
                 }
             });

            // Start the server.
            ChannelFuture f = b.bind(PORT).sync();

            // Wait until the server socket is closed.
            f.channel().closeFuture().sync();
        } finally {
            // Shut down all event loops to terminate all threads.
            bossGroup.shutdownGracefully();
            workerGroup.shutdownGracefully();
        }
    }
}
