/**
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package io.trino.hadoop.$internal.org.apache.kerby.cms.type;

import io.trino.hadoop.$internal.org.apache.kerby.asn1.Asn1FieldInfo;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.EnumType;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.ImplicitField;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.type.Asn1ObjectIdentifier;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.type.Asn1OctetString;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.type.Asn1SequenceType;

/**
 * EncryptedContentInfo ::= SEQUENCE {
 *   contentType ContentType,
 *   contentEncryptionAlgorithm ContentEncryptionAlgorithmIdentifier,
 *   encryptedContent [0] IMPLICIT EncryptedContent OPTIONAL }
 *
 * ContentType ::= OBJECT IDENTIFIER
 *
 * EncryptedContent ::= OCTET STRING
 */
public class EncryptedContentInfo extends Asn1SequenceType {
    protected enum ECInfoField implements EnumType {
        CONTENT_TYPE,
        CONTENT_ENCRYPTION_ALGORITHM,
        ENCRYPTED_CONTENT;

        @Override
        public int getValue() {
            return ordinal();
        }

        @Override
        public String getName() {
            return name();
        }
    }

    static Asn1FieldInfo[] fieldInfos = new Asn1FieldInfo[] {
            new Asn1FieldInfo(ECInfoField.CONTENT_TYPE, Asn1ObjectIdentifier.class),
            new Asn1FieldInfo(ECInfoField.CONTENT_ENCRYPTION_ALGORITHM, ContentEncryptionAlgorithmIdentifier.class),
            new ImplicitField(ECInfoField.ENCRYPTED_CONTENT, 0, Asn1OctetString.class)
    };

    public EncryptedContentInfo() {
        super(fieldInfos);
    }

    public Asn1ObjectIdentifier getContentType() {
        return getFieldAs(ECInfoField.CONTENT_TYPE, Asn1ObjectIdentifier.class);
    }

    public void setContentType(Asn1ObjectIdentifier contentType) {
        setFieldAs(ECInfoField.CONTENT_TYPE, contentType);
    }

    public ContentEncryptionAlgorithmIdentifier getContentEncryptionAlgorithmIdentifier() {
        return getFieldAs(ECInfoField.CONTENT_ENCRYPTION_ALGORITHM, ContentEncryptionAlgorithmIdentifier.class);
    }

    public void setContentEncryptionAlgorithmIdentifier(ContentEncryptionAlgorithmIdentifier
                                                                contentEncryptionAlgorithmIdentifier) {
        setFieldAs(ECInfoField.CONTENT_ENCRYPTION_ALGORITHM, contentEncryptionAlgorithmIdentifier);
    }

    public Asn1OctetString getEncryptedContent() {
        return getFieldAs(ECInfoField.ENCRYPTED_CONTENT, Asn1OctetString.class);
    }

    public void setEncryptedContent(Asn1OctetString encryptedContent) {
        setFieldAs(ECInfoField.ENCRYPTED_CONTENT, encryptedContent);
    }
}
