/**
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package io.trino.hadoop.$internal.org.apache.kerby.cms.type;

import io.trino.hadoop.$internal.org.apache.kerby.asn1.Asn1FieldInfo;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.EnumType;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.type.Asn1Integer;
import io.trino.hadoop.$internal.org.apache.kerby.asn1.type.Asn1SequenceType;
import io.trino.hadoop.$internal.org.apache.kerby.x500.type.Name;

/**
 * Ref. RFC5652
 * <pre>
 * IssuerAndSerialNumber ::= SEQUENCE {
 *     issuer Name,
 *     serialNumber CertificateSerialNumber
 * }
 *
 * CertificateSerialNumber ::= INTEGER  -- See RFC 5280
 * </pre>
 */
public class IssuerAndSerialNumber extends Asn1SequenceType {
    protected enum IssuerAndSerialNumberField implements EnumType {
        ISSUER,
        SERIAL_NUMBER;

        @Override
        public int getValue() {
            return ordinal();
        }

        @Override
        public String getName() {
            return name();
        }
    }

    static Asn1FieldInfo[] fieldInfos = new Asn1FieldInfo[]{
        new Asn1FieldInfo(IssuerAndSerialNumberField.ISSUER, Name.class),
        new Asn1FieldInfo(IssuerAndSerialNumberField.SERIAL_NUMBER, Asn1Integer.class)
    };

    public IssuerAndSerialNumber() {
        super(fieldInfos);
    }

    public Name getIssuer() {
        return getFieldAs(IssuerAndSerialNumberField.ISSUER, Name.class);
    }

    public void setIssuer(Name name) {
        setFieldAs(IssuerAndSerialNumberField.ISSUER, name);
    }

    public Asn1Integer getSerialNumber() {
        return getFieldAs(IssuerAndSerialNumberField.SERIAL_NUMBER, Asn1Integer.class);
    }

    public void setSerialNumber(int serialNumber) {
        setFieldAsInt(IssuerAndSerialNumberField.SERIAL_NUMBER, serialNumber);
    }
}
