/**
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *
 */
package io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.client.preauth.builtin;

import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.KrbCodec;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.KrbException;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.client.preauth.AbstractPreauthPlugin;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.client.request.KdcRequest;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.common.EncryptionUtil;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.preauth.PaFlag;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.preauth.PaFlags;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.preauth.PluginRequestContext;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.preauth.builtin.EncTsPreauthMeta;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.base.EncryptedData;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.base.KeyUsage;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.pa.PaData;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.pa.PaDataEntry;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.pa.PaDataType;
import io.trino.hadoop.$internal.org.apache.kerby.kerberos.kerb.type.pa.PaEncTsEnc;

public class EncTsPreauth extends AbstractPreauthPlugin {

    public EncTsPreauth() {
        super(new EncTsPreauthMeta());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void prepareQuestions(KdcRequest kdcRequest,
                                 PluginRequestContext requestContext) throws KrbException {

        kdcRequest.needAsKey();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void tryFirst(KdcRequest kdcRequest,
                         PluginRequestContext requestContext,
                         PaData outPadata) throws KrbException {

        if (kdcRequest.getAsKey() == null) {
            kdcRequest.needAsKey();
        }
        outPadata.addElement(makeEntry(kdcRequest));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean process(KdcRequest kdcRequest,
                           PluginRequestContext requestContext,
                           PaDataEntry inPadata,
                           PaData outPadata) throws KrbException {

        if (kdcRequest.getAsKey() == null) {
            kdcRequest.needAsKey();
        }
        outPadata.addElement(makeEntry(kdcRequest));

        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public PaFlags getFlags(PaDataType paType) {
        PaFlags paFlags = new PaFlags(0);
        paFlags.setFlag(PaFlag.PA_REAL);

        return paFlags;
    }

    /**
     * Make padata entry.
     *
     * @param kdcRequest The kdc request
     * @return PaDataEntry to be made.
     */
    private PaDataEntry makeEntry(KdcRequest kdcRequest) throws KrbException {
        PaEncTsEnc paTs = new PaEncTsEnc();
        paTs.setPaTimestamp(kdcRequest.getPreauthTime());

        EncryptedData paDataValue = EncryptionUtil.seal(paTs,
                kdcRequest.getAsKey(), KeyUsage.AS_REQ_PA_ENC_TS);
        PaDataEntry tsPaEntry = new PaDataEntry();
        tsPaEntry.setPaDataType(PaDataType.ENC_TIMESTAMP);
        tsPaEntry.setPaDataValue(KrbCodec.encode(paDataValue));

        return tsPaEntry;
    }
}
