/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.hadoop.$internal.org.apache.commons.configuration2.event;

import java.util.EventObject;

/**
 * <p>
 * The base class for all events generated by this library.
 * </p>
 * <p>
 * The events produced by objects in this library are arranged in an inheritance hierarchy. This base class defines some
 * basic properties common to all configuration events. Especially, an event has an {@link EventType} which describes
 * its semantics. The event type can also be used for filtering for events or for defining event listeners on a
 * fine-grained basis.
 * </p>
 *
 * @since 2.0
 */
public class Event extends EventObject {

    /**
     * The root event type for all configuration-related events. All specific event types have this type as super direct
     * (directly or indirectly).
     */
    public static final EventType<Event> ANY = new EventType<>(null, "ANY");

    private static final long serialVersionUID = -8168310049858198944L;

    /**
     * Constant for the format used in toString() for a property representation.
     */
    private static final String FMT_PROPERTY = " %s=%s";

    /**
     * Constant for the initial buffer size for the generation of the string representation.
     */
    private static final int BUF_SIZE = 256;

    /** The type of this event. */
    private final EventType<? extends Event> eventType;

    /**
     * Creates a new instance of {@code Event} and sets basic properties.
     *
     * @param source the object on which the Event initially occurred (must not be <b>null</b>)
     * @param evType the type of this event (must not be <b>null</b>)
     * @throws IllegalArgumentException if a required parameter is null
     */
    public Event(final Object source, final EventType<? extends Event> evType) {
        super(source);
        if (evType == null) {
            throw new IllegalArgumentException("Event type must not be null!");
        }
        eventType = evType;
    }

    /**
     * Returns the type of this event.
     *
     * @return the event type
     */
    public EventType<? extends Event> getEventType() {
        return eventType;
    }

    /**
     * Returns a string representation for this object. This string contains the event class and a list of all properties.
     *
     * @return a string for this object
     */
    @Override
    public String toString() {
        final StringBuilder buf = new StringBuilder(BUF_SIZE);
        buf.append(getClass().getSimpleName());
        buf.append(" [");
        appendPropertyRepresentation(buf, "source", getSource());
        appendPropertyRepresentation(buf, "eventType", getEventType());
        buf.append(" ]");
        return buf.toString();
    }

    /**
     * Helper method for appending a representation for a property to the overall string representation for this object.
     * This method is called by {@code toString()} for generating string fragments for the properties of this class. It can
     * also be used by derived classes which extend the string representation of this base class.
     *
     * @param buf the target buffer
     * @param property the name of the property
     * @param value the property value
     */
    protected void appendPropertyRepresentation(final StringBuilder buf, final String property, final Object value) {
        buf.append(String.format(FMT_PROPERTY, property, String.valueOf(value)));
    }
}
