/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package io.trino.hadoop.$internal.org.apache.http.impl.client;

import java.net.URI;
import java.net.URISyntaxException;

import io.trino.hadoop.$internal.org.apache.commons.logging.Log;
import io.trino.hadoop.$internal.org.apache.commons.logging.LogFactory;
import io.trino.hadoop.$internal.org.apache.http.Header;
import io.trino.hadoop.$internal.org.apache.http.HttpHost;
import io.trino.hadoop.$internal.org.apache.http.HttpRequest;
import io.trino.hadoop.$internal.org.apache.http.HttpResponse;
import io.trino.hadoop.$internal.org.apache.http.HttpStatus;
import io.trino.hadoop.$internal.org.apache.http.ProtocolException;
import io.trino.hadoop.$internal.org.apache.http.annotation.Contract;
import io.trino.hadoop.$internal.org.apache.http.annotation.ThreadingBehavior;
import io.trino.hadoop.$internal.org.apache.http.client.CircularRedirectException;
import io.trino.hadoop.$internal.org.apache.http.client.RedirectHandler;
import io.trino.hadoop.$internal.org.apache.http.client.methods.HttpGet;
import io.trino.hadoop.$internal.org.apache.http.client.methods.HttpHead;
import io.trino.hadoop.$internal.org.apache.http.client.params.ClientPNames;
import io.trino.hadoop.$internal.org.apache.http.client.utils.URIUtils;
import io.trino.hadoop.$internal.org.apache.http.params.HttpParams;
import io.trino.hadoop.$internal.org.apache.http.protocol.ExecutionContext;
import io.trino.hadoop.$internal.org.apache.http.protocol.HttpContext;
import io.trino.hadoop.$internal.org.apache.http.util.Args;
import io.trino.hadoop.$internal.org.apache.http.util.Asserts;

/**
 * Default implementation of {@link RedirectHandler}.
 *
 * @since 4.0
 *
 * @deprecated (4.1)  use {@link DefaultRedirectStrategy}.
 */
@Contract(threading = ThreadingBehavior.IMMUTABLE)
@Deprecated
public class DefaultRedirectHandler implements RedirectHandler {

    private final Log log = LogFactory.getLog(getClass());

    private static final String REDIRECT_LOCATIONS = "http.protocol.redirect-locations";

    public DefaultRedirectHandler() {
        super();
    }

    @Override
    public boolean isRedirectRequested(
            final HttpResponse response,
            final HttpContext context) {
        Args.notNull(response, "HTTP response");

        final int statusCode = response.getStatusLine().getStatusCode();
        switch (statusCode) {
        case HttpStatus.SC_MOVED_TEMPORARILY:
        case HttpStatus.SC_MOVED_PERMANENTLY:
        case HttpStatus.SC_TEMPORARY_REDIRECT:
            final HttpRequest request = (HttpRequest) context.getAttribute(
                    ExecutionContext.HTTP_REQUEST);
            final String method = request.getRequestLine().getMethod();
            return method.equalsIgnoreCase(HttpGet.METHOD_NAME)
                || method.equalsIgnoreCase(HttpHead.METHOD_NAME);
        case HttpStatus.SC_SEE_OTHER:
            return true;
        default:
            return false;
        } //end of switch
    }

    @Override
    public URI getLocationURI(
            final HttpResponse response,
            final HttpContext context) throws ProtocolException {
        Args.notNull(response, "HTTP response");
        //get the location header to find out where to redirect to
        final Header locationHeader = response.getFirstHeader("location");
        if (locationHeader == null) {
            // got a redirect response, but no location header
            throw new ProtocolException(
                    "Received redirect response " + response.getStatusLine()
                    + " but no location header");
        }
        final String location = locationHeader.getValue();
        if (this.log.isDebugEnabled()) {
            this.log.debug("Redirect requested to location '" + location + "'");
        }

        URI uri;
        try {
            uri = new URI(location);
        } catch (final URISyntaxException ex) {
            throw new ProtocolException("Invalid redirect URI: " + location, ex);
        }

        final HttpParams params = response.getParams();
        // rfc2616 demands the location value be a complete URI
        // Location       = "Location" ":" absoluteURI
        if (!uri.isAbsolute()) {
            if (params.isParameterTrue(ClientPNames.REJECT_RELATIVE_REDIRECT)) {
                throw new ProtocolException("Relative redirect location '"
                        + uri + "' not allowed");
            }
            // Adjust location URI
            final HttpHost target = (HttpHost) context.getAttribute(
                    ExecutionContext.HTTP_TARGET_HOST);
            Asserts.notNull(target, "Target host");

            final HttpRequest request = (HttpRequest) context.getAttribute(
                    ExecutionContext.HTTP_REQUEST);

            try {
                final URI requestURI = new URI(request.getRequestLine().getUri());
                final URI absoluteRequestURI = URIUtils.rewriteURI(requestURI, target, URIUtils.DROP_FRAGMENT_AND_NORMALIZE);
                uri = URIUtils.resolve(absoluteRequestURI, uri);
            } catch (final URISyntaxException ex) {
                throw new ProtocolException(ex.getMessage(), ex);
            }
        }

        if (params.isParameterFalse(ClientPNames.ALLOW_CIRCULAR_REDIRECTS)) {

            RedirectLocations redirectLocations = (RedirectLocations) context.getAttribute(
                    REDIRECT_LOCATIONS);

            if (redirectLocations == null) {
                redirectLocations = new RedirectLocations();
                context.setAttribute(REDIRECT_LOCATIONS, redirectLocations);
            }

            final URI redirectURI;
            if (uri.getFragment() != null) {
                try {
                    final HttpHost target = new HttpHost(
                            uri.getHost(),
                            uri.getPort(),
                            uri.getScheme());
                    redirectURI = URIUtils.rewriteURI(uri, target, URIUtils.DROP_FRAGMENT_AND_NORMALIZE);
                } catch (final URISyntaxException ex) {
                    throw new ProtocolException(ex.getMessage(), ex);
                }
            } else {
                redirectURI = uri;
            }

            if (redirectLocations.contains(redirectURI)) {
                throw new CircularRedirectException("Circular redirect to '" +
                        redirectURI + "'");
            } else {
                redirectLocations.add(redirectURI);
            }
        }

        return uri;
    }

}
