/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.accumulo;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import com.google.inject.Provider;
import io.airlift.json.JsonCodec;
import io.airlift.json.JsonCodecFactory;
import io.airlift.json.ObjectMapperProvider;
import io.airlift.slice.Slice;
import io.trino.plugin.accumulo.AccumuloErrorCode;
import io.trino.plugin.accumulo.AccumuloMetadataManager;
import io.trino.plugin.accumulo.metadata.AccumuloTable;
import io.trino.plugin.accumulo.model.AccumuloColumnHandle;
import io.trino.plugin.accumulo.model.AccumuloTableHandle;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorInsertTableHandle;
import io.trino.spi.connector.ConnectorMetadata;
import io.trino.spi.connector.ConnectorOutputMetadata;
import io.trino.spi.connector.ConnectorOutputTableHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableHandle;
import io.trino.spi.connector.ConnectorTableLayout;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.ConnectorViewDefinition;
import io.trino.spi.connector.Constraint;
import io.trino.spi.connector.ConstraintApplicationResult;
import io.trino.spi.connector.RetryMode;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.SchemaTablePrefix;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.security.TrinoPrincipal;
import io.trino.spi.statistics.ComputedStatistics;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

public class AccumuloMetadata
implements ConnectorMetadata {
    private static final JsonCodec<ConnectorViewDefinition> VIEW_CODEC = new JsonCodecFactory((Provider)new ObjectMapperProvider()).jsonCodec(ConnectorViewDefinition.class);
    private final AccumuloMetadataManager metadataManager;
    private final AtomicReference<Runnable> rollbackAction = new AtomicReference();

    @Inject
    public AccumuloMetadata(AccumuloMetadataManager metadataManager) {
        this.metadataManager = Objects.requireNonNull(metadataManager, "metadataManager is null");
    }

    public void createSchema(ConnectorSession session, String schemaName, Map<String, Object> properties, TrinoPrincipal owner) {
        Preconditions.checkArgument((boolean)properties.isEmpty(), (Object)"Can't have properties for schema creation");
        this.metadataManager.createSchema(schemaName);
    }

    public void dropSchema(ConnectorSession session, String schemaName, boolean cascade) {
        if (cascade) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "This connector does not support dropping schemas with CASCADE option");
        }
        this.metadataManager.dropSchema(schemaName);
    }

    public ConnectorOutputTableHandle beginCreateTable(ConnectorSession session, ConnectorTableMetadata tableMetadata, Optional<ConnectorTableLayout> layout, RetryMode retryMode) {
        if (retryMode != RetryMode.NO_RETRIES) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "This connector does not support query retries");
        }
        if (tableMetadata.getComment().isPresent()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "This connector does not support creating tables with table comment");
        }
        this.checkNoRollback();
        SchemaTableName tableName = tableMetadata.getTable();
        AccumuloTable table = this.metadataManager.createTable(tableMetadata);
        AccumuloTableHandle handle = new AccumuloTableHandle(tableName.getSchemaName(), tableName.getTableName(), table.getRowId(), table.isExternal(), table.getSerializerClassName(), table.getScanAuthorizations());
        this.setRollback(() -> this.rollbackCreateTable(table));
        return handle;
    }

    public Optional<ConnectorOutputMetadata> finishCreateTable(ConnectorSession session, ConnectorOutputTableHandle tableHandle, Collection<Slice> fragments, Collection<ComputedStatistics> computedStatistics) {
        this.clearRollback();
        return Optional.empty();
    }

    private void rollbackCreateTable(AccumuloTable table) {
        this.metadataManager.dropTable(table);
    }

    public void createTable(ConnectorSession session, ConnectorTableMetadata tableMetadata, boolean ignoreExisting) {
        if (tableMetadata.getComment().isPresent()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "This connector does not support creating tables with table comment");
        }
        this.metadataManager.createTable(tableMetadata);
    }

    public void dropTable(ConnectorSession session, ConnectorTableHandle tableHandle) {
        AccumuloTableHandle handle = (AccumuloTableHandle)tableHandle;
        AccumuloTable table = this.metadataManager.getTable(handle.toSchemaTableName());
        if (table != null) {
            this.metadataManager.dropTable(table);
        }
    }

    public void renameTable(ConnectorSession session, ConnectorTableHandle tableHandle, SchemaTableName newTableName) {
        if (this.metadataManager.getTable(newTableName) != null) {
            throw new TrinoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, "Table " + String.valueOf(newTableName) + " already exists");
        }
        AccumuloTableHandle handle = (AccumuloTableHandle)tableHandle;
        this.metadataManager.renameTable(handle.toSchemaTableName(), newTableName);
    }

    public void createView(ConnectorSession session, SchemaTableName viewName, ConnectorViewDefinition definition, boolean replace) {
        String viewData = VIEW_CODEC.toJson((Object)definition);
        if (replace) {
            this.metadataManager.createOrReplaceView(viewName, viewData);
        } else {
            this.metadataManager.createView(viewName, viewData);
        }
    }

    public void dropView(ConnectorSession session, SchemaTableName viewName) {
        this.metadataManager.dropView(viewName);
    }

    public Optional<ConnectorViewDefinition> getView(ConnectorSession session, SchemaTableName viewName) {
        return Optional.ofNullable(this.metadataManager.getView(viewName)).map(view -> (ConnectorViewDefinition)VIEW_CODEC.fromJson(view.getData()));
    }

    public List<SchemaTableName> listViews(ConnectorSession session, Optional<String> schemaName) {
        return this.listViews(schemaName);
    }

    private List<SchemaTableName> listViews(Optional<String> filterSchema) {
        ImmutableList.Builder builder = ImmutableList.builder();
        if (filterSchema.isPresent()) {
            for (String view : this.metadataManager.getViewNames(filterSchema.get())) {
                builder.add((Object)new SchemaTableName(filterSchema.get(), view));
            }
        } else {
            for (String schemaName : this.metadataManager.getSchemaNames()) {
                for (String view : this.metadataManager.getViewNames(schemaName)) {
                    builder.add((Object)new SchemaTableName(schemaName, view));
                }
            }
        }
        return builder.build();
    }

    public ConnectorInsertTableHandle beginInsert(ConnectorSession session, ConnectorTableHandle tableHandle, List<ColumnHandle> columns, RetryMode retryMode) {
        if (retryMode != RetryMode.NO_RETRIES) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "This connector does not support query retries");
        }
        this.checkNoRollback();
        AccumuloTableHandle handle = (AccumuloTableHandle)tableHandle;
        this.setRollback(() -> AccumuloMetadata.rollbackInsert(handle));
        return handle;
    }

    public Optional<ConnectorOutputMetadata> finishInsert(ConnectorSession session, ConnectorInsertTableHandle insertHandle, Collection<Slice> fragments, Collection<ComputedStatistics> computedStatistics) {
        this.clearRollback();
        return Optional.empty();
    }

    private static void rollbackInsert(ConnectorInsertTableHandle insertHandle) {
        AccumuloTableHandle handle = (AccumuloTableHandle)insertHandle;
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Unable to rollback insert for table %s.%s. Some rows may have been written. Please run your insert again.", handle.getSchema(), handle.getTable()));
    }

    public ConnectorTableHandle getTableHandle(ConnectorSession session, SchemaTableName tableName) {
        if (!this.listSchemaNames(session).contains(tableName.getSchemaName().toLowerCase(Locale.ENGLISH))) {
            return null;
        }
        if (!this.listViews(session, Optional.of(tableName.getSchemaName())).contains(tableName)) {
            AccumuloTable table = this.metadataManager.getTable(tableName);
            if (table == null) {
                return null;
            }
            return new AccumuloTableHandle(table.getSchema(), table.getTable(), table.getRowId(), table.isExternal(), table.getSerializerClassName(), table.getScanAuthorizations());
        }
        return null;
    }

    public ConnectorTableMetadata getTableMetadata(ConnectorSession session, ConnectorTableHandle table) {
        AccumuloTableHandle handle = (AccumuloTableHandle)table;
        SchemaTableName tableName = new SchemaTableName(handle.getSchema(), handle.getTable());
        ConnectorTableMetadata metadata = this.getTableMetadata(tableName);
        if (metadata == null) {
            throw new TableNotFoundException(tableName);
        }
        return metadata;
    }

    public Map<String, ColumnHandle> getColumnHandles(ConnectorSession session, ConnectorTableHandle tableHandle) {
        AccumuloTableHandle handle = (AccumuloTableHandle)tableHandle;
        AccumuloTable table = this.metadataManager.getTable(handle.toSchemaTableName());
        if (table == null) {
            throw new TableNotFoundException(handle.toSchemaTableName());
        }
        ImmutableMap.Builder columnHandles = ImmutableMap.builder();
        for (AccumuloColumnHandle column : table.getColumns()) {
            columnHandles.put((Object)column.getName(), (Object)column);
        }
        return columnHandles.buildOrThrow();
    }

    public ColumnMetadata getColumnMetadata(ConnectorSession session, ConnectorTableHandle tableHandle, ColumnHandle columnHandle) {
        return ((AccumuloColumnHandle)columnHandle).getColumnMetadata();
    }

    public void renameColumn(ConnectorSession session, ConnectorTableHandle tableHandle, ColumnHandle source, String target) {
        AccumuloTableHandle handle = (AccumuloTableHandle)tableHandle;
        AccumuloColumnHandle columnHandle = (AccumuloColumnHandle)source;
        AccumuloTable table = this.metadataManager.getTable(handle.toSchemaTableName());
        if (table == null) {
            throw new TableNotFoundException(new SchemaTableName(handle.getSchema(), handle.getTable()));
        }
        this.metadataManager.renameColumn(table, columnHandle.getName(), target);
    }

    public List<String> listSchemaNames(ConnectorSession session) {
        return ImmutableList.copyOf(this.metadataManager.getSchemaNames());
    }

    public List<SchemaTableName> listTables(ConnectorSession session, Optional<String> filterSchema) {
        Set schemaNames = filterSchema.map(ImmutableSet::of).orElseGet(this.metadataManager::getSchemaNames);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (String schemaName : schemaNames) {
            for (String tableName : this.metadataManager.getTableNames(schemaName)) {
                builder.add((Object)new SchemaTableName(schemaName, tableName));
            }
        }
        builder.addAll(this.listViews(session, filterSchema));
        return builder.build().asList();
    }

    public Map<SchemaTableName, List<ColumnMetadata>> listTableColumns(ConnectorSession session, SchemaTablePrefix prefix) {
        Objects.requireNonNull(prefix, "prefix is null");
        ImmutableMap.Builder columns = ImmutableMap.builder();
        for (SchemaTableName tableName : this.listTables(session, prefix)) {
            ConnectorTableMetadata tableMetadata = this.getTableMetadata(tableName);
            if (tableMetadata == null) continue;
            columns.put((Object)tableName, (Object)tableMetadata.getColumns());
        }
        return columns.buildOrThrow();
    }

    public Optional<ConstraintApplicationResult<ConnectorTableHandle>> applyFilter(ConnectorSession session, ConnectorTableHandle table, Constraint constraint) {
        TupleDomain newDomain;
        AccumuloTableHandle handle = (AccumuloTableHandle)table;
        TupleDomain<ColumnHandle> oldDomain = handle.getConstraint();
        if (oldDomain.equals((Object)(newDomain = oldDomain.intersect(constraint.getSummary())))) {
            return Optional.empty();
        }
        handle = new AccumuloTableHandle(handle.getSchema(), handle.getTable(), handle.getRowId(), (TupleDomain<ColumnHandle>)newDomain, handle.isExternal(), handle.getSerializerClassName(), handle.getScanAuthorizations());
        return Optional.of(new ConstraintApplicationResult((Object)handle, constraint.getSummary(), constraint.getExpression(), false));
    }

    private void checkNoRollback() {
        Preconditions.checkState((this.rollbackAction.get() == null ? 1 : 0) != 0, (Object)"Cannot begin a new write while in an existing one");
    }

    private void setRollback(Runnable action) {
        Preconditions.checkState((boolean)this.rollbackAction.compareAndSet(null, action), (Object)"Should not have to override existing rollback action");
    }

    private void clearRollback() {
        this.rollbackAction.set(null);
    }

    public void rollback() {
        Runnable rollbackAction = this.rollbackAction.getAndSet(null);
        if (rollbackAction != null) {
            rollbackAction.run();
        }
    }

    private ConnectorTableMetadata getTableMetadata(SchemaTableName tableName) {
        if (!this.metadataManager.getSchemaNames().contains(tableName.getSchemaName())) {
            return null;
        }
        if (!this.listViews(Optional.ofNullable(tableName.getSchemaName())).contains(tableName)) {
            AccumuloTable table = this.metadataManager.getTable(tableName);
            if (table == null) {
                return null;
            }
            return new ConnectorTableMetadata(tableName, table.getColumnsMetadata());
        }
        return null;
    }

    private List<SchemaTableName> listTables(ConnectorSession session, SchemaTablePrefix prefix) {
        if (prefix.getTable().isEmpty()) {
            return this.listTables(session, prefix.getSchema());
        }
        SchemaTableName table = prefix.toSchemaTableName();
        if (this.getTableHandle(session, table) != null) {
            return ImmutableList.of((Object)table);
        }
        return ImmutableList.of();
    }
}

