/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.cache.CacheStats;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.MoreCollectors;
import com.google.common.util.concurrent.Futures;
import io.airlift.concurrent.Threads;
import io.airlift.units.Duration;
import io.trino.plugin.base.session.SessionPropertiesProvider;
import io.trino.plugin.jdbc.CachingJdbcClient;
import io.trino.plugin.jdbc.ExtraCredentialsBasedIdentityCacheMapping;
import io.trino.plugin.jdbc.ForwardingJdbcClient;
import io.trino.plugin.jdbc.IdentityCacheMapping;
import io.trino.plugin.jdbc.JdbcClient;
import io.trino.plugin.jdbc.JdbcColumnHandle;
import io.trino.plugin.jdbc.JdbcProcedureHandle;
import io.trino.plugin.jdbc.JdbcQueryRelationHandle;
import io.trino.plugin.jdbc.JdbcRelationHandle;
import io.trino.plugin.jdbc.JdbcTableHandle;
import io.trino.plugin.jdbc.PreparedQuery;
import io.trino.plugin.jdbc.SingletonIdentityCacheMapping;
import io.trino.plugin.jdbc.TestingDatabase;
import io.trino.plugin.jdbc.credential.ExtraCredentialConfig;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.security.ConnectorIdentity;
import io.trino.spi.session.PropertyMetadata;
import io.trino.spi.statistics.Estimate;
import io.trino.spi.statistics.TableStatistics;
import io.trino.spi.testing.InterfaceTestUtils;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.Type;
import io.trino.testing.TestingConnectorSession;
import io.trino.testing.TestingNames;
import io.trino.testing.assertions.Assert;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

@Test(singleThreaded=true)
public class TestCachingJdbcClient {
    private static final Duration FOREVER = Duration.succinctDuration((double)1.0, (TimeUnit)TimeUnit.DAYS);
    private static final Duration ZERO = Duration.succinctDuration((double)0.0, (TimeUnit)TimeUnit.MILLISECONDS);
    private static final ImmutableList<PropertyMetadata<?>> PROPERTY_METADATA = ImmutableList.of((Object)PropertyMetadata.stringProperty((String)"session_name", (String)"Session name", null, (boolean)false));
    private static final Set<SessionPropertiesProvider> SESSION_PROPERTIES_PROVIDERS = Set.of(() -> PROPERTY_METADATA);
    private static final ConnectorSession SESSION = TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).build();
    private static final TableStatistics NON_EMPTY_STATS = TableStatistics.builder().setRowCount(Estimate.zero()).build();
    private TestingDatabase database;
    private CachingJdbcClient cachingJdbcClient;
    private JdbcClient jdbcClient;
    private String schema;
    private ExecutorService executor;

    @BeforeMethod
    public void setUp() throws Exception {
        this.database = new TestingDatabase();
        this.cachingJdbcClient = this.createCachingJdbcClient(true, 10000L);
        this.jdbcClient = this.database.getJdbcClient();
        this.schema = (String)this.jdbcClient.getSchemaNames(SESSION).iterator().next();
        this.executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)"TestCachingJdbcClient-%s"));
    }

    private CachingJdbcClient createCachingJdbcClient(Duration cacheTtl, boolean cacheMissing, long cacheMaximumSize) {
        return this.createCachingJdbcClient(cacheTtl, cacheTtl, cacheTtl, cacheMissing, cacheMaximumSize);
    }

    private CachingJdbcClient createCachingJdbcClient(Duration cacheTtl, Duration schemasCacheTtl, Duration tablesCacheTtl, boolean cacheMissing, long cacheMaximumSize) {
        return new CachingJdbcClient(this.database.getJdbcClient(), SESSION_PROPERTIES_PROVIDERS, (IdentityCacheMapping)new SingletonIdentityCacheMapping(), cacheTtl, schemasCacheTtl, tablesCacheTtl, cacheMissing, cacheMaximumSize);
    }

    private CachingJdbcClient createCachingJdbcClient(boolean cacheMissing, long cacheMaximumSize) {
        return this.createCachingJdbcClient(FOREVER, cacheMissing, cacheMaximumSize);
    }

    @AfterMethod(alwaysRun=true)
    public void tearDown() throws Exception {
        this.executor.shutdownNow();
        this.executor = null;
        this.database.close();
        this.database = null;
    }

    @Test
    public void testSchemaNamesCached() {
        String phantomSchema = "phantom_schema";
        this.jdbcClient.createSchema(SESSION, phantomSchema);
        TestCachingJdbcClient.assertSchemaNamesCache(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Collection)this.cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema}));
        this.jdbcClient.dropSchema(SESSION, phantomSchema);
        Assertions.assertThat((Collection)this.jdbcClient.getSchemaNames(SESSION)).doesNotContain((Object[])new String[]{phantomSchema});
        TestCachingJdbcClient.assertSchemaNamesCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Collection)this.cachingJdbcClient.getSchemaNames(SESSION)).contains((Object[])new String[]{phantomSchema}));
    }

    @Test
    public void testTableNamesCached() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        TestCachingJdbcClient.assertTableNamesCache(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable}));
        this.dropTable(phantomTable);
        Assertions.assertThat((List)this.jdbcClient.getTableNames(SESSION, Optional.of(this.schema))).doesNotContain((Object[])new SchemaTableName[]{phantomTable});
        TestCachingJdbcClient.assertTableNamesCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getTableNames(SESSION, Optional.of(this.schema))).contains((Object[])new SchemaTableName[]{phantomTable}));
    }

    @Test
    public void testTableHandleCached() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        Optional cachedTable = this.cachingJdbcClient.getTableHandle(SESSION, phantomTable);
        this.dropTable(phantomTable);
        Assertions.assertThat((Optional)this.jdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEqualTo((Object)cachedTable);
    }

    @Test
    public void testTableHandleOfQueryCached() throws Exception {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        PreparedQuery query = new PreparedQuery(String.format("SELECT * FROM %s.phantom_table", this.schema), (List)ImmutableList.of());
        JdbcTableHandle cachedTable = TestCachingJdbcClient.assertTableHandleByQueryCache(this.cachingJdbcClient).misses(1L).loads(1L).calling(() -> this.cachingJdbcClient.getTableHandle(SESSION, query));
        TestCachingJdbcClient.assertCacheStats(this.cachingJdbcClient).afterRunning(() -> this.cachingJdbcClient.getColumns(SESSION, cachedTable));
        TestCachingJdbcClient.assertStatisticsCacheStats(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> this.cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.dropTable(phantomTable);
        Assertions.assertThatThrownBy(() -> this.jdbcClient.getTableHandle(SESSION, query)).hasMessageContaining("Failed to get table handle for prepared query");
        TestCachingJdbcClient.assertTableHandleByQueryCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> {
            Assertions.assertThat((Object)this.cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(this.cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(this.cachingJdbcClient).hits(1L).afterRunning(() -> this.cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.cachingJdbcClient.createTable(SESSION, new ConnectorTableMetadata(phantomTable, Collections.emptyList()));
        TestCachingJdbcClient.assertTableHandleByQueryCache(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> {
            Assertions.assertThat((Object)this.cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(this.cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> this.cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.cachingJdbcClient.onDataChanged(phantomTable);
        TestCachingJdbcClient.assertTableHandleByQueryCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> {
            Assertions.assertThat((Object)this.cachingJdbcClient.getTableHandle(SESSION, query)).isEqualTo((Object)cachedTable);
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0);
        });
        TestCachingJdbcClient.assertCacheStats(this.cachingJdbcClient).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, cachedTable)).hasSize(0));
        TestCachingJdbcClient.assertStatisticsCacheStats(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> this.cachingJdbcClient.getTableStatistics(SESSION, cachedTable));
        this.dropTable(phantomTable);
    }

    @Test
    public void testProcedureHandleCached() throws Exception {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        this.createTable(phantomTable);
        this.createProcedure("test_procedure");
        JdbcProcedureHandle.ProcedureQuery query = new JdbcProcedureHandle.ProcedureQuery("CALL %s.test_procedure ('%s')".formatted(this.schema, phantomTable));
        JdbcProcedureHandle cachedProcedure = TestCachingJdbcClient.assertProcedureHandleByQueryCache(this.cachingJdbcClient).misses(1L).loads(1L).calling(() -> this.cachingJdbcClient.getProcedureHandle(SESSION, query));
        Assertions.assertThat((List)((List)cachedProcedure.getColumns().orElseThrow())).hasSize(0);
        this.dropProcedure("test_procedure");
        Assertions.assertThatThrownBy(() -> this.jdbcClient.getProcedureHandle(SESSION, query)).hasMessageContaining("Failed to get table handle for procedure query");
        TestCachingJdbcClient.assertProcedureHandleByQueryCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)this.cachingJdbcClient.getProcedureHandle(SESSION, query)).isEqualTo((Object)cachedProcedure));
        this.dropTable(phantomTable);
    }

    @Test
    public void testTableHandleInvalidatedOnColumnsModifications() {
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "a_table"));
        JdbcColumnHandle existingColumn = this.addColumn(table, "a_column");
        this.assertTableHandlesByNameCacheIsInvalidated(table);
        JdbcColumnHandle newColumn = this.addColumn((JdbcClient)this.cachingJdbcClient, table, "new_column");
        this.assertTableHandlesByNameCacheIsInvalidated(table);
        this.cachingJdbcClient.setColumnComment(SESSION, table, newColumn, Optional.empty());
        this.assertTableHandlesByNameCacheIsInvalidated(table);
        this.cachingJdbcClient.renameColumn(SESSION, table, newColumn, "new_column_name");
        this.assertTableHandlesByNameCacheIsInvalidated(table);
        this.cachingJdbcClient.dropColumn(SESSION, table, existingColumn);
        this.assertTableHandlesByNameCacheIsInvalidated(table);
        this.dropTable(table);
    }

    private void assertTableHandlesByNameCacheIsInvalidated(JdbcTableHandle table) {
        SchemaTableName tableName = table.asPlainTable().getSchemaTableName();
        TestCachingJdbcClient.assertTableHandleByNameCache(this.cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Object)((JdbcTableHandle)this.cachingJdbcClient.getTableHandle(SESSION, tableName).orElseThrow())).isEqualTo((Object)table));
        TestCachingJdbcClient.assertTableHandleByNameCache(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)((JdbcTableHandle)this.cachingJdbcClient.getTableHandle(SESSION, tableName).orElseThrow())).isEqualTo((Object)table));
    }

    @Test
    public void testEmptyTableHandleIsCachedWhenCacheMissingIsTrue() {
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)this.cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.dropTable(phantomTable);
    }

    @Test
    public void testEmptyTableHandleNotCachedWhenCacheMissingIsFalse() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(false, 10000L);
        SchemaTableName phantomTable = new SchemaTableName(this.schema, "phantom_table");
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isEmpty();
        this.createTable(phantomTable);
        Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(SESSION, phantomTable)).isPresent();
        this.dropTable(phantomTable);
    }

    private JdbcTableHandle createTable(SchemaTableName phantomTable) {
        this.jdbcClient.createTable(SESSION, new ConnectorTableMetadata(phantomTable, Collections.emptyList()));
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
    }

    private void createProcedure(String procedureName) throws SQLException {
        try (Statement statement = this.database.getConnection().createStatement();){
            statement.execute("CREATE ALIAS %s.%s FOR \"io.trino.plugin.jdbc.TestCachingJdbcClient.generateData\"".formatted(this.schema, procedureName));
        }
    }

    private void dropProcedure(String procedureName) throws SQLException {
        try (Statement statement = this.database.getConnection().createStatement();){
            statement.execute("DROP ALIAS %s.%s".formatted(this.schema, procedureName));
        }
    }

    public static ResultSet generateData(Connection connection, String table) throws SQLException {
        return connection.createStatement().executeQuery("SELECT * FROM " + table);
    }

    private void dropTable(JdbcTableHandle tableHandle) {
        this.jdbcClient.dropTable(SESSION, tableHandle);
    }

    private void dropTable(SchemaTableName phantomTable) {
        JdbcTableHandle tableHandle = (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, phantomTable).orElseThrow();
        this.jdbcClient.dropTable(SESSION, tableHandle);
    }

    @Test
    public void testColumnsCached() {
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        this.jdbcClient.dropColumn(SESSION, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(SESSION, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(SESSION, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
    }

    @Test
    public void testColumnsCachedPerSession() {
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle table = this.getAnyTable(this.schema);
        JdbcColumnHandle phantomColumn = this.addColumn(table);
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).contains((Object[])new JdbcColumnHandle[]{phantomColumn}));
        this.cachingJdbcClient.dropColumn(firstSession, table, phantomColumn);
        Assertions.assertThat((List)this.jdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, table)).doesNotContain((Object[])new JdbcColumnHandle[]{phantomColumn});
        });
    }

    @Test
    public void testColumnsCacheInvalidationOnTableDrop() {
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, firstTable)).contains((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        this.cachingJdbcClient.renameColumn(firstSession, firstTable, firstColumn, "another_column");
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, firstTable)).doesNotContain((Object[])new JdbcColumnHandle[]{firstColumn})).containsAll((Iterable)this.jdbcClient.getColumns(SESSION, firstTable)));
        this.cachingJdbcClient.dropTable(secondSession, firstTable);
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThatThrownBy(() -> this.cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
            Assertions.assertThatThrownBy(() -> this.cachingJdbcClient.getColumns(secondSession, firstTable)).isInstanceOf(TableNotFoundException.class);
        });
        TestCachingJdbcClient.assertColumnCacheStats(this.cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(firstSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)this.cachingJdbcClient.getColumns(secondSession, secondTable)).contains((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        this.cachingJdbcClient.dropTable(secondSession, secondTable);
    }

    @Test
    public void testColumnsNotCachedWhenCacheDisabled() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(ZERO, true, 10000L);
        ConnectorSession firstSession = TestCachingJdbcClient.createSession("first");
        ConnectorSession secondSession = TestCachingJdbcClient.createSession("second");
        JdbcTableHandle firstTable = this.createTable(new SchemaTableName(this.schema, "first_table"));
        JdbcTableHandle secondTable = this.createTable(new SchemaTableName(this.schema, "second_table"));
        JdbcColumnHandle firstColumn = this.addColumn(firstTable, "first_column");
        JdbcColumnHandle secondColumn = this.addColumn(secondTable, "second_column");
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(4L).misses(4L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, firstTable)).containsExactly((Object[])new JdbcColumnHandle[]{firstColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(firstSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
            Assertions.assertThat((List)cachingJdbcClient.getColumns(secondSession, secondTable)).containsExactly((Object[])new JdbcColumnHandle[]{secondColumn});
        });
        this.jdbcClient.dropTable(SESSION, firstTable);
        this.jdbcClient.dropTable(SESSION, secondTable);
        TestCachingJdbcClient.assertColumnCacheStats(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, firstTable)).isInstanceOf(TableNotFoundException.class);
            Assertions.assertThatThrownBy(() -> cachingJdbcClient.getColumns(firstSession, secondTable)).isInstanceOf(TableNotFoundException.class);
        });
    }

    @Test
    public void testGetTableStatistics() {
        CachingJdbcClient cachingJdbcClient = this.cachingStatisticsAwareJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("first");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "first"));
        JdbcTableHandle second = this.createTable(new SchemaTableName(this.schema, "second"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, second)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, first);
        JdbcTableHandle secondFirst = this.createTable(new SchemaTableName(this.schema, "first"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, first);
        this.jdbcClient.dropTable(SESSION, second);
    }

    @Test
    public void testCacheGetTableStatisticsWithQueryRelationHandle() {
        CachingJdbcClient cachingJdbcClient = this.cachingStatisticsAwareJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("some test session name");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "first"));
        JdbcTableHandle second = this.createTable(new SchemaTableName(this.schema, "second"));
        JdbcTableHandle queryOnFirst = new JdbcTableHandle((JdbcRelationHandle)new JdbcQueryRelationHandle(new PreparedQuery("SELECT * FROM first", List.of())), TupleDomain.all(), (List)ImmutableList.of(), Optional.empty(), OptionalLong.empty(), Optional.empty(), Optional.of(Set.of(new SchemaTableName(this.schema, "first"))), 0, Optional.empty());
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, second);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.dropTable(SESSION, first);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, queryOnFirst)).isEqualTo((Object)NON_EMPTY_STATS));
    }

    @Test
    public void testTruncateTable() {
        CachingJdbcClient cachingJdbcClient = this.cachingStatisticsAwareJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.truncateTable(SESSION, table);
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, table);
    }

    private CachingJdbcClient cachingStatisticsAwareJdbcClient(Duration duration, boolean cacheMissing, long cacheMaximumSize) {
        final JdbcClient jdbcClient = this.database.getJdbcClient();
        ForwardingJdbcClient statsAwareJdbcClient = new ForwardingJdbcClient(){

            protected JdbcClient delegate() {
                return jdbcClient;
            }

            public TableStatistics getTableStatistics(ConnectorSession session, JdbcTableHandle handle) {
                return NON_EMPTY_STATS;
            }
        };
        return new CachingJdbcClient((JdbcClient)statsAwareJdbcClient, SESSION_PROPERTIES_PROVIDERS, (IdentityCacheMapping)new SingletonIdentityCacheMapping(), duration, cacheMissing, cacheMaximumSize);
    }

    @Test
    public void testCacheEmptyStatistics() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testGetTableStatisticsDoNotCacheEmptyWhenCachingMissingIsDisabled() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(FOREVER, false, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("table");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).hits(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, table)).isEqualTo((Object)TableStatistics.empty()));
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testDifferentIdentityKeys() {
        CachingJdbcClient cachingJdbcClient = new CachingJdbcClient(this.database.getJdbcClient(), SESSION_PROPERTIES_PROVIDERS, (IdentityCacheMapping)new ExtraCredentialsBasedIdentityCacheMapping(new ExtraCredentialConfig().setUserCredentialName("user").setPasswordCredentialName("password")), FOREVER, true, 10000L);
        ConnectorSession alice = TestCachingJdbcClient.createUserSession("alice");
        ConnectorSession bob = TestCachingJdbcClient.createUserSession("bob");
        JdbcTableHandle table = this.createTable(new SchemaTableName(this.schema, "table"));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(2L).misses(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(alice, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(bob, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
        });
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).hits(2L).afterRunning(() -> {
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(alice, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
            Assertions.assertThat((List)cachingJdbcClient.getTableNames(bob, Optional.empty())).contains((Object[])new SchemaTableName[]{table.getRequiredNamedRelation().getSchemaTableName()});
        });
        this.jdbcClient.dropTable(SESSION, table);
    }

    @Test
    public void testFlushCache() {
        CachingJdbcClient cachingJdbcClient = this.cachingStatisticsAwareJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        JdbcTableHandle first = this.createTable(new SchemaTableName(this.schema, "atable"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, first)).isEqualTo((Object)NON_EMPTY_STATS));
        cachingJdbcClient.flushCache();
        JdbcTableHandle secondFirst = this.createTable(new SchemaTableName(this.schema, "first"));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        TestCachingJdbcClient.assertStatisticsCacheStats(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Object)cachingJdbcClient.getTableStatistics(session, secondFirst)).isEqualTo((Object)NON_EMPTY_STATS));
        this.jdbcClient.dropTable(SESSION, first);
    }

    @Test(timeOut=60000L)
    public void testConcurrentSchemaCreateAndDrop() {
        CachingJdbcClient cachingJdbcClient = this.cachingStatisticsAwareJdbcClient(FOREVER, true, 10000L);
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        ArrayList<Future<Object>> futures = new ArrayList<Future<Object>>();
        for (int i = 0; i < 5; ++i) {
            futures.add(this.executor.submit(() -> {
                String schemaName = "schema_" + TestingNames.randomNameSuffix();
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).doesNotContain((Object[])new String[]{schemaName});
                cachingJdbcClient.createSchema(session, schemaName);
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{schemaName});
                cachingJdbcClient.dropSchema(session, schemaName);
                Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).doesNotContain((Object[])new String[]{schemaName});
                return null;
            }));
        }
        futures.forEach(Futures::getUnchecked);
    }

    @Test(timeOut=60000L)
    public void testLoadFailureNotSharedWhenDisabled() throws Exception {
        final AtomicBoolean first = new AtomicBoolean(true);
        CyclicBarrier barrier = new CyclicBarrier(2);
        CachingJdbcClient cachingJdbcClient = new CachingJdbcClient((JdbcClient)new ForwardingJdbcClient(){
            private final JdbcClient delegate;
            {
                this.delegate = TestCachingJdbcClient.this.database.getJdbcClient();
            }

            protected JdbcClient delegate() {
                return this.delegate;
            }

            public Optional<JdbcTableHandle> getTableHandle(ConnectorSession session, SchemaTableName schemaTableName) {
                if (first.compareAndSet(true, false)) {
                    try {
                        Thread.sleep(5L);
                    }
                    catch (InterruptedException e1) {
                        throw new RuntimeException(e1);
                    }
                    throw new RuntimeException("first attempt is poised to fail");
                }
                return super.getTableHandle(session, schemaTableName);
            }
        }, SESSION_PROPERTIES_PROVIDERS, (IdentityCacheMapping)new SingletonIdentityCacheMapping(), new Duration(0.0, TimeUnit.DAYS), true, 10L);
        SchemaTableName tableName = new SchemaTableName(this.schema, "test_load_failure_not_shared");
        this.createTable(tableName);
        ConnectorSession session = TestCachingJdbcClient.createSession("session");
        ArrayList<Future<JdbcTableHandle>> futures = new ArrayList<Future<JdbcTableHandle>>();
        for (int i = 0; i < 2; ++i) {
            futures.add(this.executor.submit(() -> {
                barrier.await(10L, TimeUnit.SECONDS);
                return (JdbcTableHandle)cachingJdbcClient.getTableHandle(session, tableName).orElseThrow();
            }));
        }
        ArrayList<String> results = new ArrayList<String>();
        for (Future future : futures) {
            try {
                results.add(((JdbcTableHandle)future.get()).toString());
            }
            catch (ExecutionException e) {
                results.add(e.getCause().toString());
            }
        }
        Assertions.assertThat(results).containsExactlyInAnyOrder((Object[])new String[]{"example.test_load_failure_not_shared " + this.database.getDatabaseName() + ".EXAMPLE.TEST_LOAD_FAILURE_NOT_SHARED", "com.google.common.util.concurrent.UncheckedExecutionException: java.lang.RuntimeException: first attempt is poised to fail"});
    }

    @Test
    public void testSpecificSchemaAndTableCaches() {
        CachingJdbcClient cachingJdbcClient = this.createCachingJdbcClient(FOREVER, Duration.succinctDuration((double)3.0, (TimeUnit)TimeUnit.SECONDS), Duration.succinctDuration((double)2.0, (TimeUnit)TimeUnit.SECONDS), false, 10000L);
        String secondSchema = this.schema + "_two";
        SchemaTableName firstName = new SchemaTableName(this.schema, "first_table");
        SchemaTableName secondName = new SchemaTableName(secondSchema, "second_table");
        ConnectorSession session = TestCachingJdbcClient.createSession("asession");
        JdbcTableHandle first = this.createTable(firstName);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName})).doesNotContain((Object[])new SchemaTableName[]{secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isEmpty());
        this.jdbcClient.createSchema(SESSION, secondSchema);
        JdbcTableHandle second = this.createTable(secondName);
        TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
        TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((ListAssert)Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName})).doesNotContain((Object[])new SchemaTableName[]{secondName}));
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
        TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).misses(1L).loads(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        Assert.assertEventually((Duration)Duration.succinctDuration((double)10.0, (TimeUnit)TimeUnit.SECONDS), () -> {
            TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).hits(1L).afterRunning(() -> ((AbstractCollectionAssert)Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema})).doesNotContain((Object[])new String[]{secondSchema}));
            TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName, secondName}));
            TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
            TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        });
        Assert.assertEventually((Duration)Duration.succinctDuration((double)10.0, (TimeUnit)TimeUnit.SECONDS), () -> {
            TestCachingJdbcClient.assertSchemaNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((Collection)cachingJdbcClient.getSchemaNames(session)).contains((Object[])new String[]{this.schema, secondSchema}));
            TestCachingJdbcClient.assertTableNamesCache(cachingJdbcClient).loads(1L).misses(1L).afterRunning(() -> Assertions.assertThat((List)cachingJdbcClient.getTableNames(session, Optional.empty())).contains((Object[])new SchemaTableName[]{firstName, secondName}));
            TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, firstName)).isNotEmpty());
            TestCachingJdbcClient.assertTableHandleByNameCache(cachingJdbcClient).hits(1L).afterRunning(() -> Assertions.assertThat((Optional)cachingJdbcClient.getTableHandle(session, secondName)).isNotEmpty());
        });
        this.jdbcClient.dropTable(SESSION, first);
        this.jdbcClient.dropTable(SESSION, second);
        this.jdbcClient.dropSchema(SESSION, secondSchema);
    }

    private JdbcTableHandle getAnyTable(String schema) {
        SchemaTableName tableName = this.jdbcClient.getTableNames(SESSION, Optional.of(schema)).stream().filter(schemaTableName -> !"public".equals(schemaTableName.getTableName())).findAny().orElseThrow();
        return (JdbcTableHandle)this.jdbcClient.getTableHandle(SESSION, tableName).orElseThrow();
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle) {
        return this.addColumn(tableHandle, "phantom_column");
    }

    private JdbcColumnHandle addColumn(JdbcTableHandle tableHandle, String columnName) {
        return this.addColumn(this.jdbcClient, tableHandle, columnName);
    }

    private JdbcColumnHandle addColumn(JdbcClient client, JdbcTableHandle tableHandle, String columnName) {
        ColumnMetadata columnMetadata = new ColumnMetadata(columnName, (Type)IntegerType.INTEGER);
        client.addColumn(SESSION, tableHandle, columnMetadata);
        return (JdbcColumnHandle)client.getColumns(SESSION, tableHandle).stream().filter(jdbcColumnHandle -> jdbcColumnHandle.getColumnMetadata().equals((Object)columnMetadata)).collect(MoreCollectors.onlyElement());
    }

    private static ConnectorSession createSession(String sessionName) {
        return TestingConnectorSession.builder().setPropertyMetadata(PROPERTY_METADATA).setPropertyValues((Map)ImmutableMap.of((Object)"session_name", (Object)sessionName)).build();
    }

    private static ConnectorSession createUserSession(String userName) {
        return TestingConnectorSession.builder().setIdentity(ConnectorIdentity.forUser((String)userName).withExtraCredentials((Map)ImmutableMap.of((Object)"user", (Object)userName)).build()).build();
    }

    @Test
    public void testEverythingImplemented() {
        InterfaceTestUtils.assertAllMethodsOverridden(JdbcClient.class, CachingJdbcClient.class);
    }

    private static SingleJdbcCacheStatsAssertions assertSchemaNamesCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.SCHEMA_NAMES_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableNamesCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_NAMES_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableHandleByNameCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_HANDLES_BY_NAME_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertTableHandleByQueryCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.TABLE_HANDLES_BY_QUERY_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertProcedureHandleByQueryCache(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.PROCEDURE_HANDLES_BY_QUERY_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertColumnCacheStats(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.COLUMNS_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertStatisticsCacheStats(CachingJdbcClient client) {
        return TestCachingJdbcClient.assertCacheStats(client, CachingJdbcCache.STATISTICS_CACHE);
    }

    private static SingleJdbcCacheStatsAssertions assertCacheStats(CachingJdbcClient client, CachingJdbcCache cache) {
        return new SingleJdbcCacheStatsAssertions(client, cache);
    }

    private static JdbcCacheStatsAssertions assertCacheStats(CachingJdbcClient client) {
        return new JdbcCacheStatsAssertions(client);
    }

    private static class SingleJdbcCacheStatsAssertions {
        private CachingJdbcCache chosenCache;
        private JdbcCacheStatsAssertions delegate;

        private SingleJdbcCacheStatsAssertions(CachingJdbcClient jdbcClient, CachingJdbcCache chosenCache) {
            this.chosenCache = Objects.requireNonNull(chosenCache, "chosenCache is null");
            this.delegate = new JdbcCacheStatsAssertions(jdbcClient);
        }

        public SingleJdbcCacheStatsAssertions loads(long value) {
            this.delegate.loads(this.chosenCache, value);
            return this;
        }

        public SingleJdbcCacheStatsAssertions hits(long value) {
            this.delegate.hits(this.chosenCache, value);
            return this;
        }

        public SingleJdbcCacheStatsAssertions misses(long value) {
            this.delegate.misses(this.chosenCache, value);
            return this;
        }

        public void afterRunning(Runnable runnable) {
            this.delegate.afterRunning(runnable);
        }

        public <T> T calling(Callable<T> callable) throws Exception {
            return this.delegate.calling(callable);
        }
    }

    private static class JdbcCacheStatsAssertions {
        private final CachingJdbcClient jdbcClient;
        private final Map<CachingJdbcCache, Long> loads = new HashMap<CachingJdbcCache, Long>();
        private final Map<CachingJdbcCache, Long> hits = new HashMap<CachingJdbcCache, Long>();
        private final Map<CachingJdbcCache, Long> misses = new HashMap<CachingJdbcCache, Long>();

        public JdbcCacheStatsAssertions(CachingJdbcClient jdbcClient) {
            this.jdbcClient = Objects.requireNonNull(jdbcClient, "jdbcClient is null");
        }

        public JdbcCacheStatsAssertions loads(CachingJdbcCache cache, long value) {
            this.loads.put(cache, value);
            return this;
        }

        public JdbcCacheStatsAssertions hits(CachingJdbcCache cache, long value) {
            this.hits.put(cache, value);
            return this;
        }

        public JdbcCacheStatsAssertions misses(CachingJdbcCache cache, long value) {
            this.misses.put(cache, value);
            return this;
        }

        public void afterRunning(Runnable runnable) {
            try {
                this.calling(() -> {
                    runnable.run();
                    return null;
                });
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        public <T> T calling(Callable<T> callable) throws Exception {
            Map beforeStats = (Map)Stream.of(CachingJdbcCache.values()).collect(ImmutableMap.toImmutableMap(Function.identity(), cache -> cache.statsGetter.apply(this.jdbcClient)));
            T value = callable.call();
            Map afterStats = (Map)Stream.of(CachingJdbcCache.values()).collect(ImmutableMap.toImmutableMap(Function.identity(), cache -> cache.statsGetter.apply(this.jdbcClient)));
            for (CachingJdbcCache cache2 : CachingJdbcCache.values()) {
                long loadDelta = ((CacheStats)afterStats.get((Object)cache2)).loadCount() - ((CacheStats)beforeStats.get((Object)cache2)).loadCount();
                long missesDelta = ((CacheStats)afterStats.get((Object)cache2)).missCount() - ((CacheStats)beforeStats.get((Object)cache2)).missCount();
                long hitsDelta = ((CacheStats)afterStats.get((Object)cache2)).hitCount() - ((CacheStats)beforeStats.get((Object)cache2)).hitCount();
                ((AbstractLongAssert)Assertions.assertThat((long)loadDelta).as(cache2 + " loads (delta)", new Object[0])).isEqualTo((Object)this.loads.getOrDefault((Object)cache2, 0L));
                ((AbstractLongAssert)Assertions.assertThat((long)hitsDelta).as(cache2 + " hits (delta)", new Object[0])).isEqualTo((Object)this.hits.getOrDefault((Object)cache2, 0L));
                ((AbstractLongAssert)Assertions.assertThat((long)missesDelta).as(cache2 + " misses (delta)", new Object[0])).isEqualTo((Object)this.misses.getOrDefault((Object)cache2, 0L));
            }
            return value;
        }
    }

    static enum CachingJdbcCache {
        SCHEMA_NAMES_CACHE(CachingJdbcClient::getSchemaNamesCacheStats),
        TABLE_NAMES_CACHE(CachingJdbcClient::getTableNamesCacheStats),
        TABLE_HANDLES_BY_NAME_CACHE(CachingJdbcClient::getTableHandlesByNameCacheStats),
        TABLE_HANDLES_BY_QUERY_CACHE(CachingJdbcClient::getTableHandlesByQueryCacheStats),
        PROCEDURE_HANDLES_BY_QUERY_CACHE(CachingJdbcClient::getProcedureHandlesByQueryCacheStats),
        COLUMNS_CACHE(CachingJdbcClient::getColumnsCacheStats),
        STATISTICS_CACHE(CachingJdbcClient::getStatisticsCacheStats);

        private final Function<CachingJdbcClient, CacheStats> statsGetter;

        private CachingJdbcCache(Function<CachingJdbcClient, CacheStats> statsGetter) {
            this.statsGetter = Objects.requireNonNull(statsGetter, "statsGetter is null");
        }
    }
}

