/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.jdbc;

import com.google.common.base.Throwables;
import com.google.common.reflect.Reflection;
import com.google.inject.Binder;
import com.google.inject.Guice;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Key;
import com.google.inject.Module;
import com.google.inject.Scopes;
import com.google.inject.multibindings.Multibinder;
import io.trino.plugin.jdbc.ConnectionFactory;
import io.trino.plugin.jdbc.ForBaseJdbc;
import io.trino.plugin.jdbc.RetryingConnectionFactory;
import io.trino.plugin.jdbc.RetryingConnectionFactoryModule;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.TestingSession;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.testing.InterfaceTestUtils;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.sql.SQLTransientException;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Objects;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

public class TestRetryingConnectionFactory {
    @Test
    public void testEverythingImplemented() {
        InterfaceTestUtils.assertAllMethodsOverridden(ConnectionFactory.class, RetryingConnectionFactory.class);
    }

    @Test
    public void testSimplyReturnConnection() throws Exception {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.RETURN);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        Connection connection = factory.openConnection(TestingSession.SESSION);
        Assertions.assertThat((Object)connection).isNotNull();
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(1);
    }

    @Test
    public void testRetryAndStopOnTrinoException() {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.THROW_SQL_TRANSIENT_EXCEPTION, MockConnectorFactory.Action.THROW_TRINO_EXCEPTION);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> factory.openConnection(TestingSession.SESSION)).isInstanceOf(TrinoException.class)).hasMessage("Testing Trino exception");
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(2);
    }

    @Test
    public void testRetryAndStopOnSqlException() {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.THROW_SQL_TRANSIENT_EXCEPTION, MockConnectorFactory.Action.THROW_SQL_EXCEPTION);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> factory.openConnection(TestingSession.SESSION)).isInstanceOf(SQLException.class)).hasMessage("Testing sql exception");
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(2);
    }

    @Test
    public void testNullPointerException() {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.THROW_NPE);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> factory.openConnection(TestingSession.SESSION)).isInstanceOf(NullPointerException.class)).hasMessage("Testing NPE");
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(1);
    }

    @Test
    public void testRetryAndReturn() throws Exception {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.THROW_SQL_TRANSIENT_EXCEPTION, MockConnectorFactory.Action.RETURN);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        Connection connection = factory.openConnection(TestingSession.SESSION);
        Assertions.assertThat((Object)connection).isNotNull();
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(2);
    }

    @Test
    public void testRetryOnWrappedAndReturn() throws Exception {
        Injector injector = TestRetryingConnectionFactory.createInjector(MockConnectorFactory.Action.THROW_WRAPPED_SQL_TRANSIENT_EXCEPTION, MockConnectorFactory.Action.RETURN);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        Connection connection = factory.openConnection(TestingSession.SESSION);
        Assertions.assertThat((Object)connection).isNotNull();
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(2);
    }

    @Test
    public void testAdditionalRetryStrategyWorks() throws Exception {
        Injector injector = TestRetryingConnectionFactory.createInjectorWithAdditionalStrategy(MockConnectorFactory.Action.THROW_SQL_RECOVERABLE_EXCEPTION, MockConnectorFactory.Action.THROW_SQL_TRANSIENT_EXCEPTION, MockConnectorFactory.Action.RETURN);
        ConnectionFactory factory = (ConnectionFactory)injector.getInstance(RetryingConnectionFactory.class);
        MockConnectorFactory mock = (MockConnectorFactory)injector.getInstance(MockConnectorFactory.class);
        Connection connection = factory.openConnection(TestingSession.SESSION);
        Assertions.assertThat((Object)connection).isNotNull();
        Assertions.assertThat((int)mock.getCallCount()).isEqualTo(3);
    }

    private static Injector createInjector(MockConnectorFactory.Action ... actions) {
        return Guice.createInjector((Module[])new Module[]{binder -> {
            binder.bind(MockConnectorFactory.Action[].class).toInstance((Object)actions);
            binder.bind(MockConnectorFactory.class).in(Scopes.SINGLETON);
            binder.bind(ConnectionFactory.class).annotatedWith(ForBaseJdbc.class).to(Key.get(MockConnectorFactory.class));
            binder.install((Module)new RetryingConnectionFactoryModule());
        }});
    }

    private static Injector createInjectorWithAdditionalStrategy(MockConnectorFactory.Action ... actions) {
        return Guice.createInjector((Module[])new Module[]{binder -> {
            binder.bind(MockConnectorFactory.Action[].class).toInstance((Object)actions);
            binder.bind(MockConnectorFactory.class).in(Scopes.SINGLETON);
            binder.bind(ConnectionFactory.class).annotatedWith(ForBaseJdbc.class).to(Key.get(MockConnectorFactory.class));
            binder.install((Module)new RetryingConnectionFactoryModule());
            Multibinder.newSetBinder((Binder)binder, RetryingConnectionFactory.RetryStrategy.class).addBinding().to(AdditionalRetryStrategy.class).in(Scopes.SINGLETON);
        }});
    }

    public static class MockConnectorFactory
    implements ConnectionFactory {
        private final Deque<Action> actions = new ArrayDeque<Action>();
        private int callCount;

        @Inject
        public MockConnectorFactory(Action ... actions) {
            Stream.of(actions).forEach(this.actions::push);
        }

        public int getCallCount() {
            return this.callCount;
        }

        public Connection openConnection(ConnectorSession session) throws SQLException {
            ++this.callCount;
            Action action = Objects.requireNonNull(this.actions.pollLast(), "actions.pollFirst() is null");
            switch (action.ordinal()) {
                case 7: {
                    return (Connection)Reflection.newProxy(Connection.class, (proxy, method, args) -> null);
                }
                case 6: {
                    throw new NullPointerException("Testing NPE");
                }
                case 0: {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Testing Trino exception");
                }
                case 1: {
                    throw new SQLException("Testing sql exception");
                }
                case 2: {
                    throw new SQLRecoverableException("Testing sql recoverable exception");
                }
                case 3: {
                    throw new RuntimeException(new SQLRecoverableException("Testing sql recoverable exception"));
                }
                case 4: {
                    throw new SQLTransientException("Testing sql transient exception");
                }
                case 5: {
                    throw new RuntimeException(new SQLTransientException("Testing sql transient exception"));
                }
            }
            throw new IllegalStateException("Unsupported action:" + String.valueOf((Object)action));
        }

        public static enum Action {
            THROW_TRINO_EXCEPTION,
            THROW_SQL_EXCEPTION,
            THROW_SQL_RECOVERABLE_EXCEPTION,
            THROW_WRAPPED_SQL_RECOVERABLE_EXCEPTION,
            THROW_SQL_TRANSIENT_EXCEPTION,
            THROW_WRAPPED_SQL_TRANSIENT_EXCEPTION,
            THROW_NPE,
            RETURN;

        }
    }

    private static class AdditionalRetryStrategy
    implements RetryingConnectionFactory.RetryStrategy {
        private AdditionalRetryStrategy() {
        }

        public boolean isExceptionRecoverable(Throwable exception) {
            return Throwables.getCausalChain((Throwable)exception).stream().anyMatch(SQLRecoverableException.class::isInstance);
        }
    }
}

